package com.alibaba.hbase.haclient.dualservice;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.alibaba.hbase.client.AliHBaseConstants;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.TableName;

public class TableCounter implements ExecuteCounter{

  private static final Log LOG = LogFactory.getLog(TableCounter.class);

  private ConcurrentHashMap<String, ResultCounter> tableCounters;

  private ConcurrentHashMap<String, Boolean> tableSwitchStatus;

  private Configuration conf;

  private int activeFailLimit;

  private float activeFailLimitPercentage;

  private int standbySuccessLimit;

  private ScheduledExecutorService scheduledExecutorService = Executors.newScheduledThreadPool(1);

  public TableCounter(Configuration conf){
    this.conf = conf;
    this.tableCounters = new ConcurrentHashMap<String, ResultCounter>();
    this.tableSwitchStatus = new ConcurrentHashMap<String, Boolean>();
    this.activeFailLimit = this.conf.getInt(AliHBaseConstants.TABLE_ACTIVE_FAIL_LIMIT,
      AliHBaseConstants.DEFAULT_TABLE_ACTIVE_FAIL_LIMIT);
    this.activeFailLimitPercentage = this.conf.getFloat(AliHBaseConstants.TABLE_ACTIVE_FAIL_LIMIT_PECENTAGE,
      AliHBaseConstants.DEFAULT_TABLE_ACTIVE_FAIL_LIMIT_PECENTAGE);
    this.standbySuccessLimit = this.conf.getInt(AliHBaseConstants.TABLE_STANDBY_SUCCESS_LIMIT,
      AliHBaseConstants.DEFAULT_TABLE_STANDBY_SUCCESS_LIMIT);
    long period = conf.getLong(CLEAR_INTERVAL, DEFAULT_CLEAR_INTERVAL);
    scheduledExecutorService.scheduleAtFixedRate(new Thread("table-counter") {
      @Override
      public void run() {
        for(Map.Entry<String, ResultCounter> entry : tableCounters.entrySet()){
          entry.getValue().updateActiveErrorPercentage();
        }
      }
    }, period, period, TimeUnit.MILLISECONDS);

  }

  @Override
  public synchronized <T> void update(DualContext<T> dualContext){
    boolean isSwitch = this.isSwitchStatus(dualContext);
    if(dualContext.getExecuteStrategy() == AutoSwitch.ExecuteStrategy.DEFAULT && isSwitch){
      return;
    }
    if(dualContext.getExecuteStrategy() == AutoSwitch.ExecuteStrategy.SWITCH && !isSwitch){
      return;
    }
    String tableNameAsString = TableName.valueOf(dualContext.getTableName()).getNameAsString();
    if(tableCounters.containsKey(tableNameAsString)){
      tableCounters.get(tableNameAsString).increment(dualContext);
    }else{
      ResultCounter newCounter = new ResultCounter(ResultCounter.CountStrategy.PERCENTAGE);
      newCounter.increment(dualContext);
      tableCounters.put(tableNameAsString, newCounter);
    }
  }

  @Override
  public synchronized <T> boolean isSwitchStatus(DualContext<T> dualContext) {
    String tableNameAsString = TableName.valueOf(dualContext.getTableName()).getNameAsString();
    if(tableSwitchStatus.containsKey(tableNameAsString)){
      return tableSwitchStatus.get(tableNameAsString);
    }else{
      tableSwitchStatus.put(tableNameAsString, false);
      return false;
    }
  }

  @Override
  public synchronized <T> void setSwitchStatus(DualContext<T> dualContext, boolean switchStatus) {
    String tableNameAsString = TableName.valueOf(dualContext.getTableName()).getNameAsString();
    tableSwitchStatus.put(tableNameAsString, switchStatus);
    this.reset(dualContext);
  }

  @Override
  public synchronized <T>  boolean reachLimit(DualContext<T> dualContext){
    String tableNameAsString = TableName.valueOf(dualContext.getTableName()).getNameAsString();
    ResultCounter resultCounter = null;
    resultCounter = tableCounters.get(tableNameAsString);
    if(resultCounter == null){
      return false;
    }
    boolean result;
    if(!isSwitchStatus(dualContext)){
      result = (resultCounter.getActiveFailCount() > this.activeFailLimit) || (resultCounter.getActiveFailPercentage() > this.activeFailLimitPercentage);
    }else{
      result = resultCounter.getStandbySuccessCount() > this.standbySuccessLimit;
    }
    if(result) {
      LOG.debug("Reach limit active fail count=" + this.activeFailLimit +
        ", active fail count percentage=" + resultCounter.getActiveFailPercentage() +
        ", standby success count=" + resultCounter.getStandbySuccessCount() +
        ", total fail count=" + resultCounter.getTotalActiveFailCount());
    }
    return result;
  }

  @Override
  public synchronized <T> void reset(DualContext<T> dualContext){
    String tableNameAsString = TableName.valueOf(dualContext.getTableName()).getNameAsString();
    if(tableCounters.containsKey(tableNameAsString)){
      tableCounters.get(tableNameAsString).totalAndActiveCountClear();
      tableCounters.get(tableNameAsString).activeCountClear();
      tableCounters.get(tableNameAsString).standbyCountClear();
    }
  }

}