/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.alibaba.hbase.client.AliHBaseAPIProxy;
import com.alibaba.hbase.client.AliHBaseAPIProxyDirectImpl;
import com.alibaba.hbase.client.AliHBaseUEConnection;
import com.alibaba.hbase.client.ElementConvertor;

import com.alibaba.lindorm.client.TableService;
import com.google.common.base.Preconditions;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.CacheEvictionStats;
import org.apache.hadoop.hbase.ClusterMetrics;
import org.apache.hadoop.hbase.ClusterMetricsBuilder;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.NamespaceDescriptor;
import org.apache.hadoop.hbase.NamespaceNotFoundException;
import org.apache.hadoop.hbase.RegionMetrics;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.TableNotFoundException;
import org.apache.hadoop.hbase.client.index.AliHBaseIndexDescriptor;
import org.apache.hadoop.hbase.client.replication.TableCFs;
import org.apache.hadoop.hbase.client.security.SecurityCapability;
import org.apache.hadoop.hbase.ipc.CoprocessorRpcChannel;
import org.apache.hadoop.hbase.quotas.QuotaFilter;
import org.apache.hadoop.hbase.quotas.QuotaRetriever;
import org.apache.hadoop.hbase.quotas.QuotaSettings;
import org.apache.hadoop.hbase.quotas.SpaceQuotaSnapshot;
import org.apache.hadoop.hbase.replication.ReplicationException;
import org.apache.hadoop.hbase.replication.ReplicationPeerConfig;
import org.apache.hadoop.hbase.replication.ReplicationPeerDescription;
import org.apache.hadoop.hbase.security.access.UserPermission;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AliHBaseUEAdmin extends HBaseAdmin implements AliHBaseAdminInterface {
  private static final Logger LOG = LoggerFactory.getLogger(AliHBaseUEAdmin.class);

  private AliHBaseUEConnection connection;
  private AliHBaseUEClusterConnection clusterConnection;
  private int operationTimeout;
  private Configuration conf;
  private AliHBaseAPIProxy proxy;

  private static AliHBaseUEClusterConnection superConnection = null;

  private static AliHBaseUEClusterConnection getSuperConnection(AliHBaseUEConnection connection) throws IOException{
    if(superConnection == null || superConnection.isClosed()){
      superConnection = new AliHBaseUEClusterConnection(connection);
    }
    return superConnection;
  }

  public AliHBaseUEAdmin(AliHBaseUEConnection connection) throws IOException {
    super(getSuperConnection(connection));
    this.connection = connection;
    this.clusterConnection = new AliHBaseUEClusterConnection(connection);
    this.conf = connection.getConfiguration();
    this.operationTimeout = connection.getOperationTimeout();
    proxy = connection.getAPIProxy(null);
  }

  public TableService getTableService(String namespace) throws IOException {
    if (proxy instanceof AliHBaseAPIProxyDirectImpl) {
      return ((AliHBaseAPIProxyDirectImpl) proxy).getTableService(namespace);
    } else {
      return null;
    }
  }

  @Override
  public int getOperationTimeout() {
    return operationTimeout;
  }

  @Override
  public void abort(String why, Throwable e) {
  }

  @Override
  public boolean isAborted() {
    return false;
  }

  @Override
  public void close() throws IOException {
    proxy.close();
  }

  @Override
  public Configuration getConfiguration() {
    return conf;
  }

  @Override
  public boolean tableExists(TableName tableName) throws IOException {
    return proxy.tableExists(tableName);
  }

  @Override
  public Connection getConnection() {
    return clusterConnection;
  }

  @Override
  public HTableDescriptor[] listTables() throws IOException {
    List<TableDescriptor> tableDescriptors = listTableDescriptors();
    HTableDescriptor[] hTableDescriptors = new HTableDescriptor[tableDescriptors.size()];
    for (int i = 0; i < tableDescriptors.size(); i++) {
      hTableDescriptors[i] = new HTableDescriptor(tableDescriptors.get(i));
    }
    return hTableDescriptors;
  }

  @Override
  public List<TableDescriptor> listTableDescriptors() throws IOException {
    return proxy.listTableDescriptors();
  }

  @Override
  public HTableDescriptor[] listTables(Pattern pattern) throws IOException {
    List<HTableDescriptor> result = new ArrayList<>();

    List<TableDescriptor> tableDescriptors = listTableDescriptors();
    for (int i = 0; i < tableDescriptors.size(); i++) {
      Matcher matcher = pattern.matcher(tableDescriptors.get(i).getTableName().getNameAsString());
      if (matcher.matches()) {
          result.add(new HTableDescriptor(tableDescriptors.get(i)));
      }
    }

    return result.toArray(new HTableDescriptor[0]);
  }

  @Override
  public List<TableDescriptor> listTableDescriptors(Pattern pattern) throws IOException {
    return listTableDescriptors();
  }

  @Override
  public HTableDescriptor[] listTables(String regex) throws IOException {
    return listTables();
  }

  @Override
  public HTableDescriptor[] listTables(Pattern pattern, boolean includeSysTables)
      throws IOException {
    return listTables();
  }

  @Override
  public List<TableDescriptor> listTableDescriptors(Pattern pattern, boolean includeSysTables)
      throws IOException {
    return listTableDescriptors();
  }

  @Override
  public HTableDescriptor[] listTables(String regex, boolean includeSysTables) throws IOException {
    return listTables();
  }

  @Override
  public TableName[] listTableNames() throws IOException {
    return proxy.listTableNames();
  }

  @Override
  public TableName[] listTableNames(Pattern pattern) throws IOException {
    TableName[] allTables = listTableNames();
    List<TableName> results = new ArrayList<>();
    for (int i = 0; i < allTables.length; i++) {
      if(pattern.matcher(allTables[i].getNameAsString()).matches()){
        results.add(allTables[i]);
      }
    }
    return results.toArray(new TableName[0]);
  }

  @Override
  public TableName[] listTableNames(String regex) throws IOException {
    return listTableNames(Pattern.compile(regex));
  }

  @Override
  public TableName[] listTableNames(Pattern pattern, boolean includeSysTables) throws IOException {
    return listTableNames(pattern);
  }

  @Override
  public TableName[] listTableNames(String regex, boolean includeSysTables) throws IOException {
    return listTableNames(Pattern.compile(regex));
  }

  @Override
  public HTableDescriptor getTableDescriptor(TableName tableName)
      throws TableNotFoundException, IOException {
    TableDescriptor tableDescriptor = getDescriptor(tableName);
    return new HTableDescriptor(tableDescriptor);
  }

  @Override
  public TableDescriptor getDescriptor(TableName tableName)
      throws TableNotFoundException, IOException {
    return proxy.getDescriptor(tableName);
  }

  @Override
  public HTableDescriptor[] listTableDescriptorsByNamespace(String name) throws IOException {
    List<TableDescriptor> descriptors = listTableDescriptorsByNamespace(Bytes.toBytes(name));
    HTableDescriptor[] hTableDescriptors = new HTableDescriptor[descriptors.size()];
    for (int i = 0; i < descriptors.size(); i++) {
      hTableDescriptors[i] = new HTableDescriptor(descriptors.get(i));
    }
    return hTableDescriptors;
  }

  @Override
  public List<TableDescriptor> listTableDescriptorsByNamespace(byte[] name) throws IOException {
    return proxy.listTableDescriptorsByNamespace(name);
  }

  @Override
  public TableName[] listTableNamesByNamespace(String name) throws IOException {
    return proxy.listTableNamesByNamespace(name);
  }

  @Override
  public HTableDescriptor[] getTableDescriptorsByTableName(List<TableName> tableNames)
      throws IOException {
    HTableDescriptor[] tableDescriptors = new HTableDescriptor[tableNames.size()];
    for (int i = 0; i < tableNames.size(); i++) {
      tableDescriptors[i] = getTableDescriptor(tableNames.get(i));
    }
    return tableDescriptors;
  }

  @Override
  public List<TableDescriptor> listTableDescriptors(List<TableName> tableNames) throws IOException {
    List<TableDescriptor> tableDescriptors = new ArrayList<>();
    for (TableName tableName : tableNames) {
      tableDescriptors.add(getDescriptor(tableName));
    }
    return tableDescriptors;
  }

  @Override
  public HTableDescriptor[] getTableDescriptors(List<String> names) throws IOException {
    List<TableName> tableNames = names.stream().map(name -> TableName.valueOf(name)).collect(
        Collectors.toList());
    return getTableDescriptorsByTableName(tableNames);
  }

  @Override
  public void createTable(TableDescriptor desc) throws IOException {
    createTable(desc, null);
  }

  @Override
  public void createTable(TableDescriptor desc, byte[] startKey, byte[] endKey, int numRegions)
      throws IOException {
    if(numRegions < 3) {
      throw new IllegalArgumentException("Must create at least three regions");
    } else if(Bytes.compareTo(startKey, endKey) >= 0) {
      throw new IllegalArgumentException("Start key must be smaller than end key");
    }
    if (numRegions == 3) {
      createTable(desc, new byte[][]{startKey, endKey});
      return;
    }
    byte [][] splitKeys = Bytes.split(startKey, endKey, numRegions - 3);
    if(splitKeys == null || splitKeys.length != numRegions - 1) {
      throw new IllegalArgumentException("Unable to split key range into enough regions");
    }
    createTable(desc, splitKeys);
  }

  @Override
  public void createTable(TableDescriptor desc, byte[][] splitKeys) throws IOException {
    TableName.isLegalFullyQualifiedTableName(desc.getTableName().getName());
    proxy.createTable(desc, splitKeys);
  }

  @Override
  public void deleteTable(TableName tableName) throws IOException {
    proxy.deleteTable(tableName);
  }

  @Override
  public void truncateTable(TableName tableName, boolean preserveSplits) throws IOException {
    proxy.truncateTable(tableName, preserveSplits);
  }

  @Override
  public void enableTable(TableName tableName) throws IOException {
    proxy.enableTable(tableName);
  }

  @Override
  public void disableTable(TableName tableName) throws IOException {
    proxy.disableTable(tableName);
  }

  @Override
  public boolean isTableEnabled(TableName tableName) throws IOException {
    return proxy.isTableEnabled(tableName);
  }

  @Override
  public boolean isTableDisabled(TableName tableName) throws IOException {
    return proxy.isTableDisabled(tableName);
  }

  @Override
  public boolean isTableAvailable(TableName tableName) throws IOException {
    return isTableEnabled(tableName);
  }

  @Override
  public boolean isTableAvailable(TableName tableName, byte[][] splitKeys) throws IOException {
    boolean enable = isTableEnabled(tableName);
    if (enable) {
      AliHBaseAPIProxy localProxy = null;
      try {
        localProxy = connection.getAPIProxy(tableName);
        Pair<byte[][], byte[][]> pair = localProxy.getStartEndKeys();
        byte[][] startKeys = pair.getFirst();
        byte[][] endKeys = pair.getSecond();

        if (startKeys.length != splitKeys.length + 1 || endKeys.length != splitKeys.length + 1) {
          return false;
        }

        for (int i = 0; i < splitKeys.length; i++) {
          if (!Bytes.equals(startKeys[i + 1], splitKeys[i]) || !Bytes.equals(endKeys[i], splitKeys[i])) {
            return false;
          }
        }
        return enable;
      } finally {
        if (localProxy != null) {
          localProxy.close();
        }
      }
    } else {
      return enable;
    }
  }

  @Override
  public void addColumnFamily(TableName tableName, ColumnFamilyDescriptor columnFamily)
      throws IOException {
    proxy.addColumnFamily(tableName, columnFamily);
  }

  @Override
  public void deleteColumn(TableName tableName, byte[] columnFamily) throws IOException {
    deleteColumnFamily(tableName, columnFamily);
  }

  @Override
  public void deleteColumnFamily(TableName tableName, byte[] columnFamily) throws IOException {
    proxy.deleteColumnFamily(tableName, columnFamily);
  }

  @Override
  public void modifyColumnFamily(TableName tableName, ColumnFamilyDescriptor columnFamily)
      throws IOException {
    proxy.modifyColumnFamily(tableName, columnFamily);
  }

  @Override
  public void modifyTable(TableName tableName, TableDescriptor td) throws IOException {
    modifyTable(td);
  }

  @Override
  public void modifyTable(TableDescriptor td) throws IOException {
    proxy.modifyTable(td);
  }

  @Override
  public void modifyNamespace(NamespaceDescriptor descriptor) throws IOException {

  }

  @Override
  public void deleteNamespace(String name) throws IOException {
    proxy.deleteNamespace(name);
  }

  @Override
  public NamespaceDescriptor getNamespaceDescriptor(String name)
      throws NamespaceNotFoundException, IOException {
    return proxy.getNamespaceDescriptor(name);
  }

  @Override
  public NamespaceDescriptor[] listNamespaceDescriptors() throws IOException {
    return proxy.listNamespaceDescriptors();
  }

  @Override
  public void createNamespace(NamespaceDescriptor descriptor) throws IOException {
    String namespace = descriptor.getName();
    TableName.isLegalNamespaceName(Bytes.toBytes(namespace));
    proxy.createNamespace(descriptor);
  }

  @Override
  public void flush(TableName tableName) {
    proxy.flushTable(tableName);
  }

  @Override
  public void flushRegion(byte[] regionName) {
    proxy.flushRegion(Bytes.toString(regionName));
  }

  @Override
  public void compact(TableName tableName) {
    proxy.compactTable(tableName);
  }

  @Override
  public void compactRegion(byte[] regionName) {
    proxy.compactRegion(Bytes.toString(regionName));
  }

  @Override
  public void compact(TableName tableName, byte[] columnFamily) {
    compact(tableName);
  }

  @Override
  public void compactRegion(byte[] regionName, byte[] columnFamily) {
    compactRegion(regionName);
  }

  @Override
  public void compact(TableName tableName, CompactType compactType) {
    compact(tableName);

  }

  @Override
  public void compact(TableName tableName, byte[] columnFamily, CompactType compactType) {
    compact(tableName);
  }

  @Override
  public void majorCompact(TableName tableName) {
    proxy.majorCompactTable(tableName);
  }

  @Override
  public void majorCompactRegion(byte[] regionName) {
    proxy.majorCompactRegion(Bytes.toString(regionName));
  }

  @Override
  public void majorCompact(TableName tableName, byte[] columnFamily) {
    majorCompact(tableName);

  }

  @Override
  public void majorCompactRegion(byte[] regionName, byte[] columnFamily) {
    majorCompactRegion(regionName);

  }

  @Override
  public void majorCompact(TableName tableName, CompactType compactType) {
    majorCompact(tableName);

  }

  @Override
  public void majorCompact(TableName tableName, byte[] columnFamily, CompactType compactType) {
    majorCompact(tableName);

  }

  // only avaliable in HBase-2.2+
  public boolean switchRpcThrottle(boolean enable) throws IOException {
    return false;
  }

  // only avaliable in HBase-2.2+
  public boolean isRpcThrottleEnabled() throws IOException {
    return false;
  }

  // only avaliable in HBase-2.2+
  public boolean exceedThrottleQuotaSwitch(boolean enable) throws IOException {
    return false;
  }

  @Override
  public HTableDescriptor[] disableTables(String regex) throws IOException {
    return disableTables(Pattern.compile(regex));
  }

  @Override
  public HTableDescriptor[] disableTables(Pattern pattern) throws IOException {
    List<HTableDescriptor> failed = new LinkedList<>();
    for (HTableDescriptor table : listTables(pattern)) {
      if (isTableEnabled(table.getTableName())) {
        try {
          disableTable(table.getTableName());
        } catch (IOException ex) {
          LOG.info("Failed to enable table " + table.getTableName(), ex);
          failed.add(table);
        }
      }
    }
    return failed.toArray(new HTableDescriptor[failed.size()]);
  }

  @Override
  public HTableDescriptor[] enableTables(String regex) throws IOException {
    return enableTables(Pattern.compile(regex));
  }

  @Override
  public HTableDescriptor[] enableTables(Pattern pattern) throws IOException {
    List<HTableDescriptor> failed = new LinkedList<>();
    for (HTableDescriptor table : listTables(pattern)) {
      if (isTableDisabled(table.getTableName())) {
        try {
          enableTable(table.getTableName());
        } catch (IOException ex) {
          LOG.info("Failed to enable table " + table.getTableName(), ex);
          failed.add(table);
        }
      }
    }
    return failed.toArray(new HTableDescriptor[failed.size()]);
  }

  @Override
  public HTableDescriptor[] deleteTables(String regex) throws IOException {
    throw new UnsupportedOperationException("deleteTables by pattern not supported");
  }

  @Override
  public HTableDescriptor[] deleteTables(Pattern pattern) throws IOException {
    throw new UnsupportedOperationException("deleteTables by pattern not supported");
  }

  @Override
  public void closeRegion(String regionname, String serverName) {

  }

  @Override
  public void closeRegion(byte[] regionname, String serverName) {
  }

  @Override
  public boolean closeRegionWithEncodedRegionName(String encodedRegionName, String serverName) {
      return true;
  }

  @Override
  public void closeRegion(ServerName sn, HRegionInfo hri) {
  }

  @Override
  public List<HRegionInfo> getOnlineRegions(ServerName sn) {
      return new ArrayList<>();
  }

  @Override
  public List<RegionInfo> getRegions(ServerName serverName) {
    return new ArrayList<>();
  }

  @Override
  public void flushRegionServer(ServerName serverName) {
  }

  // only avaliable in HBase-2.2+
  public Map<ServerName, Boolean> compactionSwitch(boolean switchState,
      List<String> serverNamesList) {
    return new HashMap<>(); // fake;
  }

  @Override
  public void compactRegionServer(ServerName serverName) {
  }

  @Override
  public void majorCompactRegionServer(ServerName serverName) {
  }

  // only avaliable in HBase-2.2+
  public void move(byte[] encodedRegionName) {
      // trivial
  }

  // only avaliable in HBase-2.2+
  public void move(byte[] encodedRegionName, ServerName destServerName) {
    // trivial
  }

  @Override
  public void assign(byte[] regionName) {
  }

  @Override
  public void unassign(byte[] regionName, boolean force) {
  }

  @Override
  public void offline(byte[] regionName) {
  }

  @Override
  public boolean balancerSwitch(boolean onOrOff, boolean synchronous) {
      return true;
  }

  @Override
  public boolean balance() {
      return true;
  }

  @Override
  public boolean balance(boolean force) {
      return true;
  }

  @Override
  public boolean isBalancerEnabled() {
      return true;
  }

  @Override
  public CacheEvictionStats clearBlockCache(TableName tableName) {
      return null;
  }

  @Override
  public boolean normalize() {
      return false;
  }

  @Override
  public boolean isNormalizerEnabled() {
      return false;
  }

  @Override
  public boolean normalizerSwitch(boolean on) {
      return false;
  }

  @Override
  public boolean catalogJanitorSwitch(boolean onOrOff) {
      return true;
  }

  @Override
  public int runCatalogJanitor() {
      return 0;
  }

  @Override
  public boolean isCatalogJanitorEnabled() {
      return true;
  }

  @Override
  public boolean cleanerChoreSwitch(boolean onOrOff) {
      return true;
  }

  @Override
  public boolean runCleanerChore() {
      return false;
  }

  @Override
  public boolean isCleanerChoreEnabled() {
      return true;
  }

  @Override
  public void mergeRegions(byte[] nameOfRegionA, byte[] nameOfRegionB, boolean forcible) {
    // note: only support full region name
    String regionA = Bytes.toString(nameOfRegionA);
    String regionB = Bytes.toString(nameOfRegionB);

    String[] splitA = regionA.split(",");
    String[] splitB = regionB.split(",");
    Preconditions.checkArgument(splitA.length > 1 && splitB.length > 1,
      "only support full region name");

    String tableNameA = splitA[0];
    String tableNameB = splitB[0];
    Preconditions.checkArgument(StringUtils.equals(tableNameA, tableNameB),
      "regions to be merged should belong to the same table");

    String shortNameA = regionA.split("\\.")[2];
    String shortNameB = regionB.split("\\.")[2];

    proxy.mergeRegions(tableNameA, shortNameA, shortNameB);
  }

  @Override
  public Future<Void> mergeRegionsAsync(byte[] nameOfRegionA, byte[] nameOfRegionB,
      boolean forcible) {
    mergeRegions(nameOfRegionA, nameOfRegionB, forcible);
    return getVoidFuture();
  }

  @Override
  public Future<Void> mergeRegionsAsync(byte[][] nameofRegionsToMerge, boolean forcible) {
    throw new UnsupportedOperationException("mergeRegionsAsync not supported");
  }

  @Override
  public void split(TableName tableName) {
    String table = ElementConvertor.toLindormTableFullName(tableName);
    proxy.split(table, null);
  }

  @Override
  public void splitRegion(byte[] regionName) {
    proxy.split(Bytes.toString(regionName), null);
  }

  @Override
  public void split(TableName tableName, byte[] splitPoint) {
    String table = ElementConvertor.toLindormTableFullName(tableName);
    proxy.split(table, splitPoint);
  }

  @Override
  public void splitRegion(byte[] regionName, byte[] splitPoint) {
    proxy.split(Bytes.toString(regionName), splitPoint);
  }

  @Override
  public Future<Void> splitRegionAsync(byte[] regionName, byte[] splitPoint) {
    splitRegion(regionName, splitPoint);
    return getVoidFuture();
  }

  @Override
  public Future<Void> modifyTableAsync(TableName tableName, TableDescriptor td) throws IOException {
    modifyTable(tableName, td);
    return getVoidFuture();
  }

  @Override
  public Future<Void> modifyTableAsync(TableDescriptor td) throws IOException {
    modifyTable(td);
    return getVoidFuture();
  }

  @Override
  public void shutdown() {
  }

  @Override
  public void stopMaster() {
  }

  @Override
  public boolean isMasterInMaintenanceMode() {
    return false;
  }

  @Override
  public void stopRegionServer(String hostnamePort) {
  }

  @Override
  public ClusterMetrics getClusterMetrics(EnumSet<ClusterMetrics.Option> options) {
    ClusterMetricsBuilder clusterMetricsBuilder = ClusterMetricsBuilder.newBuilder();
    return clusterMetricsBuilder.build();
  }

  @Override
  public List<RegionMetrics> getRegionMetrics(ServerName serverName,
      TableName tableName) {
    List<RegionMetrics> metricses = new ArrayList<>();
    return metricses;
  }

  @Override
  public Future<Void> createNamespaceAsync(NamespaceDescriptor descriptor) throws IOException {
    createNamespace(descriptor);
    return getVoidFuture();
  }

  @Override
  public Future<Void> modifyNamespaceAsync(NamespaceDescriptor descriptor) throws IOException {
    modifyNamespace(descriptor);
    return getVoidFuture();
  }

  @Override
  public List<HRegionInfo> getTableRegions(TableName tableName) throws IOException {
    AliHBaseAPIProxy localProxy = null;
    try {
      localProxy = connection.getAPIProxy(tableName);
      List<HRegionLocation> list = localProxy.getAllRegionLocations();
      List<HRegionInfo> result = new ArrayList<>(list.size());
      for(HRegionLocation location : list) {
        result.add((HRegionInfo)location.getRegion());
      }
      return result;
    } finally {
      if (localProxy != null) {
        localProxy.close();
      }
    }
  }

  @Override
  public List<RegionInfo> getRegions(TableName tableName) {
    if (TableName.isMetaTableName(tableName)) {
      return Arrays.asList(RegionInfoBuilder.FIRST_META_REGIONINFO);
    } else {
      return new ArrayList<>();
    }
  }

  @Override
  public boolean abortProcedure(long procId, boolean mayInterruptIfRunning) {
    return false;
  }

  @Override
  public Future<Boolean> abortProcedureAsync(long procId, boolean mayInterruptIfRunning) {
    return new Future<Boolean>() {
      @Override
      public boolean cancel(boolean mayInterruptIfRunning) {
        return false;
      }

      @Override
      public boolean isCancelled() {
        return false;
      }

      @Override
      public boolean isDone() {
        return true;
      }

      @Override
      public Boolean get() throws InterruptedException, ExecutionException {
        return false;
      }

      @Override
      public Boolean get(long timeout, TimeUnit unit)
          throws InterruptedException, ExecutionException, TimeoutException {
        return false;
      }
    };
  }

  @Override
  public String getProcedures() {
      return "";
  }

  @Override
  public String getLocks() {
      return "";
  }

  @Override
  public void rollWALWriter(ServerName serverName) {
  }

  @Override
  public CompactionState getCompactionState(TableName tableName) {
      return CompactionState.NONE;
  }

  @Override
  public CompactionState getCompactionState(TableName tableName, CompactType compactType) {
    return CompactionState.NONE;
  }

  @Override
  public CompactionState getCompactionStateForRegion(byte[] regionName) {
    return CompactionState.NONE;
  }

  @Override
  public long getLastMajorCompactionTimestamp(TableName tableName) {
      return 0;
  }

  @Override
  public long getLastMajorCompactionTimestampForRegion(byte[] regionName) {
    return 0;
  }

  @Override
  public void snapshot(String snapshotName, TableName tableName) {
    throw new UnsupportedOperationException("snapshot not supported");

  }

  @Override
  public void snapshot(byte[] snapshotName, TableName tableName) {
    throw new UnsupportedOperationException("snapshot not supported");

  }

  @Override
  public void snapshot(String snapshotName, TableName tableName, SnapshotType type) {
    throw new UnsupportedOperationException("snapshot not supported");

  }

  @Override
  public void snapshot(SnapshotDescription snapshot) {
    throw new UnsupportedOperationException("snapshot not supported");

  }

  @Override
  public void snapshotAsync(SnapshotDescription snapshot) {
    throw new UnsupportedOperationException("snapshotAsync not supported");

  }

  @Override
  public boolean isSnapshotFinished(SnapshotDescription snapshot) {
    throw new UnsupportedOperationException("isSnapshotFinished not supported");
  }

  @Override
  public void restoreSnapshot(byte[] snapshotName) {
    throw new UnsupportedOperationException("restoreSnapshot not supported");

  }

  @Override
  public void restoreSnapshot(String snapshotName) {
    throw new UnsupportedOperationException("restoreSnapshot not supported");

  }

  @Override
  public Future<Void> restoreSnapshotAsync(String snapshotName) {
    throw new UnsupportedOperationException("restoreSnapshotAsync not supported");
  }

  @Override
  public void restoreSnapshot(byte[] snapshotName, boolean takeFailSafeSnapshot) {
    throw new UnsupportedOperationException("restoreSnapshot not supported");

  }

  @Override
  public void restoreSnapshot(String snapshotName, boolean takeFailSafeSnapshot) {
    throw new UnsupportedOperationException("restoreSnapshot not supported");

  }

  @Override
  public void restoreSnapshot(String snapshotName, boolean takeFailSafeSnapshot,
      boolean restoreAcl) {
    throw new UnsupportedOperationException("restoreSnapshot not supported");

  }

  @Override
  public void cloneSnapshot(byte[] snapshotName, TableName tableName) {
    throw new UnsupportedOperationException("cloneSnapshot not supported");

  }

  @Override
  public void cloneSnapshot(String snapshotName, TableName tableName, boolean restoreAcl) {
    throw new UnsupportedOperationException("cloneSnapshot not supported");

  }

  @Override
  public void cloneSnapshot(String snapshotName, TableName tableName) {
    throw new UnsupportedOperationException("cloneSnapshot not supported");

  }

  @Override
  public Future<Void> cloneSnapshotAsync(String snapshotName, TableName tableName) {
    throw new UnsupportedOperationException("cloneSnapshotAsync not supported");
  }

  @Override
  public void execProcedure(String signature, String instance, Map<String, String> props) {
    throw new UnsupportedOperationException("execProcedure not supported");

  }

  @Override
  public byte[] execProcedureWithReturn(String signature, String instance,
      Map<String, String> props) {
    throw new UnsupportedOperationException("execProcedureWithReturn not supported");
  }

  @Override
  public boolean isProcedureFinished(String signature, String instance, Map<String, String> props) {
    throw new UnsupportedOperationException("isProcedureFinished not supported");
  }

  @Override
  public List<SnapshotDescription> listSnapshots() {
    throw new UnsupportedOperationException("listSnapshots not supported");
  }

  @Override
  public List<SnapshotDescription> listSnapshots(String regex) {
    throw new UnsupportedOperationException("listSnapshots not supported");
  }

  @Override
  public List<SnapshotDescription> listSnapshots(Pattern pattern) {
    throw new UnsupportedOperationException("listSnapshots not supported");
  }

  @Override
  public List<SnapshotDescription> listTableSnapshots(String tableNameRegex,
      String snapshotNameRegex) {
    throw new UnsupportedOperationException("listTableSnapshots not supported");
  }

  @Override
  public List<SnapshotDescription> listTableSnapshots(Pattern tableNamePattern,
      Pattern snapshotNamePattern) {
    throw new UnsupportedOperationException("listTableSnapshots not supported");
  }

  @Override
  public void deleteSnapshot(byte[] snapshotName) {
    throw new UnsupportedOperationException("deleteSnapshot not supported");

  }

  @Override
  public void deleteSnapshot(String snapshotName) {
    throw new UnsupportedOperationException("deleteSnapshot not supported");

  }

  @Override
  public void deleteSnapshots(String regex) {
    throw new UnsupportedOperationException("deleteSnapshots not supported");

  }

  @Override
  public void deleteSnapshots(Pattern pattern) {
    throw new UnsupportedOperationException("deleteSnapshots not supported");

  }

  @Override
  public void deleteTableSnapshots(String tableNameRegex, String snapshotNameRegex) {
    throw new UnsupportedOperationException("deleteTableSnapshots not supported");

  }

  @Override
  public void deleteTableSnapshots(Pattern tableNamePattern, Pattern snapshotNamePattern) {
    throw new UnsupportedOperationException("deleteTableSnapshots not supported");

  }

  @Override
  public void setQuota(QuotaSettings quota) {
    throw new UnsupportedOperationException("setQuota not supported");

  }

  @Override
  public QuotaRetriever getQuotaRetriever(QuotaFilter filter) {
    throw new UnsupportedOperationException("getQuotaRetriever not supported");
  }

  @Override
  public List<QuotaSettings> getQuota(QuotaFilter filter) {
    throw new UnsupportedOperationException("getQuota not supported");
  }

  @Override
  public CoprocessorRpcChannel coprocessorService() {
    throw new UnsupportedOperationException("coprocessorService not supported");
  }

  @Override
  public CoprocessorRpcChannel coprocessorService(ServerName serverName) {
    throw new UnsupportedOperationException("coprocessorService not supported");
  }

  @Override
  public void updateConfiguration(ServerName server) {

  }

  @Override
  public void updateConfiguration() {
  }

  @Override
  public List<SecurityCapability> getSecurityCapabilities() {
    return new ArrayList<>();
  }

  @Override
  public boolean splitSwitch(boolean enabled, boolean synchronous) {
      return true;
  }

  @Override
  public boolean mergeSwitch(boolean enabled, boolean synchronous) {
    return true;
  }

  @Override
  public boolean isSplitEnabled() {
      return true;
  }

  @Override
  public boolean isMergeEnabled() {
      return true;
  }

  @Override
  public void addReplicationPeer(String peerId, ReplicationPeerConfig peerConfig, boolean enabled) {
    throw new UnsupportedOperationException("addReplicationPeer not supported");

  }

  // only avaliable in HBase-2.1+
  public Future<Void> addReplicationPeerAsync(String peerId, ReplicationPeerConfig peerConfig,
      boolean enabled) {
    throw new UnsupportedOperationException("addReplicationPeerAsync not supported");
  }

  @Override
  public void removeReplicationPeer(String peerId) {
    throw new UnsupportedOperationException("removeReplicationPeer not supported");

  }

  // only avaliable in HBase-2.1+
  public Future<Void> removeReplicationPeerAsync(String peerId) {
    throw new UnsupportedOperationException("removeReplicationPeerAsync not supported");
  }

  @Override
  public void enableReplicationPeer(String peerId) {
    throw new UnsupportedOperationException("enableReplicationPeer not supported");

  }

  // only avaliable in HBase-2.1+
  public Future<Void> enableReplicationPeerAsync(String peerId) {
    throw new UnsupportedOperationException("enableReplicationPeerAsync not supported");
  }

  @Override
  public void disableReplicationPeer(String peerId) {
    throw new UnsupportedOperationException("disableReplicationPeer not supported");

  }

  // only avaliable in HBase-2.1+
  public Future<Void> disableReplicationPeerAsync(String peerId) {
    throw new UnsupportedOperationException("disableReplicationPeerAsync not supported");
  }

  @Override
  public ReplicationPeerConfig getReplicationPeerConfig(String peerId) {
    throw new UnsupportedOperationException("getReplicationPeerConfig not supported");
  }

  @Override
  public void updateReplicationPeerConfig(String peerId, ReplicationPeerConfig peerConfig) {
    throw new UnsupportedOperationException("updateReplicationPeerConfig not supported");

  }

  // only avaliable in HBase-2.1+
  public Future<Void> updateReplicationPeerConfigAsync(String peerId,
      ReplicationPeerConfig peerConfig) {
    throw new UnsupportedOperationException(
        "updateReplicationPeerConfigAsync not supported");
  }

  @Override
  public void appendReplicationPeerTableCFs(String id, Map<TableName, List<String>> tableCfs)
      throws ReplicationException, IOException {
    throw new UnsupportedOperationException("appendReplicationPeerTableCFs not supported");

  }

  @Override
  public void removeReplicationPeerTableCFs(String id, Map<TableName, List<String>> tableCfs)
      throws ReplicationException, IOException {
    throw new UnsupportedOperationException("removeReplicationPeerTableCFs not supported");

  }

  @Override
  public List<ReplicationPeerDescription> listReplicationPeers() {
    throw new UnsupportedOperationException("listReplicationPeers not supported");
  }

  @Override
  public List<ReplicationPeerDescription> listReplicationPeers(Pattern pattern) {
    throw new UnsupportedOperationException("listReplicationPeers not supported");
  }

  @Override
  public void decommissionRegionServers(List<ServerName> servers, boolean offload) {
  }

  @Override
  public List<ServerName> listDecommissionedRegionServers() {
      return new ArrayList<>();
  }

  @Override
  public void recommissionRegionServer(ServerName server, List<byte[]> encodedRegionNames) {
  }

  @Override
  public List<TableCFs> listReplicatedTableCFs() {
    throw new UnsupportedOperationException("listReplicatedTableCFs not supported");
  }

  @Override
  public void enableTableReplication(TableName tableName) {
    throw new UnsupportedOperationException("enableTableReplication not supported");

  }

  @Override
  public void disableTableReplication(TableName tableName) {
    throw new UnsupportedOperationException("disableTableReplication not supported");

  }

  @Override
  public void clearCompactionQueues(ServerName serverName, Set<String> queues) {
  }

  @Override
  public List<ServerName> clearDeadServers(List<ServerName> servers) {
      return new ArrayList<>();
  }

  // only avaliable in HBase-2.1+
  public void cloneTableSchema(TableName tableName, TableName newTableName,
      boolean preserveSplits) {
    throw new UnsupportedOperationException("cloneTableSchema not supported");

  }

  // only avaliable in HBase-2.2+
  public Future<Void> createTableAsync(TableDescriptor desc) throws IOException {
    createTable(desc);
    return getVoidFuture();
  }

  @Override
  public Future<Void> createTableAsync(TableDescriptor desc, byte[][] splitKeys) throws IOException {
    createTable(desc, splitKeys);
    return getVoidFuture();
  }

  @Override
  public Future<Void> deleteTableAsync(TableName tableName) throws IOException {
    deleteTable(tableName);
    return getVoidFuture();
  }

  @Override
  public Future<Void> truncateTableAsync(TableName tableName, boolean preserveSplits) throws IOException {
    truncateTable(tableName, preserveSplits);
    return getVoidFuture();
  }

  @Override
  public Future<Void> enableTableAsync(TableName tableName) throws IOException {
    enableTable(tableName);
    return getVoidFuture();
  }

  @Override
  public Future<Void> disableTableAsync(TableName tableName) throws IOException {
    disableTable(tableName);
    return getVoidFuture();
  }

  @Override
  public Pair<Integer, Integer> getAlterStatus(TableName tableName) {
    // return a fake pair
    return new Pair<>(0, 1);
  }

  @Override
  public Pair<Integer, Integer> getAlterStatus(byte[] tableName) {
    // return a fake pair
    return new Pair<>(0, 1);
  }

  @Override
  public Future<Void> deleteColumnFamilyAsync(TableName tableName, byte[] columnFamily) throws IOException {
    deleteColumnFamily(tableName, columnFamily);
    return getVoidFuture();
  }

  @Override
  public Future<Void> addColumnFamilyAsync(TableName tableName, ColumnFamilyDescriptor columnFamily) throws IOException {
    addColumnFamily(tableName, columnFamily);
    return getVoidFuture();
  }

  @Override
  public Future<Void> modifyColumnFamilyAsync(TableName tableName, ColumnFamilyDescriptor columnFamily) throws IOException {
    modifyColumnFamily(tableName, columnFamily);
    return getVoidFuture();
  }

  @Override
  public Future<Void> deleteNamespaceAsync(String name) throws IOException {
    deleteNamespace(name);
    return getVoidFuture();
  }

  // only avaliable in HBase-2.2+
  public Map<TableName, Long> getSpaceQuotaTableSizes() throws IOException {
    throw new UnsupportedOperationException("getSpaceQuotaTableSizes not supported");
  }
  // only avaliable in HBase-2.2+@Override
  public Map<TableName, SpaceQuotaSnapshot> getRegionServerSpaceQuotaSnapshots(
      ServerName serverName) throws IOException {
    throw new UnsupportedOperationException(
        "getRegionServerSpaceQuotaSnapshots not supported");
  }

  // only avaliable in HBase-2.2+
  public SpaceQuotaSnapshot getCurrentSpaceQuotaSnapshot(String namespace) throws IOException {
    throw new UnsupportedOperationException("getCurrentSpaceQuotaSnapshot not supported");
  }

  // only avaliable in HBase-2.2+
  public SpaceQuotaSnapshot getCurrentSpaceQuotaSnapshot(TableName tableName) throws IOException {
    throw new UnsupportedOperationException("getCurrentSpaceQuotaSnapshot not supported");
  }

  // only avaliable in HBase-2.2+
  public void grant(UserPermission userPermission, boolean mergeExistingPermissions) {
    // trivial
  }

  // only avaliable in HBase-2.2+
  public void revoke(UserPermission userPermission) {
    // trivial
  }

  // only avaliable in HBase-2.2+
  /** @Override
  public List<UserPermission>
  getUserPermissions(GetUserPermissionsRequest getUserPermissionsRequest) {
  throw new UnsupportedOperationException("getUserPermissions not supported");
  } */

  // only avaliable in HBase-2.2+
  public Future<Void> splitRegionAsync(byte[] regionName) throws IOException {
    return splitRegionAsync(regionName, null);
  }

  @Override
  public void move(byte[] bytes, byte[] bytes1) throws IOException {
  }


  @Override
  public void splitRegionSync(byte[] regionName, byte[] splitPoint) throws IOException {
    splitRegion(regionName, splitPoint);
  }


  @Override
  public void splitRegionSync(byte[] regionName, byte[] splitPoint,
      final long timeout, final TimeUnit units) throws IOException {
    splitRegionSync(regionName, splitPoint);
  }

  @Override
  public byte[][] rollHLogWriter(String serverName) {
    return null;
  }

  @Override
  public void mergeRegionsSync(byte[] nameOfRegionA, byte[] nameOfRegionB, boolean forcible)
      throws IOException {
    mergeRegions(nameOfRegionA, nameOfRegionB, forcible);
  }

  @Override
  Future<Void> splitRegionAsync(RegionInfo hri, byte[] splitPoint) throws IOException {
    splitRegion(hri.getRegionName(), splitPoint);
    return getVoidFuture();
  }

  @Override
  Pair<RegionInfo, ServerName> getRegion(byte[] regionName) throws IOException {
    return new Pair<>();
  }

  @Override
  public List<AliHBaseIndexDescriptor> describeIndex(TableName dataTableName) throws IOException {
    return proxy.describeIndex(dataTableName);
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor indexDescriptor) throws IOException {
    proxy.createIndex(indexDescriptor);
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor indexDescriptor, byte[][] splitKeys) throws IOException {
    proxy.createIndex(indexDescriptor, splitKeys);
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor index,
      byte[] startKey, byte[] endKey, int numRegions) throws IOException {
    proxy.createIndex(index, startKey, endKey, numRegions);
  }

  @Override
  public void deleteIndex(String indexName, TableName dataTable) throws IOException {
    proxy.deleteIndex(indexName, dataTable);
  }

  @Override
  public void offlineIndex(String indexName, TableName dataTable)
      throws IOException {
    proxy.offlineIndex(indexName, dataTable);
  }

  @Override
  public void registerBDSCluster(String hbaseSourceName, String bdsClusterkey,
      String hbaseConnectionString, String username, String password) throws IOException {
    proxy.registerBDSCluster(hbaseSourceName, bdsClusterkey, hbaseConnectionString, username,
        password);
  }

  @Override
  public void registerSolrCluster(String solrSourceName, String solrConnectionStr)
      throws IOException {
    proxy.registerSolrCluster(solrSourceName, solrConnectionStr);
  }

  @Override
  public void registerESCluster(String esSourceName, String esConnectionStr, String userName,
      String password) throws IOException {
    proxy.registerESCluster(esSourceName, esConnectionStr, userName, password);
  }

  @Override
  public void unregisterSolrCluster(boolean force) throws IOException {
    proxy.unregisterSolrCluster(force);
  }

  @Override
  public void unregisterESCluster(boolean force) throws IOException {
    proxy.unregisterESCluster(force);
  }

  @Override
  public void unregisterBDSCluster(boolean force) throws IOException {
    proxy.unregisterBDSCluster(force);
  }

  private Future<Void> getVoidFuture() {
    return new Future<Void>() {
      @Override
      public boolean cancel(boolean mayInterruptIfRunning) {
        return false;
      }

      @Override
      public boolean isCancelled() {
        return false;
      }

      @Override
      public boolean isDone() {
        return true;
      }

      @Override
      public Void get() throws InterruptedException, ExecutionException {
        return null;
      }

      @Override
      public Void get(long timeout, TimeUnit unit)
          throws InterruptedException, ExecutionException, TimeoutException {
        return null;
      }
    };
  }
}
