/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.ExecutorService;

import com.alibaba.hbase.client.AliHBaseUEClientService;
import com.alibaba.hbase.client.AliHBaseUEConnection;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.MasterNotRunningException;
import org.apache.hadoop.hbase.RegionLocations;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.ZooKeeperConnectionException;
import org.apache.hadoop.hbase.client.backoff.ClientBackoffPolicy;
import org.apache.hadoop.hbase.ipc.RpcControllerFactory;
import org.apache.hadoop.hbase.security.User;

import org.apache.hadoop.hbase.shaded.protobuf.generated.AdminProtos;
import org.apache.hadoop.hbase.shaded.protobuf.generated.ClientProtos;

public class AliHBaseUEClusterConnection implements ClusterConnection {

  private AliHBaseUEConnection connection;

  public AliHBaseUEClusterConnection(Configuration conf, ExecutorService pool, final User user) throws
      IOException {
    this.connection = new AliHBaseUEConnection(conf, pool, user);
  }

  public AliHBaseUEClusterConnection(AliHBaseUEConnection connection) {
    this.connection = connection;
  }

  public AliHBaseUEConnection getAliHBaseUEConnection(){
    return connection;
  }

  /**
   * Only meaningful in Thrift impl
   * @param cuLimit
   */
  public void setCULimit(int cuLimit) {
    this.connection.setCULimit(cuLimit);
  }

  @Override
  public boolean isMasterRunning() throws MasterNotRunningException, ZooKeeperConnectionException {
    return true;
  }

  @Override
  public boolean isTableAvailable(TableName tableName, byte[][] splitKeys) throws IOException {
    try (Admin admin = getAdmin()) {
      return admin.isTableAvailable(tableName, splitKeys);
    }
  }

  @Override
  public boolean isTableEnabled(TableName tableName) throws IOException {
    try(Admin admin = getAdmin()) {
      return admin.isTableAvailable(tableName);
    }
  }

  @Override
  public boolean isTableDisabled(TableName tableName) throws IOException {
    try(Admin admin = getAdmin()) {
      return admin.isTableDisabled(tableName);
    }
  }

  @Override
  public TableState getTableState(TableName tableName) throws IOException {
    try(Admin admin = getAdmin()) {
      if (isTableEnabled(tableName)) {
        return new TableState(tableName, TableState.State.ENABLED);
      } else if (isTableDisabled(tableName)) {
        return new TableState(tableName, TableState.State.DISABLED);
      } else {
        return new TableState(tableName, TableState.State.DISABLING);
      }
    }
  }

  @Override
  public HRegionLocation locateRegion(TableName tableName, byte[] row) throws IOException {
    try (AliHBaseUETable table = (AliHBaseUETable)getTable(tableName)) {
      return table.getRegionLocation(row);
    }
  }

  @Override
  public void clearRegionCache() {

  }

  @Override
  public void cacheLocation(TableName tableName, RegionLocations location) {

  }

  @Override
  public void clearRegionCache(TableName tableName) {

  }

  @Override
  public void deleteCachedRegionLocation(HRegionLocation location) {

  }

  @Override
  public HRegionLocation relocateRegion(TableName tableName, byte[] row) throws IOException {
    return locateRegion(tableName, row);
  }

  @Override
  public RegionLocations relocateRegion(TableName tableName, byte[] row, int replicaId)
      throws IOException {
    RegionLocations regionLocations = new RegionLocations(locateRegion(tableName, row));
    return regionLocations;
  }

  @Override
  public void updateCachedLocations(TableName tableName, byte[] regionName, byte[] rowkey,
      Object exception, ServerName source) {

  }

  @Override
  public HRegionLocation locateRegion(byte[] regionName) throws IOException {
    throw new UnsupportedOperationException("locateRegion by region name is not supported");
  }

  @Override
  public List<HRegionLocation> locateRegions(TableName tableName) throws IOException {
    return locateRegions(tableName, false, false);
  }

  @Override
  public List<HRegionLocation> locateRegions(TableName tableName, boolean useCache,
      boolean offlined) throws IOException {
    try (AliHBaseUETable table = (AliHBaseUETable)getTable(tableName)) {
      return table.getAllRegionLocations();
    }
  }

  @Override
  public RegionLocations locateRegion(TableName tableName, byte[] row, boolean useCache,
      boolean retry) throws IOException {
    RegionLocations regionLocations = new RegionLocations(locateRegion(tableName, row));
    return regionLocations;

  }

  @Override
  public RegionLocations locateRegion(TableName tableName, byte[] row, boolean useCache,
      boolean retry, int replicaId) throws IOException {
    return locateRegion(tableName, row, useCache, retry);
  }

  @Override
  public MasterKeepAliveConnection getMaster() throws IOException {
    return null;
  }

  @Override
  public AdminProtos.AdminService.BlockingInterface getAdminForMaster() throws IOException {
    return null;
  }

  @Override
  public AdminProtos.AdminService.BlockingInterface getAdmin(ServerName serverName)
      throws IOException {
    return null;
  }

  @Override
  public ClientProtos.ClientService.BlockingInterface getClient(ServerName serverName)
      throws IOException {
    return new AliHBaseUEClientService(connection);
  }

  @Override
  public HRegionLocation getRegionLocation(TableName tableName, byte[] row, boolean reload)
      throws IOException {
    return locateRegion(tableName, row);
  }

  @Override
  public void clearCaches(ServerName sn) {
  }

  @Override
  public NonceGenerator getNonceGenerator() {
    return null;
  }

  @Override
  public AsyncProcess getAsyncProcess() {
    return null;
  }

  @Override
  public RpcRetryingCallerFactory getNewRpcRetryingCallerFactory(Configuration conf) {
    return null;
  }

  @Override
  public RpcRetryingCallerFactory getRpcRetryingCallerFactory() {
    return null;
  }

  @Override
  public RpcControllerFactory getRpcControllerFactory() {
    return null;
  }

  @Override
  public ConnectionConfiguration getConnectionConfiguration() {
    return new ConnectionConfiguration(connection.getConfiguration());
  }

  @Override
  public ServerStatisticTracker getStatisticsTracker() {
    return null;
  }

  @Override
  public ClientBackoffPolicy getBackoffPolicy() {
    return null;
  }

  @Override
  public MetricsConnection getConnectionMetrics() {
    return null;
  }

  @Override
  public boolean hasCellBlockSupport() {
    return false;
  }

  @Override
  public int getCurrentNrHRS() throws IOException {
    return 0;
  }


  @Override
  public Table getTable(TableName tableName) throws IOException {
    return connection.getTable(tableName);
  }

  @Override
  public Table getTable(TableName tableName, ExecutorService pool) throws IOException {
    return connection.getTable(tableName, pool);
  }

  @Override
  public Configuration getConfiguration() {
    return connection.getConfiguration();
  }

  @Override
  public BufferedMutator getBufferedMutator(TableName tableName) throws IOException {
    return connection.getBufferedMutator(tableName);
  }

  @Override
  public BufferedMutator getBufferedMutator(BufferedMutatorParams params) throws IOException {
    return connection.getBufferedMutator(params);
  }

  @Override
  public RegionLocator getRegionLocator(TableName tableName) throws IOException {
    return connection.getRegionLocator(tableName);
  }

  @Override
  public Admin getAdmin() throws IOException {
    return connection.getAdmin();
  }

  @Override
  public void close() throws IOException {
    connection.close();
  }

  @Override
  public boolean isClosed() {
    return connection.isClosed();
  }

  @Override
  public TableBuilder getTableBuilder(TableName tableName, ExecutorService pool) {
    return connection.getTableBuilder(tableName, pool);
  }

  @Override
  public void abort(String why, Throwable e) {
    connection.abort(why, e);
  }

  @Override
  public boolean isAborted() {
    return connection.isAborted();
  }
}
