/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.client;

import java.util.concurrent.ExecutorService;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.security.User;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Created by mazhenlin on 2019/1/22.
 */
@InterfaceAudience.Private
public class AsyncLayeredConnectionImpl extends AsyncConnectionImpl {
  private static final Logger LOG = LoggerFactory.getLogger(AsyncLayeredConnectionImpl.class);

  public AsyncLayeredConnectionImpl(Configuration conf, AsyncRegistry registry, String clusterId,
      User user) {
    super(conf,registry,clusterId,user);
  }

  @Override
  public AsyncTableBuilder<AdvancedScanResultConsumer> getTableBuilder(TableName tableName) {
    return new AsyncTableBuilderBase<AdvancedScanResultConsumer>(tableName, connConf) {

      @Override
      public AsyncTable<AdvancedScanResultConsumer> build() {
        return AsyncLayeredConnectionImpl.getRawAsyncTable(AsyncLayeredConnectionImpl.this,this);
      }
    };
  }

  @Override
  public AsyncTableBuilder<ScanResultConsumer> getTableBuilder(TableName tableName,
      ExecutorService pool) {
    return new AsyncTableBuilderBase<ScanResultConsumer>(tableName, connConf) {
      @Override
      public AsyncTable<ScanResultConsumer> build() {
        RawAsyncTableImpl rawTable =
            AsyncLayeredConnectionImpl.getRawAsyncTable(AsyncLayeredConnectionImpl.this,this);
        return new AsyncTableImpl(AsyncLayeredConnectionImpl.this, rawTable, pool);
      }
    };
  }

  private static RawAsyncTableImpl getRawAsyncTable(AsyncConnectionImpl conn,
      AsyncTableBuilderBase<?> builder) {
    long hotTtl = -1;

    if( builder.tableName.isSystemTable() ) {
      return new RawAsyncTableImpl(conn,builder);
    }

    AsyncAdmin admin = conn.getAdmin();
    try {
      TableDescriptor descriptor = admin.getDescriptor(builder.tableName).get();
      if (descriptor.getValue(HConstants.SINK_TABLE_NAME) != null) { /// hot table
        hotTtl = Long.parseLong(descriptor.getValue(HConstants.SOURCE_TTL));
        if (hotTtl > 0) {
          return new RawAsyncHotTableImpl(conn,builder,hotTtl);
        }
      } else if (descriptor.getValue(HConstants.SOURCE_TABLE_NAME) != null) { /// cold table
        TableName hotTable = TableName.valueOf(descriptor.getValue(HConstants.SOURCE_TABLE_NAME));
        TableDescriptor hotDescriptor = admin.getDescriptor(hotTable).get();
        if (builder.tableName.toString().equals(hotDescriptor.getValue(HConstants.SINK_TABLE_NAME))) {
          hotTtl = Long.parseLong(hotDescriptor.getValue(HConstants.SOURCE_TTL));
          if (hotTtl > 0) {
            return new RawAsyncColdTableImpl(conn,builder,hotTable,hotTtl);
          }
        }
      }
    } catch (Exception e) {
      LOG.error("exception caught when getting hot/cold tables", e);
    }
    return new RawAsyncTableImpl(conn,builder);
  }

}
