/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.ExecutorService;

import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.ipc.RpcControllerFactory;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;

/**
 * Created by mazhenlin on 2019/1/22.
 */
@InterfaceAudience.Private
@InterfaceStability.Stable
public class ColdHTable extends LayeredHTable {
  protected TableName hotTableName = null;
  protected LayeredHTable hotTable = null;
  protected long hotTtl = -1;
  /**
   * Creates an object to access a HBase table.
   * Used by HBase internally.  DO NOT USE. See {@link ConnectionFactory} class comment for how to
   * get a {@link Table} instance (use {@link Table} instead of {@link LayeredHTable}).
   *
   * @param connection           Connection to be used.
   * @param builder              The table builder
   * @param rpcCallerFactory     The RPC caller factory
   * @param rpcControllerFactory The RPC controller factory
   * @param pool                 ExecutorService to be used.
   */
  @InterfaceAudience.Private protected ColdHTable(final ClusterConnection connection,
      final TableBuilderBase builder, final RpcRetryingCallerFactory rpcCallerFactory,
      final RpcControllerFactory rpcControllerFactory, final ExecutorService pool,
      TableName hotTableName, long hotTtl ) {
    super(connection, builder, rpcCallerFactory, rpcControllerFactory, pool);
    this.hotTableName = hotTableName;
    this.hotTtl = hotTtl;
  }

  private LayeredHTable getHotTable() throws IOException {
    if( hotTable != null ) {
      return hotTable;
    } else if( hotTableName == null ) {
      return null;
    } else synchronized (this) {
      if( hotTable == null ) {
        hotTable = (LayeredHTable) getConnection().getTable(hotTableName);
      }
      return hotTable;
    }
  }

  /**
   * The underlying {@link LayeredHTable} must not be closed.
   * {@link Table#getScanner(Scan)} has other usage details.
   */
  @Override public ResultScanner getScanner(Scan scan) throws IOException {
    LayeredHTable hot = getHotTable();
    Scan hotScan = ClientUtil.makeHotTableScan(scan,hotTtl);
    return new MergeScanner(scan, hot.getRawScanner(hotScan), getRawScanner(scan));
  }

  @Override protected Result get(Get get, final boolean checkExistenceOnly) throws IOException {
    LayeredHTable hot = getHotTable();
    Get hotGet = ClientUtil.makeHotTableGet(get,hotTtl);
    Result hotr = hot.rawGet(hotGet, checkExistenceOnly);
    Result coldr = rawGet(get, checkExistenceOnly);
    return MergeScanner.mergeResultsOfSameRow(hotr, coldr, get.getMaxVersions());
  }

  @Override public Result[] get(List<Get> gets) throws IOException {
    if (gets.size() == 1) {
      return new Result[] { get(gets.get(0)) };
    }
    LayeredHTable hot = getHotTable();
    List<Get> hotGets = ClientUtil.makeHotTableGets(gets,hotTtl);

    Object[] r1 = rawGet(gets);
    Object[] r2 = hot.rawGet(hotGets);
    // Translate.
    Result[] results = new Result[r1.length];
    for (int i = 0; i < gets.size(); i++) {
      results[i] = MergeScanner.mergeResultsOfSameRow((Result) r1[i], (Result) r2[i],
          gets.get(i).getMaxVersions());
    }
    return results;
  }

}
