/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.concurrent.ExecutorService;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.security.User;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Created by mazhenlin on 2019/1/22.
 */
@InterfaceAudience.Private
public class LayeredConnectionImplementation extends ConnectionImplementation {
  private static final Logger LOG = LoggerFactory.getLogger(LayeredConnectionImplementation.class);
  /**
   * constructor
   *
   * @param conf Configuration object
   */
  LayeredConnectionImplementation(Configuration conf, ExecutorService pool, User user)
      throws IOException {
    super(conf,pool,user);
  }

  @Override
  public TableBuilder getTableBuilder(TableName tableName, ExecutorService pool) {
    return new TableBuilderBase(tableName, getConnectionConfiguration()) {
      @Override public Table build() {
        long hotTtl = -1;

        Admin admin = null;
        try {
          admin = getAdmin();
          TableDescriptor descriptor = admin.getDescriptor(tableName);
          if (descriptor.getValue(HConstants.SINK_TABLE_NAME) != null) { /// hot table
            hotTtl = Long.parseLong(descriptor.getValue(HConstants.SOURCE_TTL));
            if (hotTtl > 0) {
              return new HotHTable(LayeredConnectionImplementation.this, this,
                  getRpcRetryingCallerFactory(), getRpcControllerFactory(), pool, hotTtl);
            }
          } else if (descriptor.getValue(HConstants.SOURCE_TABLE_NAME) != null) { /// cold table
            TableName hotTableName = TableName.valueOf(descriptor.getValue(HConstants.SOURCE_TABLE_NAME));
            TableDescriptor hotDescriptor = admin.getDescriptor(hotTableName);
            if (tableName.toString().equals(hotDescriptor.getValue(HConstants.SINK_TABLE_NAME))) {
              hotTtl = Long.parseLong(hotDescriptor.getValue(HConstants.SOURCE_TTL));
              if (hotTtl > 0) {
                return new ColdHTable(LayeredConnectionImplementation.this, this,
                    getRpcRetryingCallerFactory(), getRpcControllerFactory(), pool, hotTableName,
                    hotTtl);
              }
            }
          }
          return new HTable(LayeredConnectionImplementation.this, this,
              getRpcRetryingCallerFactory(), getRpcControllerFactory(), pool);
        } catch (IOException e) {
          LOG.error("exception caught when getting hot/cold tables", e);
          return new HTable(LayeredConnectionImplementation.this, this,
              getRpcRetryingCallerFactory(), getRpcControllerFactory(), pool);
        } finally {
          if (admin != null) {
            try {
              admin.close();
            } catch (IOException e) {
              LOG.error("exception caught when getting hot/cold tables", e);
            }
          }
        }
      }
    };
  }

}
