/**
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.Arrays;
import java.util.LinkedList;

import org.apache.commons.lang.ArrayUtils;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CellComparatorImpl;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.client.metrics.ScanMetrics;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;

@InterfaceAudience.Private
@InterfaceStability.Stable
public class MergeScanner implements ResultScanner {
  ResultScanner scanner1;
  ResultScanner scanner2;
  Scan scan;
  Result r1,r2;
  boolean r1Exhausted = false,r2Exhausted = false;
  MergeScanner(Scan scan,ResultScanner scanner1,ResultScanner scanner2) {
    this.scan = scan;
    this.scanner1 = scanner1;
    this.scanner2 = scanner2;
  }
  @Override public Result next() throws IOException {
    Result r;
    if( r1 == null && !r1Exhausted ) {
      r1 = scanner1.next();
    }
    if( r2 == null && !r2Exhausted ) {
      r2 = scanner2.next();
    }
    if( r1 == null ) {
      r = r2;
      r1Exhausted = true;
      r2 = null;
    } else if( r2 == null ) {
      r = r1;
      r2Exhausted = true;
      r1 = null;
    } else {
      int compare = Bytes.compareTo(r1.getRow(),r2.getRow() );
      if( scan.isReversed() ) {
        compare *= -1;
      }
      if( compare < 0 ) {
        r = r1;
        r1 = null;
      } else if( compare == 0 ) {
        r = mergeResultsOfSameRow(r1,r2,scan.getMaxVersions());
        r1 = null;
        r2 = null;
      } else {
        r = r2;
        r2 = null;
      }
    }
    return r;
  }

  @Override public void close() {
    scanner1.close();
    scanner2.close();
  }

  @Override public boolean renewLease() {
    return scanner1.renewLease() && scanner2.renewLease();
  }

  ///TODO:metric的正确实现应该是把两个表的统计加起来
  @Override public ScanMetrics getScanMetrics() {
    return scanner2.getScanMetrics();
  }

  public static Result mergeResultsOfSameRow(Result r1,Result r2,int maxVersions) {
    if( r1 == null ) return r2;
    if( r2 == null ) return r1;
    Cell[] allCells = (Cell[]) ArrayUtils.addAll(r1.rawCells(),r2.rawCells());
    Arrays.sort(allCells,CellComparatorImpl.COMPARATOR);
    LinkedList<Cell> l = new LinkedList<>();
    Cell prevCell = null;
    int versions = 0;
    for( Cell cell: allCells ) {
      if( prevCell == null || !CellUtil.matchingColumn(prevCell,cell) ) {
        versions = 1;
        l.add(cell);
      } else {
        versions++;
        if( versions <= maxVersions && prevCell.getTimestamp() != cell.getTimestamp() ) {
          l.add(cell);
        }
      }
      prevCell = cell;
    }
    Cell[] result = new Cell[l.size()];
    l.toArray(result);
    return Result.create(result);
  }
}
