/**
 * Copyright (C) Alibaba Cloud Computing, 2012
 * All rights reserved.
 *
 * 版权所有 （C）阿里巴巴云计算，2012
 */

package com.aliyun.mns.common.auth;

import com.aliyuncs.auth.AlibabaCloudCredentials;
import com.aliyuncs.auth.AlibabaCloudCredentialsProvider;
import com.aliyuncs.auth.InstanceProfileCredentials;

import static com.aliyun.mns.common.utils.CodingUtils.assertParameterNotNull;

/**
 * 表示用户访问的授权信息。
 */
public class ServiceCredentials {
    private String accessKeyId;
    private String accessKeySecret;
    private String securityToken;
    private AlibabaCloudCredentialsProvider credentialsProvider;

    /**
     * 构造函数。
     */
    public ServiceCredentials() {
    }

    /**
     * 构造函数。
     *
     * @param accessKeyId     Access Key ID.
     * @param accessKeySecret Access Key Secret.
     * @param securityToken   security temp token. (optional)
     * @throws NullPointerException accessKeyId.accessKeySecret.....
     */
    public ServiceCredentials(String accessKeyId, String accessKeySecret, String securityToken) {
        setAccessKeyId(accessKeyId);
        setAccessKeySecret(accessKeySecret);
        setSecurityToken(securityToken);
    }

    /**
     * 构造函数。
     *
     * @param accessKeyId     Access Key ID。
     * @param accessKeySecret Access Key Secret。
     * @throws NullPointerException accessKeyId或accessKeySecret为空指针。
     */
    public ServiceCredentials(String accessKeyId, String accessKeySecret) {
        this(accessKeyId, accessKeySecret, "");
    }

    /**
     * 构造函数。
     *
     * @param credentialProvider credential provider.
     * @throws NullPointerException
     */
    public ServiceCredentials(AlibabaCloudCredentialsProvider credentialProvider) {
        setCredentialsProvider(credentialProvider);
    }


    /**
     * 获取访问用户的Access Key ID。
     *
     * @return Access Key ID。
     */
    public String getAccessKeyId() {
        return accessKeyId;
    }

    /**
     * 设置访问用户的Access ID。
     *
     * @param accessKeyId Access Key ID。
     */
    public void setAccessKeyId(String accessKeyId) {
        assertParameterNotNull(accessKeyId, "accessKeyId");
        this.accessKeyId = accessKeyId;
    }

    /**
     * 获取访问用户的Access Key Secret。
     *
     * @return Access Key Secret。
     */
    public String getAccessKeySecret() {
        return accessKeySecret;
    }

    /**
     * 设置访问用户的Access Key Secret。
     *
     * @param accessKeySecret Access Key Secret。
     */
    public void setAccessKeySecret(String accessKeySecret) {
        assertParameterNotNull(accessKeySecret, "accessKeySecret");

        this.accessKeySecret = accessKeySecret;
    }

    /**
     * 获取security token。
     *
     * @return security token
     */
    public String getSecurityToken() {
        return securityToken;
    }

    /**
     * 设置访问用户的security token
     *
     * @param securityToken token.
     */
    public void setSecurityToken(String securityToken) {
        this.securityToken = securityToken;
    }

    /**
     * 获取credential provider
     *
     * @return predential provider.
     */
    public AlibabaCloudCredentialsProvider getCredentialsProvider() {
        return credentialsProvider;
    }

    /**
     * 设置访问用户的credential provider
     *
     * @param credentialsProvider Provider.
     */
    public void setCredentialsProvider(AlibabaCloudCredentialsProvider credentialsProvider) {
        assertParameterNotNull(credentialsProvider, "credentialsProvider");
        this.credentialsProvider = credentialsProvider;
    }

    /**
     * 通过credential provider获取access id
     *
     * @return accessKeyId.
     */
    public String getAccessKeyIdByProvider() {
        if (credentialsProvider == null) {
            return null;
        }
        String tmpAccessKeyId;
        try {
            tmpAccessKeyId = credentialsProvider.getCredentials().getAccessKeyId();
        } catch (Exception e) {
            tmpAccessKeyId = null;
        }
        return tmpAccessKeyId;
    }

    /**
     * 通过credential provider获取access key
     *
     * @return accessKeySecret.
     */
    public String getAccessKeySecretByProvider() {
        if (credentialsProvider == null) {
            return null;
        }
        String tmpAccesskeySecret;
        try {
            tmpAccesskeySecret = credentialsProvider.getCredentials().getAccessKeySecret();
        } catch (Exception e) {
            tmpAccesskeySecret = null;
        }
        return tmpAccesskeySecret;
    }

    /**
     * 通过credential provider获取security token
     *
     * @return securityToken.
     */
    public String getSecurityTokenByProvider() {
        if (credentialsProvider == null) {
            return null;
        }

        String tmpSecurityToken;
        try {
            AlibabaCloudCredentials credential = credentialsProvider.getCredentials();
            if (credential instanceof InstanceProfileCredentials) {
                tmpSecurityToken = ((InstanceProfileCredentials) credential).getSessionToken();
            } else {
                tmpSecurityToken = null;
            }
        } catch (Exception e) {
            tmpSecurityToken = null;
        }
        return tmpSecurityToken;
    }
}
