package com.aliyun.odps.io;

import com.google.gson.*;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;

public class IoUtils {
  /**
   * If data has been output by ODPS through external table, this returns the expected location of the
   * meta file generated by ODPS, see {@link parseOutputSubDirectoriesFromMeta(String)} for getting the
   * list of valid subdirectories output from meta file.
   * @param externalTableLocation the location specified when creating external table
   * @return the expected meta file location
   */
  public static String getMetaFileLocation(String externalTableLocation) {
    return  getSystemOutputDirectory(externalTableLocation) + ".meta";
  }

  /**
   * Get the system directory ODPS used when outputting unstructured data
   * @param externalTableLocation the location specified when creating external table
   * @return the directory in location path used for system output
   */
  public static String getSystemOutputDirectory(String externalTableLocation){
    if (!externalTableLocation.endsWith("/")){
      throw new RuntimeException("external table location must end with '/'.");
    }
    URI locationUri;
    try{
      locationUri = new URI(externalTableLocation);
    } catch (URISyntaxException e) {
      throw new RuntimeException(externalTableLocation
          + " is not a valid uri and cannot be used as external table location.");
    }
    return locationUri.toString() + ".odps/";
  }

  /**
   * Get valid directories for unstructured data output by ODPS.
   * Note: most of time system output directory will only include valid data (i.e., valid subdirectories)
   *       however, when an output job was interrupted (failed or killed), it is possible some invalid data will
   *       be left-over. In those cases, meta content provides the ground-truth as to which directories are
   *       complete and valid.
   * @param metaContent file content read from ODPS meta file
   * @return array of directories (inside the ODPS system directory)
   */
  public static String[] parseOutputSubDirectoriesFromMeta(String metaContent) {
    // actual parsing
    Gson gson = new GsonBuilder().disableHtmlEscaping().create();
    MetaContent content;
    try {
      content = gson.fromJson(metaContent, MetaContent.class);
    } catch(JsonParseException e) {
      throw new RuntimeException(
          "input metaContent cannot be parsed, make sure to provide the entire file content read from meta file.");
    }
    return content.getContentArray();
  }

  private class MetaContent{
    private List<String> dirs;
    public String[] getContentArray(){
      return dirs.toArray(new String[dirs.size()]);
    }
  }
}


