package com.aliyun.openservices.log.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.JSONSerializer;
import com.alibaba.fastjson.serializer.ObjectSerializer;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.aliyun.openservices.log.internal.Unmarshaller;
import net.sf.json.JSONArray;
import net.sf.json.JSONException;
import net.sf.json.JSONObject;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public final class JsonUtils {

    /**
     * Serialize Date to Unix timestamp and deserialize Unix timestamp to date.
     */
    private static final SerializeConfig SERIALIZE_CONFIG = new SerializeConfig();

    static {
        SERIALIZE_CONFIG.put(Date.class, new DateToUnixTimestampSerializer());
    }

    private JsonUtils() {
    }

    public static String serialize(Object object) {
        return JSON.toJSONString(object, SERIALIZE_CONFIG);
    }

    public static <T> List<T> readList(JSONObject value, String key, Unmarshaller<T> unmarshaller) {
        return readList(value.getJSONArray(key), unmarshaller);
    }

    public static <T> List<T> readList(JSONArray list, Unmarshaller<T> unmarshaller) {
        if (list == null || list.isEmpty()) {
            return Collections.emptyList();
        }
        List<T> values = new ArrayList<T>(list.size());
        for (int i = 0; i < list.size(); i++) {
            values.add(unmarshaller.unmarshal(list, i));
        }
        return values;
    }

    public static List<String> readOptionalStrings(JSONObject object, String key) {
        if (!object.has(key)) {
            return Collections.emptyList();
        }
        try {
            return readStringList(object, key);
        } catch (JSONException ex) {
            return Collections.emptyList();
        }
    }

    public static List<String> readStringList(JSONObject object, String key) {
        return readList(object, key, new Unmarshaller<String>() {
            @Override
            public String unmarshal(JSONArray value, int index) {
                return value.getString(index);
            }
        });
    }

    public static String readOptionalString(JSONObject object, String key) {
        return object.has(key) ? object.getString(key) : null;
    }

    public static boolean readBool(JSONObject object, String key, boolean defaultValue) {
        return object.has(key) ? object.getBoolean(key) : defaultValue;
    }

    public static Integer readOptionalInt(JSONObject object, String key) {
        return object.has(key) ? object.getInt(key) : null;
    }

    public static Date readOptionalDate(JSONObject object, String key) {
        return object.has(key) ? readDate(object, key) : null;
    }

    public static Date readDate(JSONObject object, String key) {
        return Utils.timestampToDate(object.getInt(key));
    }

    public static Map<String, String> readOptionalMap(JSONObject object, final String key) {
        if (!object.has(key)) {
            return Collections.emptyMap();
        }
        JSONObject value = object.getJSONObject(key);
        if (value.isNullObject()) {
            return Collections.emptyMap();
        }
        JSONArray names = value.names();
        Map<String, String> map = new HashMap<String, String>(names.size());
        for (int i = 0; i < names.size(); i++) {
            String fieldName = names.getString(i);
            map.put(fieldName, value.getString(fieldName));
        }
        return map;
    }

    /**
     * Serialize date to unix timestamp.
     */
    private static class DateToUnixTimestampSerializer implements ObjectSerializer {

        @Override
        public void write(JSONSerializer serializer,
                          Object date,
                          Object fieldName,
                          Type fieldType,
                          int features) {
            if (date != null) {
                serializer.write(Utils.dateToTimestamp((Date) date));
            }
        }
    }
}
