/*
 * Copyright (C) 2015 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.gms.googleservices

import java.util.HashSet
import java.util.HashMap
import java.util.SortedSet
import java.util.TreeSet
import java.util.regex.Matcher
import java.util.regex.Pattern
import org.gradle.BuildListener
import org.gradle.BuildResult
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.Dependency
import org.gradle.api.artifacts.DependencyResolutionListener
import org.gradle.api.artifacts.ProjectDependency
import org.gradle.api.artifacts.ResolvableDependencies
import org.gradle.api.initialization.Settings
import org.gradle.api.internal.artifacts.dependencies.DefaultProjectDependency
import org.gradle.api.invocation.Gradle
import org.gradle.api.GradleException
import org.gradle.api.Plugin
import org.gradle.api.Project

class GoogleServicesPlugin implements Plugin<Project> {

  public final static String JSON_FILE_NAME = 'google-services.json'

  public final static String MODULE_GROUP = "com.google.android.gms"
  public final static String MODULE_GROUP_FIREBASE = "com.google.firebase"
  public final static String MODULE_CORE = "firebase-core"
  public final static String MODULE_VERSION = "11.4.2"
  public final static String MINIMUM_VERSION = "9.0.0"
  // Some example of things that match this pattern are:
  // "aBunchOfFlavors/release"
  // "flavor/debug"
  // "test"
  // And here is an example with the capture groups in [square brackets]
  // [a][BunchOfFlavors]/[release]
  public final static Pattern VARIANT_PATTERN = ~/(?:([^\p{javaUpperCase}]+)((?:\p{javaUpperCase}[^\p{javaUpperCase}]*)*)\/)?([^\/]*)/
  // Some example of things that match this pattern are:
  // "TestTheFlavor"
  // "FlavorsOfTheRainbow"
  // "Test"
  // And here is an example with the capture groups in [square brackets]
  // "[Flavors][Of][The][Rainbow]"
  // Note: Pattern must be applied in a loop, not just once.
  public final static Pattern FLAVOR_PATTERN = ~/(\p{javaUpperCase}[^\p{javaUpperCase}]*)/
  // Some example of things that match this pattern are:
  // "[1]"
  // "[10]"
  // "[10.3.234]"
  // And here is an example with the capture group 1 in <triangle brackets>
  // [<1.2.3.4>]
  public static final Pattern VERSION_RANGE_PATTERN = ~/\[(\d+(\.\d+)*)\]/
  // These are the plugin types and the set of associated plugins whose presence should be checked for.
  private final static enum PluginType{
    APPLICATION([
      "android",
      "com.android.application"
    ]),
    LIBRARY([
      "android-library",
      "com.android.library"
    ]),
    FEATURE([
      "android-feature",
      "com.android.feature"
    ]),
    MODEL_APPLICATION([
      "com.android.model.application"
    ]),
    MODEL_LIBRARY(["com.android.model.library"])
    public PluginType(Collection plugins) {
      this.plugins = plugins
    }
    private final Collection plugins
    public Collection plugins() {
      return plugins
    }
  }

  public static HashMap<String, HashMap<String, HashSet<VersionRange>>> versionsByGroupAndName =new HashMap<>()

  public static checkForCoreRan = false

  public static GoogleServicesPluginConfig config = new GoogleServicesPluginConfig()

  @Override
  void apply(Project project) {
    config = project.extensions.create('googleServices', GoogleServicesPluginConfig)

    failOnVersionConflictForGroup(project, MODULE_GROUP)
    failOnVersionConflictForGroup(project, MODULE_GROUP_FIREBASE)
    for (PluginType pluginType : PluginType.values()) {
      for (String plugin : pluginType.plugins()) {
        if (project.plugins.hasPlugin(plugin)) {
          setupPlugin(project, pluginType)
          return
        }
      }
    }
    // If the google-service plugin is applied before any android plugin.
    // We should warn that google service plugin should be applied at
    // the bottom of build file.
    showWarningForPluginLocation(project)

    // Setup google-services plugin after android plugin is applied.
    project.plugins.withId("android", {
      setupPlugin(project, PluginType.APPLICATION)
    })
    project.plugins.withId("android-library", {
      setupPlugin(project, PluginType.LIBRARY)
    })
    project.plugins.withId("android-feature", {
      setupPlugin(project, PluginType.FEATURE)
    })
  }

  private void showWarningForPluginLocation(Project project) {
    project.getLogger().warn(
        "Warning: Please apply google-services plugin at the bottom of the build file.")
  }

  private void setupPlugin(Project project, PluginType pluginType) {
    switch (pluginType) {
      case PluginType.APPLICATION:
        project.android.applicationVariants.all { variant ->
          handleVariant(project, variant)
        }
        break
      case PluginType.LIBRARY:
        project.android.libraryVariants.all { variant ->
          handleVariant(project, variant)
        }
        break
      case PluginType.FEATURE:
        project.android.featureVariants.all { variant ->
          handleVariant(project, variant)
        }
        break
      case PluginType.MODEL_APPLICATION:
        project.model.android.applicationVariants.all { variant ->
          handleVariant(project, variant)
        }
        break
      case PluginType.MODEL_LIBRARY:
        project.model.android.libraryVariants.all { variant ->
          handleVariant(project, variant)
        }
        break
    }
  }


  private static void handleVariant(Project project,
      def variant) {

    File quickstartFile = null
    List<String> fileLocations = getJsonLocations("$variant.dirName", project)
    String searchedLocation = System.lineSeparator()
    for (String location : fileLocations) {
      File jsonFile = project.file(location + '/' + JSON_FILE_NAME)
      searchedLocation = searchedLocation + jsonFile.getPath() + System.lineSeparator()
      if (jsonFile.isFile()) {
        quickstartFile = jsonFile
        break
      }
    }

    if (quickstartFile == null) {
      project.getLogger().warn("Could not find $JSON_FILE_NAME while looking in $fileLocations")
      quickstartFile = project.file(JSON_FILE_NAME)
      searchedLocation = searchedLocation + quickstartFile.getPath()
    }

    File outputDir =
        project.file("$project.buildDir/generated/res/google-services/$variant.dirName")

    GoogleServicesTask task = project.tasks
        .create("process${variant.name.capitalize()}GoogleServices",
        GoogleServicesTask)

    task.quickstartFile = quickstartFile
    task.intermediateDir = outputDir
    task.packageName = variant.applicationId
    task.searchedLocation = searchedLocation

    // Use the target version for the task.
    variant.registerResGeneratingTask(task, outputDir)
  }

  private static List<String> splitVariantNames(String variant) {
    if (variant == null) {
      return Collections.emptyList()
    }
    List<String> flavors = new ArrayList<>()
    Matcher flavorMatcher = FLAVOR_PATTERN.matcher(variant)
    while (flavorMatcher.find()) {
      String match = flavorMatcher.group(1)
      if (match != null) {
        flavors.add(match.toLowerCase())
      }
    }
    return flavors
  }

  private static long countSlashes(String input) {
    return input.codePoints().filter{x -> x == '/'}.count()
  }

  static List<String> getJsonLocations(String variantDirname, Project project) {
    Matcher variantMatcher = VARIANT_PATTERN.matcher(variantDirname)
    List<String> fileLocations = new ArrayList<>()
    if (!variantMatcher.matches()) {
      project.getLogger().warn("$variantDirname failed to parse into flavors. Please start " +
        "all flavors with a lowercase character")
      fileLocations.add("src/$variantDirname")
      return fileLocations
    }
    List<String> flavorNames = new ArrayList<>()
    if (variantMatcher.group(1) != null) {
      flavorNames.add(variantMatcher.group(1).toLowerCase())
    }
    flavorNames.addAll(splitVariantNames(variantMatcher.group(2)))
    String buildType = variantMatcher.group(3)
    String flavorName = "${variantMatcher.group(1)}${variantMatcher.group(2)}"
    fileLocations.add("src/$flavorName/$buildType")
    fileLocations.add("src/$buildType/$flavorName")
    fileLocations.add("src/$flavorName")
    fileLocations.add("src/$buildType")
    fileLocations.add("src/$flavorName${buildType.capitalize()}")
    fileLocations.add("src/$buildType")
    String fileLocation = "src"
    for(String flavor : flavorNames) {
      fileLocation += "/$flavor"
      fileLocations.add(fileLocation)
      fileLocations.add("$fileLocation/$buildType")
      fileLocations.add("$fileLocation${buildType.capitalize()}")
    }
    fileLocations.unique().sort{a,b -> countSlashes(b) <=> countSlashes(a)}
    return fileLocations
  }

  static int versionCompare(String str1, String str2) {
    String[] vals1 = str1.split("\\.")
    String[] vals2 = str2.split("\\.")
    int i = 0
    while (i < vals1.length && i < vals2.length && vals1[i].equals(vals2[i])) {
      i++
    }
    if (i < vals1.length && i < vals2.length) {
      int diff = Integer.valueOf(vals1[i]).compareTo(Integer.valueOf(vals2[i]))
      return Integer.signum(diff)
    }
    return Integer.signum(vals1.length - vals2.length)
  }

  static void failOnVersionConflictForGroup(Project project, String groupPrefix) {
    versionsByGroupAndName = new HashMap<>()
    project.configurations.all {
      resolutionStrategy.eachDependency { details ->
        if (config.disableVersionCheck) {
          return
        }
        String group = details.requested.group
        String name = details.requested.name
        String version = details.requested.version
        if (group == null || name == null || version == null) {
          return
        }
        if (group.startsWith(groupPrefix)) {
          // Reset this when new dependencies are added so that the check can be performed again.
          // This is only relevant for when different flavors have different dependencies.
          checkForCoreRan = false
          if(!versionsByGroupAndName.containsKey(group)) {
            versionsByGroupAndName.put(group, new HashMap<>())
          }
          if(!versionsByGroupAndName.get(group).containsKey(name)) {
            versionsByGroupAndName.get(group).put(name, new HashSet<>())
          }
          VersionRange versionRange = VersionRange.fromString(version)
          if (versionRange != null) {
            versionsByGroupAndName.get(group).get(name).add(versionRange)
          }
        }
      }
    }
    project.getGradle().addListener(new DependencyResolutionListener() {
          @Override
          void beforeResolve(ResolvableDependencies resolvableDependencies) {
          }

          @Override
          void afterResolve(ResolvableDependencies resolvableDependencies) {
            checkForCore(project)
            resolvableDependencies.getResolutionResult().allComponents {artifact ->
              if (config.disableVersionCheck) {
                return
              }
              String group = artifact.moduleVersion.group
              String name = artifact.moduleVersion.name
              Version version = Version.fromString(artifact.moduleVersion.version)
              if (group == null || name == null || version == null) {
                return
              }
              if (group.startsWith(groupPrefix)) {
                HashSet<VersionRange> ranges = versionsByGroupAndName
                        .getOrDefault(group, new HashMap())
                        .getOrDefault(name, new HashSet())
                for (VersionRange range : ranges) {
                  if (!range.versionInRange(version)) {
                    throw new GradleException("The library $group:$name is being requested by "
                    + "various other libraries at $ranges, but resolves to $version.rawVersion. "
                    + "Disable the plugin and check your dependencies tree using ./gradlew :app:dependencies.")
                  }
                }
              }
            }
          }
        })
  }

  /**
   * Generates a warning if a granular project includes firebase, but does not include firebase-core
   */
  static void checkForCore(Project project) {
    if (checkForCoreRan) {
      return
    }
    if (versionsByGroupAndName.containsKey(MODULE_GROUP_FIREBASE)) {
      if (!versionsByGroupAndName.get(MODULE_GROUP_FIREBASE).containsKey(MODULE_CORE)) {
        checkForCoreRan = true
        project.getLogger().warn("Warning: The app gradle file must have a dependency on "
          + "com.google.firebase:firebase-core for Firebase services to work as intended.")
      }
    }
  }

  @groovy.transform.Immutable static class Version {
    String rawVersion, trimmedVersion
    public static Version fromString(String version) {
      if (version == null) {
        return null
      }
      return new Version(version, version.split("-")[0])
    }
  }
  @groovy.transform.Immutable static class VersionRange {
    boolean closedStart
    boolean closedEnd
    Version rangeStart
    Version rangeEnd

    public static VersionRange fromString(String versionRange) {
      Matcher versionRangeMatcher = VERSION_RANGE_PATTERN.matcher(versionRange)
      if (versionRangeMatcher.matches()) {
        Version version = Version.fromString(versionRangeMatcher.group(1))
        return new VersionRange(
            true,
            true,
            version,
            version)
      }
      return null
    }

    boolean versionInRange(Version version) {
      if (closedStart) {
        if (versionCompare(rangeStart.trimmedVersion, version.trimmedVersion) > 0) {
          return false
        }
      } else {
        if (versionCompare(rangeStart.trimmedVersion, version.trimmedVersion) >= 0) {
          return false
        }
      }
      if (closedEnd) {
        if (versionCompare(rangeEnd.trimmedVersion, version.trimmedVersion) < 0) {
          return false
        }
      } else {
        if (versionCompare(rangeEnd.trimmedVersion, version.trimmedVersion) <= 0) {
          return false
        }
      }
      return true
    }

    public String toString() {
      return ((closedStart ? "[" : "(") + rangeStart.trimmedVersion + "," + rangeEnd.trimmedVersion + (closedEnd ? "]" : ")"))
    }
  }

  public static class GoogleServicesPluginConfig {
    boolean disableVersionCheck = false
  }
}
