/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.clouddriver.kubernetes.op.job;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSetMultimap;
import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import com.google.gson.stream.JsonReader;
import com.netflix.spinnaker.clouddriver.jobs.JobExecutor;
import com.netflix.spinnaker.clouddriver.jobs.JobRequest;
import com.netflix.spinnaker.clouddriver.jobs.JobResult;
import com.netflix.spinnaker.clouddriver.jobs.local.ReaderConsumer;
import com.netflix.spinnaker.clouddriver.kubernetes.description.JsonPatch;
import com.netflix.spinnaker.clouddriver.kubernetes.description.KubernetesPatchOptions;
import com.netflix.spinnaker.clouddriver.kubernetes.description.KubernetesPodMetric;
import com.netflix.spinnaker.clouddriver.kubernetes.description.manifest.KubernetesKind;
import com.netflix.spinnaker.clouddriver.kubernetes.description.manifest.KubernetesManifest;
import com.netflix.spinnaker.clouddriver.kubernetes.op.job.MetricParser;
import com.netflix.spinnaker.clouddriver.kubernetes.security.KubernetesCredentials;
import com.netflix.spinnaker.clouddriver.kubernetes.security.KubernetesSelectorList;
import io.kubernetes.client.openapi.models.V1DeleteOptions;
import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.InputStream;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

@Component
public class KubectlJobExecutor {
    private static final Logger log = LoggerFactory.getLogger(KubectlJobExecutor.class);
    private static final String NOT_FOUND_STRING = "(NotFound)";
    private final JobExecutor jobExecutor;
    private final String executable;
    private final String oAuthExecutable;
    private final Gson gson = new Gson();

    @Autowired
    KubectlJobExecutor(JobExecutor jobExecutor, @Value(value="${kubernetes.kubectl.executable:kubectl}") String executable, @Value(value="${kubernetes.o-auth.executable:oauth2l}") String oAuthExecutable) {
        this.jobExecutor = jobExecutor;
        this.executable = executable;
        this.oAuthExecutable = oAuthExecutable;
    }

    public String logs(KubernetesCredentials credentials, String namespace, String podName, String containerName) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("logs");
        command.add(podName);
        command.add("-c=" + containerName);
        JobResult status = this.jobExecutor.runJob(new JobRequest(command));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to get logs from " + podName + "/" + containerName + " in " + namespace + ": " + status.getError());
        }
        return (String)status.getOutput();
    }

    public String jobLogs(KubernetesCredentials credentials, String namespace, String jobName, String containerName) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("logs");
        command.add("job/" + jobName);
        command.add("-c=" + containerName);
        JobResult status = this.jobExecutor.runJob(new JobRequest(command));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to get logs from job/" + jobName + " in " + namespace + ": " + status.getError());
        }
        return (String)status.getOutput();
    }

    public List<String> delete(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, KubernetesSelectorList labelSelectors, V1DeleteOptions deleteOptions) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("delete");
        command = this.kubectlLookupInfo(command, kind, name, labelSelectors);
        command.add("--ignore-not-found=true");
        if (deleteOptions.getOrphanDependents() != null) {
            command.add("--cascade=" + (deleteOptions.getOrphanDependents() == false));
        }
        if (deleteOptions.getGracePeriodSeconds() != null) {
            command.add("--grace-period=" + deleteOptions.getGracePeriodSeconds());
        }
        if (!Strings.isNullOrEmpty((String)deleteOptions.getPropagationPolicy())) {
            throw new IllegalArgumentException("Propagation policy is not yet supported as a delete option");
        }
        JobResult status = this.jobExecutor.runJob(new JobRequest(command));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            Object id = !Strings.isNullOrEmpty((String)name) ? kind + "/" + name : labelSelectors.toString();
            throw new KubectlException("Failed to delete " + (String)id + " from " + namespace + ": " + status.getError());
        }
        if (Strings.isNullOrEmpty((String)((String)status.getOutput())) || ((String)status.getOutput()).equals("No output from command.") || ((String)status.getOutput()).startsWith("No resources found")) {
            return new ArrayList<String>();
        }
        return Arrays.stream(((String)status.getOutput()).split("\n")).map(m -> m.substring(m.indexOf("\"") + 1)).map(m -> m.substring(0, m.lastIndexOf("\""))).collect(Collectors.toList());
    }

    public Void scale(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, int replicas) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("scale");
        command = this.kubectlLookupInfo(command, kind, name, null);
        command.add("--replicas=" + replicas);
        JobResult status = this.jobExecutor.runJob(new JobRequest(command));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to scale " + kind + "/" + name + " from " + namespace + ": " + status.getError());
        }
        return null;
    }

    public List<Integer> historyRollout(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("rollout");
        command.add("history");
        command.add(kind.toString() + "/" + name);
        JobResult status = this.jobExecutor.runJob(new JobRequest(command));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to get rollout history of " + kind + "/" + name + " from " + namespace + ": " + status.getError());
        }
        String stdout = (String)status.getOutput();
        if (Strings.isNullOrEmpty((String)stdout)) {
            return new ArrayList<Integer>();
        }
        List splitOutput = Arrays.stream(stdout.split("\n")).collect(Collectors.toList());
        if (splitOutput.size() <= 2) {
            return new ArrayList<Integer>();
        }
        splitOutput = splitOutput.subList(2, splitOutput.size());
        return splitOutput.stream().map(l -> l.split("[ \t]")).filter(l -> ((String[])l).length > 0).map(l -> l[0]).map(Integer::valueOf).collect(Collectors.toList());
    }

    public Void undoRollout(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, int revision) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("rollout");
        command.add("undo");
        command.add(kind.toString() + "/" + name);
        command.add("--to-revision=" + revision);
        JobResult status = this.jobExecutor.runJob(new JobRequest(command));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to undo rollout " + kind + "/" + name + " from " + namespace + ": " + status.getError());
        }
        return null;
    }

    public Void pauseRollout(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("rollout");
        command.add("pause");
        command.add(kind.toString() + "/" + name);
        JobResult status = this.jobExecutor.runJob(new JobRequest(command));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to pause rollout " + kind + "/" + name + " from " + namespace + ": " + status.getError());
        }
        return null;
    }

    public Void resumeRollout(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("rollout");
        command.add("resume");
        command.add(kind.toString() + "/" + name);
        JobResult status = this.jobExecutor.runJob(new JobRequest(command));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to resume rollout " + kind + "/" + name + " from " + namespace + ": " + status.getError());
        }
        return null;
    }

    public Void rollingRestart(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("rollout");
        command.add("restart");
        command.add(kind.toString() + "/" + name);
        JobResult status = this.jobExecutor.runJob(new JobRequest(command));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to complete rolling restart of " + kind + "/" + name + " from " + namespace + ": " + status.getError());
        }
        return null;
    }

    @Nullable
    public KubernetesManifest get(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name) {
        List<String> command = this.kubectlNamespacedGet(credentials, (List<KubernetesKind>)ImmutableList.of((Object)kind), namespace);
        command.add(name);
        JobResult status = this.jobExecutor.runJob(new JobRequest(command));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            if (status.getError().contains(NOT_FOUND_STRING)) {
                return null;
            }
            throw new KubectlException("Failed to read " + kind + " from " + namespace + ": " + status.getError());
        }
        try {
            return (KubernetesManifest)this.gson.fromJson((String)status.getOutput(), KubernetesManifest.class);
        }
        catch (JsonSyntaxException e) {
            throw new KubectlException("Failed to parse kubectl output: " + e.getMessage(), e);
        }
    }

    @Nonnull
    public ImmutableList<KubernetesManifest> eventsFor(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name) {
        List<String> command = this.kubectlNamespacedGet(credentials, (List<KubernetesKind>)ImmutableList.of((Object)KubernetesKind.EVENT), namespace);
        command.add("--field-selector");
        command.add(String.format("involvedObject.name=%s,involvedObject.kind=%s", name, StringUtils.capitalize((String)kind.toString())));
        JobResult status = this.jobExecutor.runJob(new JobRequest(command), this.parseManifestList());
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to read events from " + namespace + ": " + status.getError());
        }
        if (status.getError().contains("No resources found")) {
            return ImmutableList.of();
        }
        return (ImmutableList)status.getOutput();
    }

    @Nonnull
    public ImmutableList<KubernetesManifest> list(KubernetesCredentials credentials, List<KubernetesKind> kinds, String namespace, KubernetesSelectorList selectors) {
        JobResult status;
        List<String> command = this.kubectlNamespacedGet(credentials, kinds, namespace);
        if (selectors.isNotEmpty()) {
            command.add("-l=" + selectors.toString());
        }
        if ((status = this.jobExecutor.runJob(new JobRequest(command), this.parseManifestList())).getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to read " + kinds + " from " + namespace + ": " + status.getError());
        }
        if (status.getError().contains("No resources found")) {
            return ImmutableList.of();
        }
        return (ImmutableList)status.getOutput();
    }

    public KubernetesManifest deploy(KubernetesCredentials credentials, KubernetesManifest manifest) {
        List<String> command = this.kubectlAuthPrefix(credentials);
        String manifestAsJson = this.gson.toJson((Object)manifest);
        command.add("apply");
        command.add("-o");
        command.add("json");
        command.add("-f");
        command.add("-");
        JobResult status = this.jobExecutor.runJob(new JobRequest(command, (InputStream)new ByteArrayInputStream(manifestAsJson.getBytes(StandardCharsets.UTF_8))));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Deploy failed: " + status.getError());
        }
        try {
            return (KubernetesManifest)this.gson.fromJson((String)status.getOutput(), KubernetesManifest.class);
        }
        catch (JsonSyntaxException e) {
            throw new KubectlException("Failed to parse kubectl output: " + e.getMessage(), e);
        }
    }

    public KubernetesManifest replace(KubernetesCredentials credentials, KubernetesManifest manifest) {
        List<String> command = this.kubectlAuthPrefix(credentials);
        String manifestAsJson = this.gson.toJson((Object)manifest);
        command.add("replace");
        command.add("-o");
        command.add("json");
        command.add("-f");
        command.add("-");
        JobResult status = this.jobExecutor.runJob(new JobRequest(command, (InputStream)new ByteArrayInputStream(manifestAsJson.getBytes(StandardCharsets.UTF_8))));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            if (status.getError().contains(NOT_FOUND_STRING)) {
                throw new KubectlNotFoundException("Replace failed: " + status.getError());
            }
            throw new KubectlException("Replace failed: " + status.getError());
        }
        try {
            return (KubernetesManifest)this.gson.fromJson((String)status.getOutput(), KubernetesManifest.class);
        }
        catch (JsonSyntaxException e) {
            throw new KubectlException("Failed to parse kubectl output: " + e.getMessage(), e);
        }
    }

    public KubernetesManifest create(KubernetesCredentials credentials, KubernetesManifest manifest) {
        List<String> command = this.kubectlAuthPrefix(credentials);
        String manifestAsJson = this.gson.toJson((Object)manifest);
        command.add("create");
        command.add("-o");
        command.add("json");
        command.add("-f");
        command.add("-");
        JobResult status = this.jobExecutor.runJob(new JobRequest(command, (InputStream)new ByteArrayInputStream(manifestAsJson.getBytes(StandardCharsets.UTF_8))));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Create failed: " + status.getError());
        }
        try {
            return (KubernetesManifest)this.gson.fromJson((String)status.getOutput(), KubernetesManifest.class);
        }
        catch (JsonSyntaxException e) {
            throw new KubectlException("Failed to parse kubectl output: " + e.getMessage(), e);
        }
    }

    private List<String> kubectlAuthPrefix(KubernetesCredentials credentials) {
        ArrayList<String> command = new ArrayList<String>();
        if (!Strings.isNullOrEmpty((String)credentials.getKubectlExecutable())) {
            command.add(credentials.getKubectlExecutable());
        } else {
            command.add(this.executable);
        }
        if (credentials.getKubectlRequestTimeoutSeconds() != null) {
            command.add("--request-timeout=" + credentials.getKubectlRequestTimeoutSeconds());
        }
        if (credentials.isDebug()) {
            command.add("-v");
            command.add("9");
        }
        if (!credentials.isServiceAccount()) {
            String context;
            String kubeconfigFile;
            if (credentials.getOAuthServiceAccount() != null && !credentials.getOAuthServiceAccount().isEmpty()) {
                command.add("--token=" + this.getOAuthToken(credentials));
            }
            if (!Strings.isNullOrEmpty((String)(kubeconfigFile = credentials.getKubeconfigFile()))) {
                command.add("--kubeconfig=" + kubeconfigFile);
            }
            if (!Strings.isNullOrEmpty((String)(context = credentials.getContext()))) {
                command.add("--context=" + context);
            }
        }
        return command;
    }

    private List<String> kubectlLookupInfo(List<String> command, KubernetesKind kind, String name, KubernetesSelectorList labelSelectors) {
        if (!Strings.isNullOrEmpty((String)name)) {
            command.add(kind + "/" + name);
        } else {
            command.add(kind.toString());
        }
        if (labelSelectors != null && !labelSelectors.isEmpty()) {
            command.add("-l=" + labelSelectors);
        }
        return command;
    }

    private List<String> kubectlNamespacedAuthPrefix(KubernetesCredentials credentials, String namespace) {
        List<String> command = this.kubectlAuthPrefix(credentials);
        if (!Strings.isNullOrEmpty((String)namespace)) {
            command.add("--namespace=" + namespace);
        }
        return command;
    }

    private List<String> kubectlNamespacedGet(KubernetesCredentials credentials, List<KubernetesKind> kind, String namespace) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("-o");
        command.add("json");
        command.add("get");
        command.add(kind.stream().map(KubernetesKind::toString).collect(Collectors.joining(",")));
        return command;
    }

    private String getOAuthToken(KubernetesCredentials credentials) {
        ArrayList<String> command = new ArrayList<String>();
        command.add(this.oAuthExecutable);
        command.add("fetch");
        command.add("--json");
        command.add(credentials.getOAuthServiceAccount());
        command.addAll(credentials.getOAuthScopes());
        JobResult status = this.jobExecutor.runJob(new JobRequest(command));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Could not fetch OAuth token: " + status.getError());
        }
        return (String)status.getOutput();
    }

    public ImmutableList<KubernetesPodMetric> topPod(KubernetesCredentials credentials, String namespace, @Nonnull String pod) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("top");
        command.add("po");
        if (!pod.isEmpty()) {
            command.add(pod);
        }
        command.add("--containers");
        JobResult status = this.jobExecutor.runJob(new JobRequest(command));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            if (status.getError().toLowerCase().contains("not available") || status.getError().toLowerCase().contains("not found")) {
                log.warn(String.format("Error fetching metrics for account %s: %s", credentials.getAccountName(), status.getError()));
                return ImmutableList.of();
            }
            throw new KubectlException("Could not read metrics: " + status.getError());
        }
        ImmutableSetMultimap<String, KubernetesPodMetric.ContainerMetric> metrics = MetricParser.parseMetrics((String)status.getOutput());
        return (ImmutableList)metrics.asMap().entrySet().stream().map(podMetrics -> KubernetesPodMetric.builder().podName((String)podMetrics.getKey()).namespace(namespace).containerMetrics((Iterable)podMetrics.getValue()).build()).collect(ImmutableList.toImmutableList());
    }

    public Void patch(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, KubernetesPatchOptions options, List<JsonPatch> patches) {
        return this.patch(credentials, kind, namespace, name, options, this.gson.toJson(patches));
    }

    public Void patch(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, KubernetesPatchOptions options, KubernetesManifest manifest) {
        return this.patch(credentials, kind, namespace, name, options, this.gson.toJson((Object)manifest));
    }

    private Void patch(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, KubernetesPatchOptions options, String patchBody) {
        String mergeStrategy;
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("patch");
        command.add(kind.toString());
        command.add(name);
        if (options.isRecord()) {
            command.add("--record");
        }
        if (!Strings.isNullOrEmpty((String)(mergeStrategy = options.getMergeStrategy().toString()))) {
            command.add("--type");
            command.add(mergeStrategy);
        }
        command.add("--patch");
        command.add(patchBody);
        JobResult status = this.jobExecutor.runJob(new JobRequest(command));
        if (status.getResult() != JobResult.Result.SUCCESS) {
            String errMsg = status.getError();
            if (Strings.isNullOrEmpty((String)errMsg)) {
                errMsg = (String)status.getOutput();
            }
            if (errMsg.contains("not patched")) {
                log.warn("No change occurred after patching {} {}:{}, ignoring", new Object[]{kind, namespace, name});
                return null;
            }
            throw new KubectlException("Patch failed: " + errMsg);
        }
        return null;
    }

    private ReaderConsumer<ImmutableList<KubernetesManifest>> parseManifestList() {
        return r -> {
            try {
                JsonReader reader = new JsonReader((Reader)r);
                try {
                    try {
                        reader.beginObject();
                    }
                    catch (EOFException e) {
                        ImmutableList immutableList = ImmutableList.of();
                        reader.close();
                        return immutableList;
                    }
                    ImmutableList.Builder manifestList = new ImmutableList.Builder();
                    while (reader.hasNext()) {
                        if (reader.nextName().equals("items")) {
                            reader.beginArray();
                            while (reader.hasNext()) {
                                KubernetesManifest manifest = (KubernetesManifest)this.gson.fromJson(reader, KubernetesManifest.class);
                                manifestList.add((Object)manifest);
                            }
                            reader.endArray();
                            continue;
                        }
                        reader.skipValue();
                    }
                    reader.endObject();
                    ImmutableList immutableList = manifestList.build();
                    return immutableList;
                }
                finally {
                    try {
                        reader.close();
                    }
                    catch (Throwable throwable2) {
                        Throwable throwable;
                        throwable.addSuppressed(throwable2);
                    }
                }
            }
            catch (JsonSyntaxException | IllegalStateException e) {
                throw new KubectlException("Failed to parse kubectl output: " + e.getMessage(), e);
            }
        };
    }

    public static class KubectlNotFoundException
    extends KubectlException {
        public KubectlNotFoundException(String message) {
            super(message);
        }
    }

    public static class KubectlException
    extends RuntimeException {
        public KubectlException(String message) {
            super(message);
        }

        public KubectlException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}

