package cn.com.duiba.activity.center.api.enums;


import cn.com.duiba.activity.center.api.dto.sign.SignRuleConfigDto;
import cn.com.duiba.activity.center.api.dto.sign.SignRuleItemDto;
import cn.com.duiba.activity.center.api.exception.ActivityCenterRuntimeException;

import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;

/**
 * 签到类型
 * Created by xiaoxuda on 2017/5/31.
 */
public enum SignTypeEnum {
    CONTINUE(0, "连续签到") {
        @Override
        public SignRuleItemDto getRuleItem(Integer continueDay, SignRuleConfigDto signRuleConfig) {
            int circle = signRuleConfig.getItems().size();
            int index = continueDay > circle ? circle : continueDay;
            return signRuleConfig.getItems().get(index - 1);
        }
    },
    CIRCLE(1, "周期签到") {
        @Override
        public SignRuleItemDto getRuleItem(Integer continueDay, SignRuleConfigDto signRuleConfig) {
            int circle = signRuleConfig.getItems().size();
            int index = continueDay % circle;
            index = index == 0 ? circle : index;
            return signRuleConfig.getItems().get(index - 1);
        }
    },
    CUSTOM(2, "自定义签到") {
        @Override
        public SignRuleItemDto getRuleItem(Integer continueDay, SignRuleConfigDto signRuleConfig) {
            return null;
        }
    },
    MONTH(3, "自然月签到") {
        @Override
        public SignRuleItemDto getRuleItem(Integer continueDay, SignRuleConfigDto signRuleConfig) {
            Calendar cal = Calendar.getInstance();
            int daysInMonth = cal.getActualMaximum(Calendar.DATE);
            int daysOfMonth = cal.get(Calendar.DAY_OF_MONTH);
            if(continueDay > daysOfMonth){
                continueDay = daysOfMonth;
            }
            if(continueDay == daysInMonth){
                return signRuleConfig.getItems().get(1);
            }
            SignRuleItemDto item =  signRuleConfig.getItems().get(0);
            if(item.getCredits() > 0) {//起始值设为0表示不执行加积分
                item.setCredits(item.getCredits() + (continueDay - 1) * signRuleConfig.getIncr());
            }
            if(item.getActivityCount() > 0) {//起始值设为0表示不执行加活动次数
                item.setActivityCount(item.getActivityCount() + (continueDay - 1) * signRuleConfig.getIncr());
            }
            return item;
        }
    };

    private static Map<Integer, SignTypeEnum> typeEnumMap = new HashMap<>();

    static {
        for (SignTypeEnum typeEnum : SignTypeEnum.values()) {
            typeEnumMap.put(typeEnum.getCode(), typeEnum);
        }
    }

    /**
     * 通过code获取签到类型
     *
     * @param code
     * @return
     */
    public static SignTypeEnum getByCode(int code) {
        SignTypeEnum type = typeEnumMap.get(code);
        if (type == null) {
            throw new ActivityCenterRuntimeException("不支持的签到类型");
        }
        return type;
    }

    SignTypeEnum(Integer code, String desc) {
        this.code = code;
        this.desc = desc;
    }

    private Integer code;
    private String desc;

    /**
     * 根据签到规则计算连续签到第continueDay天所适用的签到奖励条目
     *
     * @param continueDay    连续签到天数
     * @param signRuleConfig 签到规则
     * @return
     */
    public abstract SignRuleItemDto getRuleItem(Integer continueDay, SignRuleConfigDto signRuleConfig);

    public Integer getCode() {
        return code;
    }

    public String getDesc() {
        return desc;
    }
}
