package cn.com.duiba.activity.center.api.enums;


import cn.com.duiba.activity.center.api.dto.sign.SignRuleConfigDto;
import cn.com.duiba.activity.center.api.dto.sign.system.SignActivityDto;
import cn.com.duiba.activity.center.api.exception.ActivityCenterException;
import cn.com.duiba.wolf.utils.DateUtils;

import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;

/**
 * 签到类型
 * Created by xiaoxuda on 2017/5/31.
 */
public enum SignTypeEnum {
    CONTINUE(0, "连续签到"),
    CIRCLE(1, "连续周期签到") {
        @Override
        public Integer getContinueDayInCircle(SignActivityDto activity, int continueDay, boolean todaySigned) {
            SignRuleConfigDto ruleConfig = activity.getSignRuleConfig();
            int days = continueDay % ruleConfig.getItems().size();
            if (days == 0 && continueDay >= ruleConfig.getItems().size()) {
                if (!todaySigned) {//今天是周期的第一天
                    continueDay = 0;
                } else {//今天是周期的最后一天
                    continueDay = ruleConfig.getItems().size();
                }
            } else {
                continueDay = days;
            }
            return continueDay;
        }
    },
    CUSTOM(2, "自定义签到"),
    MONTH(3, "自然月签到") {
        @Override
        public Calendar getCirStartDate(SignActivityDto activity){
            Calendar calendar = Calendar.getInstance();
            calendar.set(Calendar.DAY_OF_MONTH, 1);
            return calendar;
        }

        @Override
        public Calendar getCirEndDate(Calendar start, SignActivityDto activity){
            Calendar calendar = Calendar.getInstance();
            calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
            return calendar;
        }

        @Override
        public Integer getCircleDays(SignActivityDto activity) {
            return Calendar.getInstance().getActualMaximum(Calendar.DAY_OF_MONTH);
        }

        @Override
        public Integer getContinueDayInCircle(SignActivityDto activity, int continueDay, boolean todaySigned) {
            Calendar calendar = Calendar.getInstance();
            if (continueDay >= calendar.get(Calendar.DAY_OF_MONTH)) {
                continueDay = calendar.get(Calendar.DAY_OF_MONTH);
                if (!todaySigned) {
                    continueDay = continueDay - 1;
                }
            }
            return continueDay;
        }
    },
    CUSTOM_CIRCLE(4, "自定义周期") {//暂时按连续签到计算奖励规则
        @Override
        public Calendar getCirStartDate(SignActivityDto activity) {
            SignRuleConfigDto config = activity.getSignRuleConfig();
            Calendar today = Calendar.getInstance();
            Calendar cusCirStart = Calendar.getInstance();
            cusCirStart.setTime(config.getCusCirStart());
            Calendar startDate = Calendar.getInstance();
            if (CustomCircleTypeEnum.FIXED_DAYS.equals(config.getCusCirType())) {
                int cirNum = (DateUtils.daysBetween(cusCirStart.getTime(), today.getTime()) + 1) % config.getCusCirDays();
                if (cirNum == 0) {
                    startDate.setTime(DateUtils.daysAddOrSub(today.getTime(), 1 - config.getCusCirDays()));
                } else {
                    startDate.setTime(DateUtils.daysAddOrSub(today.getTime(), 1 - cirNum));
                }
            } else if (CustomCircleTypeEnum.MONTH.equals(config.getCusCirType())) {
                int dayGap = today.get(Calendar.DAY_OF_MONTH) - cusCirStart.get(Calendar.DAY_OF_MONTH);
                startDate.set(Calendar.DAY_OF_MONTH, cusCirStart.get(Calendar.DAY_OF_MONTH));
                if (dayGap < 0) {
                    startDate.setTime(DateUtils.monthsAddOrSub(startDate.getTime(), -1));
                }
            }
            return startDate;
        }

        @Override
        public Calendar getCirEndDate(Calendar start, SignActivityDto activity) {
            SignRuleConfigDto config = activity.getSignRuleConfig();
            Calendar endDate = Calendar.getInstance();
            endDate.setTime(start.getTime());
            if (CustomCircleTypeEnum.FIXED_DAYS.equals(config.getCusCirType())) {
                endDate.set(Calendar.DAY_OF_MONTH, start.get(Calendar.DAY_OF_MONTH) + config.getCusCirDays() - 1);
            } else if (CustomCircleTypeEnum.MONTH.equals(config.getCusCirType())) {
                endDate.set(Calendar.DAY_OF_MONTH, endDate.getActualMaximum(Calendar.DAY_OF_MONTH));
            }
            return endDate;
        }

        @Override
        public Integer getCircleDays(SignActivityDto activity) {
            SignRuleConfigDto config = activity.getSignRuleConfig();
            if (CustomCircleTypeEnum.FIXED_DAYS.equals(config.getCusCirType())) {
                return config.getCusCirDays();
            } else if (CustomCircleTypeEnum.MONTH.equals(config.getCusCirType())) {
                return getCirStartDate(activity).getActualMaximum(Calendar.DAY_OF_MONTH);
            }
            return super.getCircleDays(activity);
        }

        @Override
        public Integer getContinueDayInCircle(SignActivityDto activity, int continueDay, boolean todaySigned) {
            Calendar calendar = Calendar.getInstance();
            int days = DateUtils.daysBetween(getCirStartDate(activity).getTime(), calendar.getTime()) + 1;
            if (continueDay >= days) {
                continueDay = days;
                if (!todaySigned) {
                    continueDay = continueDay - 1;
                }
            }
            return continueDay;
        }

    };

    private static Map<Integer, SignTypeEnum> typeEnumMap = new HashMap<>();

    static {
        for (SignTypeEnum typeEnum : SignTypeEnum.values()) {
            typeEnumMap.put(typeEnum.getCode(), typeEnum);
        }
    }

    /**
     * 通过code获取签到类型
     *
     * @param code
     * @return
     */
    public static SignTypeEnum getByCode(int code) {
        SignTypeEnum type = typeEnumMap.get(code);
        if (type == null) {
            throw new ActivityCenterException("不支持的签到类型");
        }
        return type;
    }

    private Integer code;
    private String desc;

    SignTypeEnum(Integer code, String desc) {
        this.code = code;
        this.desc = desc;
    }

    /**
     * 周期开始日期
     * @param activity
     * @return
     */
    public Calendar getCirStartDate(SignActivityDto activity){
        //默认已配置中配的为周期
        Calendar calendar = Calendar.getInstance();
        Integer beforeDate = activity.getDateArea(SignActivityDto.KEY_BEFOREDATE);
        if (beforeDate == null) {
            beforeDate = 15;//没有填的话默认15天
        }
        calendar.add(Calendar.DAY_OF_MONTH, -beforeDate);
        return calendar;
    }

    /**
     * 周期结束日期
     * @param start
     * @param activity
     * @return
     */
    public Calendar getCirEndDate(Calendar start, SignActivityDto activity){
        //默认已配置中配的为周期
        Calendar calendar = Calendar.getInstance();
        Integer afterDate = activity.getDateArea(SignActivityDto.KEY_AFTERDATE);
        if (afterDate == null) {
            afterDate = 15;//没有填的话默认15天
        }
        calendar.add(Calendar.DAY_OF_MONTH, afterDate);
        return calendar;
    }

    /**
     * 获取周期内的天数
     *
     * @return
     */
    public Integer getCircleDays(SignActivityDto activity) {
        SignRuleConfigDto config = activity.getSignRuleConfig();
        return config.getItems().size();
    }

    /**
     * 周期内连续签到天数计算
     *
     * @param activity
     * @param continueDay
     * @param todaySigned
     * @return
     */
    public Integer getContinueDayInCircle(SignActivityDto activity, int continueDay, boolean todaySigned) {
        return continueDay;
    }

    public Integer getCode() {
        return code;
    }

    public String getDesc() {
        return desc;
    }
}
