package cn.com.duiba.activity.center.api.enums;


import cn.com.duiba.activity.center.api.dto.sign.SignRewardConfigDto;
import cn.com.duiba.activity.center.api.dto.sign.SignRewardRuleDto;
import cn.com.duiba.activity.center.api.dto.sign.SignRuleConfigDto;
import cn.com.duiba.activity.center.api.exception.ActivityCenterException;
import cn.com.duiba.wolf.utils.DateUtils;

import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * 签到类型
 * Created by xiaoxuda on 2017/5/31.
 */
public enum SignTypeEnum {
    CONTINUE(0, "连续签到"),
    CIRCLE(1, "连续周期签到") {
        @Override
        public Integer getContinueDayInCircle(SignRuleConfigDto ruleConfig, SignRewardConfigDto rewardConfig,
                                              int continueDay, boolean todaySigned, Date signDate) {
            int cirDays = getCircleDays(ruleConfig, rewardConfig);
            int days = continueDay % cirDays;
            if (days == 0 && continueDay >= cirDays) {
                if (!todaySigned) {//今天是周期的第一天
                    continueDay = 0;
                } else {//今天是周期的最后一天
                    continueDay = cirDays;
                }
            } else {
                continueDay = days;
            }
            return continueDay;
        }

        @Override
        public Integer getCircleDays(SignRuleConfigDto ruleConfig, SignRewardConfigDto rewardConfig){
            int cirDays = 1;
            if(ruleConfig.getCusCirDays() != null){
                cirDays = ruleConfig.getCusCirDays();
            }else {
                SignRewardRuleDto reward = rewardConfig.getRwRules().get(SignRewardTypeEnum.CIRCLE);
                if(reward != null){
                    cirDays = reward.getItems().size();
                }
            }
            return cirDays;
        }

    },
    CUSTOM(2, "自定义签到"),
    CUSTOM_CIRCLE(4, "自定义周期") {//暂时按连续签到计算奖励规则
        @Override
        public Calendar getCirStartDate(SignRuleConfigDto ruleConfig, SignRewardConfigDto rewardConfig) {
            Calendar today = Calendar.getInstance();
            Calendar cusCirStart = Calendar.getInstance();
            cusCirStart.setTime(ruleConfig.getCusCirStart());
            Calendar startDate = Calendar.getInstance();
            if (CustomCircleTypeEnum.FIXED_DAYS.equals(ruleConfig.getCusCirType())) {
                int cirNum = (DateUtils.daysBetween(cusCirStart.getTime(), today.getTime()) + 1) % ruleConfig.getCusCirDays();
                if (cirNum == 0) {
                    startDate.setTime(DateUtils.daysAddOrSub(today.getTime(), 1 - ruleConfig.getCusCirDays()));
                } else {
                    startDate.setTime(DateUtils.daysAddOrSub(today.getTime(), 1 - cirNum));
                }
            } else if (CustomCircleTypeEnum.MONTH.equals(ruleConfig.getCusCirType())) {
                int dayGap = today.get(Calendar.DAY_OF_MONTH) - cusCirStart.get(Calendar.DAY_OF_MONTH);
                startDate.set(Calendar.DAY_OF_MONTH, cusCirStart.get(Calendar.DAY_OF_MONTH));
                if (dayGap < 0) {
                    startDate.setTime(DateUtils.monthsAddOrSub(startDate.getTime(), -1));
                }
            }
            return startDate;
        }

        @Override
        public Calendar getCirEndDate(Calendar start, SignRuleConfigDto ruleConfig, SignRewardConfigDto rewardConfig) {
            Calendar endDate = Calendar.getInstance();
            endDate.setTime(start.getTime());
            if (CustomCircleTypeEnum.FIXED_DAYS.equals(ruleConfig.getCusCirType())) {
                endDate.set(Calendar.DAY_OF_MONTH, start.get(Calendar.DAY_OF_MONTH) + ruleConfig.getCusCirDays() - 1);
            } else if (CustomCircleTypeEnum.MONTH.equals(ruleConfig.getCusCirType())) {
                endDate.set(Calendar.DAY_OF_MONTH, endDate.getActualMaximum(Calendar.DAY_OF_MONTH));
            }
            return endDate;
        }

        @Override
        public Integer getCircleDays(SignRuleConfigDto ruleConfig, SignRewardConfigDto rewardConfig) {
            if (CustomCircleTypeEnum.FIXED_DAYS.equals(ruleConfig.getCusCirType())) {
                return ruleConfig.getCusCirDays();
            } else if (CustomCircleTypeEnum.MONTH.equals(ruleConfig.getCusCirType())) {
                return getCirStartDate(ruleConfig, rewardConfig).getActualMaximum(Calendar.DAY_OF_MONTH);
            }
            return super.getCircleDays(ruleConfig, rewardConfig);
        }

        @Override
        public Integer getContinueDayInCircle(SignRuleConfigDto ruleConfig, SignRewardConfigDto rewardConfig,
                                              int continueDay, boolean todaySigned, Date signDate) {
            Calendar today = Calendar.getInstance();
            Calendar startDate = getCirStartDate(ruleConfig, rewardConfig);
            Calendar endDate = getCirEndDate(startDate, ruleConfig, rewardConfig);
            int days = DateUtils.daysBetween(startDate.getTime(), today.getTime()) + 1;
            int daysNextCir = DateUtils.daysBetween(endDate.getTime(), signDate);
            if(daysNextCir > 0){
                days = daysNextCir;
            }else{
                daysNextCir = DateUtils.daysBetween(today.getTime(), signDate);
                if(daysNextCir > 0){
                    days += daysNextCir;
                }
            }
            if (continueDay >= days) {
                continueDay = days;
                if (!todaySigned) {
                    continueDay = continueDay - 1;
                }
            }
            return continueDay;
        }

    };

    private static Map<Integer, SignTypeEnum> typeEnumMap = new HashMap<>();

    static {
        for (SignTypeEnum typeEnum : SignTypeEnum.values()) {
            typeEnumMap.put(typeEnum.getCode(), typeEnum);
        }
    }

    /**
     * 通过code获取签到类型
     *
     * @param code
     * @return
     */
    public static SignTypeEnum getByCode(int code) {
        SignTypeEnum type = typeEnumMap.get(code);
        if (type == null) {
            throw new ActivityCenterException("不支持的签到类型");
        }
        return type;
    }

    private Integer code;
    private String desc;

    SignTypeEnum(Integer code, String desc) {
        this.code = code;
        this.desc = desc;
    }

    /**
     * 周期开始日期
     * @param rewardConfig
     * @param ruleConfig
     * @return
     */
    public Calendar getCirStartDate(SignRuleConfigDto ruleConfig, SignRewardConfigDto rewardConfig){
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_MONTH, 1 - getCircleDays(ruleConfig, rewardConfig));
        return calendar;
    }

    /**
     * 周期结束日期
     * @param start
     * @param rewardConfig
     * @param ruleConfig
     * @return
     */
    public Calendar getCirEndDate(Calendar start, SignRuleConfigDto ruleConfig, SignRewardConfigDto rewardConfig){
        return Calendar.getInstance();
    }

    /**
     * 获取周期内的天数
     * @param rewardConfig
     * @param ruleConfig
     * @return
     */
    public Integer getCircleDays(SignRuleConfigDto ruleConfig, SignRewardConfigDto rewardConfig) {
        return 31;//默认最大31天
    }

    /**
     * 周期内连续签到天数计算
     * @param rewardConfig
     * @param ruleConfig
     * @param continueDay
     * @param todaySigned
     * @return
     */
    public Integer getContinueDayInCircle(SignRuleConfigDto ruleConfig, SignRewardConfigDto rewardConfig,
                                          int continueDay, boolean todaySigned, Date signDate) {
        return continueDay;
    }

    public Integer getCode() {
        return code;
    }

    public String getDesc() {
        return desc;
    }
}
