package cn.com.duiba.activity.center.api.rank;

import cn.com.duiba.activity.center.api.dto.ngame.SimpleConsumerRankDto;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ZSetOperations;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * 游戏排行榜
 * Created by hww on 2018/5/12 下午3:41.
 */
public final class NGameRank {

    /*游戏成绩上限*/
    private static final long MAX_SCORE = 137371844607L;
    /*游戏成绩下线*/
    private static final long MIN_SCORE = -1L;
    /*游戏时间计数器上限*/
    private static final long MAX_TIME_SEQ = 67108864L;
    /*计算战胜多少用户时，舍入分割分数线*/
    private static final BigDecimal CUT_PERCENTAGE = BigDecimal.valueOf(0.999);



    /** 数据key */
    private String dataKey;

    /** 最后排名key */
    private String lastRankKey;

    /** 游戏时间序列key */
    private String timeSepKey;

    /** 数据存在时间  单位：天 */
    private int dataKeyExpire;

    /** 分数排序方式 */
    private boolean isDesc;

    /** 排行榜最大数据量 */
    private Integer rankLimit;

    /** redis客户端 */
    private RedisTemplate<String, String> redisClient;

    //构造方法为包私有 只有工厂方法才可以调用
    NGameRank(String dataKey, String lastRankKey, String timeSepKey, int dataKeyExpire, boolean isDesc, Integer rankLimit, RedisTemplate<String, String> redisClient) {
        this.dataKey = dataKey;
        this.lastRankKey = lastRankKey;
        this.timeSepKey = timeSepKey;
        this.dataKeyExpire = dataKeyExpire;
        this.isDesc = isDesc;
        this.rankLimit = rankLimit;
        this.redisClient = redisClient;
    }

    public boolean add(String cid, long score) {
        //获取最低排名成绩
        long score5000 = getLimitRankScoreNew(redisClient);
        //根据isDesc判断该成绩是否在5000名以内
        boolean needTime = isDesc ? score > score5000 : score < score5000;
        //获取含时间成绩
        double nScore = getTimeScore(score, getTimeSeqNew(needTime));
        //写入redis
        Boolean ret = redisClient.opsForZSet().add(dataKey, cid, nScore);
        redisClient.expire(dataKey, dataKeyExpire, TimeUnit.DAYS);
        Long count = redisClient.opsForZSet().size(dataKey);
        //数据总数超过排名限制，则删除多余数据
        if (count > rankLimit) {
            removeOutData(count);
        }
        return ret;
    }

    public BigDecimal percentage(Long rank) {
        try {
            if (rank == null) {
                return BigDecimal.ZERO;
            }
            //获取游戏总用户
            Long count = redisClient.opsForZSet().zCard(dataKey);
            if (count == null || count == 0L) {
                return BigDecimal.ZERO;
            }
            BigDecimal percentage = new BigDecimal(count - rank + 1).divide(new BigDecimal(count), 6, RoundingMode.DOWN);
            if (percentage.compareTo(CUT_PERCENTAGE) > 0) {
                return percentage.setScale(4, RoundingMode.DOWN).movePointRight(2);
            } else {
                return percentage.setScale(4, RoundingMode.HALF_UP).movePointRight(2);
            }
        } catch (Exception e) {
            return BigDecimal.ZERO;
        }
    }

    public Long getCount() {
        return redisClient.opsForZSet().zCard(dataKey);
    }

    public List<SimpleConsumerRankDto> getConsumerRank(Integer count) {
        return findRankPage(0, count);
    }

    public List<SimpleConsumerRankDto> findRankPage(Integer start, Integer end) {
        List<SimpleConsumerRankDto> result = Lists.newArrayList();
        if (end == null) {
            return result;
        }
        Set<ZSetOperations.TypedTuple<String>> set = isDesc
                ? redisClient.opsForZSet().reverseRangeWithScores(dataKey, start, end)
                : redisClient.opsForZSet().rangeWithScores(dataKey, start, end);
        Iterator<ZSetOperations.TypedTuple<String>> iterator =  set.iterator();
        int rank = 0;
        while (iterator.hasNext()) {
            rank++;
            ZSetOperations.TypedTuple<String> temp = iterator.next();
            if (null == temp) {
                continue;
            }
            SimpleConsumerRankDto simpleConsumerRankDto = new SimpleConsumerRankDto();
            simpleConsumerRankDto.setConsumerId(temp.getValue());
            simpleConsumerRankDto.setMaxScoreStr(String.valueOf(getScore(temp.getScore())));
            simpleConsumerRankDto.setRank(String.valueOf(rank));
            result.add(simpleConsumerRankDto);
        }
        return result;
    }



    public Long getScore(String consumerId) {
        Long rank = getRank(consumerId);
        if (rank == null) {
            return null;
        } else {
            rank--;
        }
        Set<ZSetOperations.TypedTuple<String>> set;
        if (isDesc) {
            set = redisClient.opsForZSet().reverseRangeWithScores(dataKey, rank, rank);
        } else {
            set = redisClient.opsForZSet().rangeWithScores(dataKey, rank, rank);
        }
        if (CollectionUtils.isEmpty(set)) {
            return null;
        }
        Iterator<ZSetOperations.TypedTuple<String>> it = set.iterator();
        return getScore(it.next().getScore());
    }

    public Long getScoreByRank(Integer rank) {
        if (rank == null) {
            return null;
        } else {
            rank--;
        }
        Set<ZSetOperations.TypedTuple<String>> set;
        if (isDesc) {
            set = redisClient.opsForZSet().reverseRangeWithScores(dataKey, rank, rank);
        } else {
            set = redisClient.opsForZSet().rangeWithScores(dataKey, rank, rank);
        }
        if (CollectionUtils.isEmpty(set)) {
            return null;
        }
        Iterator<ZSetOperations.TypedTuple<String>> it = set.iterator();
        return getScore(it.next().getScore());
    }

    public Long getRank(String consumerId) {
        Long rawRank = isDesc
                ? redisClient.opsForZSet().reverseRank(dataKey, consumerId)
                : redisClient.opsForZSet().rank(dataKey, consumerId);
        return rawRank == null ? null : rawRank + 1;
    }

    private void removeOutData(long count) {
        //redis只会降序排列，如果实际是按照逆序排名，需要删除最前面的数据
        if (isDesc) {
            redisClient.opsForZSet().removeRange(dataKey, 0, count - (rankLimit + 1));
        } else {
            redisClient.opsForZSet().removeRange(dataKey, rankLimit, count);
        }
    }

    private double getTimeScore(long sc, long time) {
        if (sc > MAX_SCORE || sc < MIN_SCORE) {
            sc = sc > MAX_SCORE ? MAX_SCORE : MIN_SCORE;
        }
        long n = sc << 26;
        long last = n + time;
        return Double.longBitsToDouble(last);
    }

    private long getTimeSeqNew(boolean needTime) {
        if (!needTime) {
            return isDesc ? 0L : MAX_TIME_SEQ;
        }
        long timeSeq = redisClient.opsForValue().increment(timeSepKey, 1);
        redisClient.expire(timeSepKey, dataKeyExpire, TimeUnit.DAYS);
        timeSeq = timeSeq > MAX_TIME_SEQ ? MAX_TIME_SEQ : timeSeq;
        if (isDesc) {
            return MAX_TIME_SEQ - timeSeq;
        }
        return timeSeq;
    }

    private long getLimitRankScoreNew(RedisTemplate<String, String> redisClient) {
        //获取排行榜最后一名的分数
        String score = redisClient.opsForValue().get(lastRankKey);
        score = StringUtils.trim(score);
        if (score == null) {
            //写入新的最后一名分数
            Set<ZSetOperations.TypedTuple<String>> set;
            int index = rankLimit - 1;
            if (isDesc) {
                set = redisClient.opsForZSet().reverseRangeWithScores(dataKey, index, index);
                if (set == null || set.size() == 0) {
                    score = String.valueOf(MIN_SCORE);
                }
            } else {
                set = redisClient.opsForZSet().rangeWithScores(dataKey, index, index);
                if (set == null || set.size() == 0) {
                    score = String.valueOf(MAX_SCORE);
                }
            }
            if (score == null) {
                Iterator<ZSetOperations.TypedTuple<String>> it = set.iterator();
                score = String.valueOf(getScore(it.next().getScore()));
            }
            redisClient.opsForValue().set(lastRankKey, score, 30, TimeUnit.SECONDS);
        }
        try {
            return Long.valueOf(score);
        } catch (NumberFormatException e) {
            return 1;
        }
    }

    private long getScore(double timeScore) {
        return Double.doubleToLongBits(timeScore) >> 26;
    }

    /**
     * 更新指定用户的分数
     * 若分数未达到排行榜的最低限制,则无法进入排行榜,相当于移除了排名信息
     */
    public void update(String cid, Long totalScore) {
        if (StringUtils.isBlank(cid) || totalScore == null) {
            return;
        }
        clean(cid);
        add(cid, totalScore);
    }

    /**
     * 清除指定用户的分数
     */
    public void clean(String cid) {
        if (StringUtils.isBlank(cid)) {
            return;
        }
        redisClient.opsForZSet().remove(dataKey, cid);
    }
}
