package cn.com.duiba.activity.center.api.tool;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.List;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;

/**
 * 预约抽奖中签计算工具类
 * 
 * @author cgq
 * @since JDK 1.8
 */
@Slf4j
public class LotteryCalculatorUtil {
    
    /**
     * 中签计算结果
     */
    @Data
    public static class LotteryCalculationResult {
        
        /**
         * 基数A
         */
        private Long baseNumber;
        
        /**
         * 翻转数B
         */
        private Long flipNumber;
        
        /**
         * 起始中签号Y
         */
        private Long startWinningNumber;
        
        /**
         * 阶数Z
         */
        private Long orderLevel;
        
        /**
         * 中签号码列表
         */
        private List<Long> winningNumbers;
        
        /**
         * 预约次数X
         */
        private Long totalCount;
        
        /**
         * 中签配额
         */
        private Long winningQuota;
    }
    
    /**
     * 计算中签名单
     * 
     * @param shenzhenIndexStr 深证成指指数
     * @param sme100IndexStr 中小100指数
     * @param winningQuota 中签配额
     * @param totalCount 预约次数
     * @return 中签计算结果
     */
    public static LotteryCalculationResult calculateWinningList(
            String shenzhenIndexStr,
            String sme100IndexStr,
            Long winningQuota,
            Long totalCount) {
        BigDecimal shenzhenIndex = parseIndexFromString(shenzhenIndexStr, new BigDecimal("0"));
        BigDecimal sme100Index = parseIndexFromString(sme100IndexStr, new BigDecimal("0"));
        log.info("开始计算中签名单，深证成指: {}, 中小100指数: {}, 中签配额: {}, 预约次数: {}", 
                shenzhenIndex, sme100Index, winningQuota, totalCount);
        
        if (shenzhenIndex == null || sme100Index == null || winningQuota == null || totalCount == null) {
            throw new IllegalArgumentException("计算参数不能为空");
        }

        if (winningQuota <= 0 || totalCount <= 0) {
            throw new IllegalArgumentException("中签配额和预约次数必须大于0");
        }
        
        LotteryCalculationResult result = new LotteryCalculationResult();
        result.setTotalCount(totalCount);
        result.setWinningQuota(winningQuota);

        try {
            // 步骤1：计算基数A
            Long baseNumber = calculateBaseNumber(shenzhenIndex, sme100Index);
            result.setBaseNumber(baseNumber);
            log.info("计算基数A: {}", baseNumber);
            
            // 步骤2：计算翻转数B
            Long flipNumber = calculateFlipNumber(baseNumber);
            result.setFlipNumber(flipNumber);
            log.info("计算翻转数B: {}", flipNumber);
            
            // 步骤3：计算起始中签号Y
            Long startWinningNumber = calculateStartWinningNumber(flipNumber, totalCount);
            result.setStartWinningNumber(startWinningNumber);
            log.info("计算起始中签号Y: {}", startWinningNumber);
            
            // 步骤4：计算阶数Z
            Long orderLevel = calculateOrderLevel(totalCount, winningQuota);
            result.setOrderLevel(orderLevel);
            log.info("计算阶数Z: {}", orderLevel);

            // 如果中签配额大于预约次数，则中签号码为1开始,到预约次数结束
            if (winningQuota > totalCount) {
                List<Long> winningNumbers = new ArrayList<>();
                for (long i = 1; i <= totalCount; i++) {
                    winningNumbers.add(i);
                }
                result.setWinningNumbers(winningNumbers);
                return result;
            }
            
            // 步骤5：计算所有中签号码
            List<Long> winningNumbers = calculateAllWinningNumbers(
                    startWinningNumber, orderLevel, winningQuota, totalCount);
            result.setWinningNumbers(winningNumbers);
            log.info("计算中签号码完成，共{}个中签号: {}", winningNumbers.size(), winningNumbers);
            
            return result;
            
        } catch (Exception e) {
            log.error("计算中签名单失败", e);
            throw new RuntimeException("计算中签名单失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 计算基数A
     * A = (深证成指 × 100) × (中小100指数 × 100) × 10000
     * 
     * @param shenzhenIndex 深证成指指数
     * @param sme100Index 中小100指数
     * @return 基数A
     */
    private static Long calculateBaseNumber(BigDecimal shenzhenIndex, BigDecimal sme100Index) {
        // A = (深证成指*100 + 中小100*100) * 10000
        BigDecimal shenzhenPart = shenzhenIndex.multiply(BigDecimal.valueOf(100));
        BigDecimal sme100Part = sme100Index.multiply(BigDecimal.valueOf(100));
        BigDecimal sum = shenzhenPart.multiply(sme100Part);
        BigDecimal result = sum.multiply(BigDecimal.valueOf(10000));
        
        // 转换为整数，去掉小数位
        return result.setScale(0, RoundingMode.DOWN).longValue();
    }
    
    /**
     * 计算翻转数B
     * B = 将基数A对应的数字倒序排列（如首位是0，则直接抹去）
     * 
     * @param baseNumber 基数A
     * @return 翻转数B
     */
    private static Long calculateFlipNumber(Long baseNumber) {
        // 将数字转换为字符串
        String baseStr = baseNumber.toString();
        
        // 倒序排列
        StringBuilder reversed = new StringBuilder(baseStr).reverse();
        
        // 去掉前导0
        while (reversed.length() > 1 && reversed.charAt(0) == '0') {
            reversed.deleteCharAt(0);
        }
        
        return Long.valueOf(reversed.toString());
    }
    
    /**
     * 计算起始中签号Y
     * Y = B / X 后所得的余数加1
     * 
     * @param flipNumber 翻转数B
     * @param totalCount 预约次数X
     * @return 起始中签号Y
     */
    private static Long calculateStartWinningNumber(Long flipNumber, Long totalCount) {
        long remainder = flipNumber % totalCount;
        return remainder + 1;
    }
    
    /**
     * 计算阶数Z
     * Z = X / 中签配额，取整数（去掉小数点后的整数）
     * 
     * @param totalCount 预约次数X
     * @param winningQuota 中签配额
     * @return 阶数Z
     */
    private static Long calculateOrderLevel(Long totalCount, Long winningQuota) {
        return totalCount / winningQuota;
    }

    /**
     * 计算所有中签号码
     * 第N个中签号 = Y + Z * (N-1)
     * 如果第N个中签号码 > 预约次数，那么第N个中签号码 = Y + Z * (N-1) - 预约次数
     * 
     * @param startWinningNumber 起始中签号Y
     * @param orderLevel 阶数Z
     * @param winningQuota 中签配额
     * @param totalCount 预约次数X
     * @return 中签号码列表
     */
    private static List<Long> calculateAllWinningNumbers(
            Long startWinningNumber, Long orderLevel, Long winningQuota, Long totalCount) {
        
        List<Long> winningNumbers = new ArrayList<>();
        
        for (int n = 1; n <= winningQuota; n++) {
            // 第N个中签号 = Y + Z * (N-1)
            long winningNumber = startWinningNumber + orderLevel * (n - 1);
            
            // 如果中签号 > 预约次数，则减去预约次数
            while (winningNumber > totalCount) {
                winningNumber -= totalCount;
            }
            
            winningNumbers.add(winningNumber);
        }
        
        return winningNumbers;
    }

    /**
     * 解析指数字符串为BigDecimal
     *
     * @param indexStr 指数字符串
     * @param defaultValue 默认值
     * @return BigDecimal指数值
     */
    private static BigDecimal parseIndexFromString(String indexStr, BigDecimal defaultValue) {
        try {
            if (indexStr != null && !indexStr.trim().isEmpty()) {
                return new BigDecimal(indexStr.trim());
            }
        } catch (NumberFormatException e) {
            log.warn("解析指数字符串失败: {}, 使用默认值: {}", indexStr, defaultValue);
        }
        return defaultValue;
    }


    /**
     * 验证计算结果示例
     * 
     * @param args 主函数参数
     */
    public static void main(String[] args) {
        // 测试用例：深证成指=10393.72，中小100=6504.50，中签配额=40，预约次数=123456
        String shenzhenIndex = "12571.37";
        String sme100Index = "7718.66";
        Long winningQuota = 1L;
        Long totalCount = 2L;

        LotteryCalculationResult result = calculateWinningList(
                shenzhenIndex, sme100Index, winningQuota, totalCount);
        
        System.out.println("=== 中签计算结果 ===");
        System.out.println("深证成指指数: " + shenzhenIndex);
        System.out.println("中小100指数: " + sme100Index);
        System.out.println("基数A: " + result.getBaseNumber());
        System.out.println("翻转数B: " + result.getFlipNumber());
        System.out.println("预约次数X: " + result.getTotalCount());
        System.out.println("起始中签号Y: " + result.getStartWinningNumber());
        System.out.println("中签配额: " + result.getWinningQuota());
        System.out.println("阶数Z: " + result.getOrderLevel());
        System.out.println("中签号码: " + result.getWinningNumbers());
        
        // 验证前几个中签号码
        List<Long> winningNumbers = result.getWinningNumbers();
        System.out.println("\n=== 验证中签号码 ===");
        for (int i = 0; i < Math.min(5, winningNumbers.size()); i++) {
            System.out.println("中签号" + (i + 1) + " = " + winningNumbers.get(i));
        }
        if (winningNumbers.size() > 5) {
            System.out.println("...");
            for (int i = Math.max(winningNumbers.size() - 3, 5); i < winningNumbers.size(); i++) {
                System.out.println("中签号" + (i + 1) + " = " + winningNumbers.get(i));
            }
        }
    }
} 