package cn.com.duiba.activity.center.biz.service.elasticgifts.impl;

import cn.com.duiba.activity.center.api.dto.elasticgifts.*;
import cn.com.duiba.activity.center.api.dto.prize.ActivityPrizeOptionDto;
import cn.com.duiba.activity.center.api.enums.DeletedEnum;
import cn.com.duiba.activity.center.api.enums.ElasticGiftsBizCodeEnum;
import cn.com.duiba.activity.center.api.enums.ElasticGiftsStatusEnum;
import cn.com.duiba.activity.center.api.enums.ElasticGiftsTermTypeEnum;
import cn.com.duiba.activity.center.api.tool.Page;
import cn.com.duiba.activity.center.biz.dao.elasticgifts.ElasticGiftsAppDao;
import cn.com.duiba.activity.center.biz.dao.elasticgifts.ElasticGiftsDao;
import cn.com.duiba.activity.center.biz.dao.elasticgifts.ElasticGiftsTermDao;
import cn.com.duiba.activity.center.biz.entity.elasticgifts.ElasticGiftsAppEntity;
import cn.com.duiba.activity.center.biz.entity.elasticgifts.ElasticGiftsEntity;
import cn.com.duiba.activity.center.biz.entity.elasticgifts.ElasticGiftsTermEntity;
import cn.com.duiba.activity.center.biz.plugin.event.DuibaEventsDispatcher;
import cn.com.duiba.activity.center.biz.plugin.event.order.ActivityOrderSyncEvent;
import cn.com.duiba.activity.center.biz.plugin.event.order.ActivityOrdersEvent;
import cn.com.duiba.activity.center.biz.service.elasticgifts.ElasticGiftsService;
import cn.com.duiba.activity.center.biz.tools.service.FrequentExchangeLimitService;
import cn.com.duiba.activity.center.biz.utils.MemLock;
import cn.com.duiba.dcommons.domain.Tuple;
import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.dcommons.flowwork.ActivityLotteryFlowworkService;
import cn.com.duiba.order.center.api.dto.ActivityOrderDto;
import cn.com.duiba.order.center.api.remoteservice.RemoteActivityOrderService;
import cn.com.duiba.service.domain.dataobject.ConsumerDO;
import cn.com.duiba.service.domain.dataobject.ItemDO;
import cn.com.duiba.service.item.remoteservice.RemoteCouponFlowService;
import cn.com.duiba.service.item.remoteservice.RemoteItemKeyService;
import cn.com.duiba.service.remoteservice.RemoteConsumerService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

@Service("elasticGiftsService")
public class ElasticGiftsServiceImpl implements ElasticGiftsService {

    private static Logger LOGGER = LoggerFactory.getLogger(ElasticGiftsServiceImpl.class);

    @Autowired
    private ElasticGiftsDao elasticGiftsDao;

    @Autowired
    private RemoteItemKeyService remoteItemKeySerivce;

    @Autowired
    private ElasticGiftsAppDao elasticGiftsAppDao;

    @Autowired
    private ElasticGiftsTermDao elasticGiftsTermDao;

    @Autowired
    private RemoteConsumerService remoteConsumerService;

    @Autowired
    private RemoteCouponFlowService remoteCouponFlowService;

    @Autowired
    private RemoteActivityOrderService remoteActivityOrderService;

    @Autowired
    private ActivityLotteryFlowworkService activityLotteryFlowworkService;

    @Autowired
    private FrequentExchangeLimitService frequentExchangeLimitService;

    @Override
    public Page<ElasticGiftsListDto> findElasticGiftsPage(Integer currentPage, Integer pageSize, ElasticGiftsBizCodeEnum bizCode, Long elasticGiftsId, String title4admin) {
        Preconditions.checkNotNull(bizCode, "业务编号参数不全");
        if (currentPage == null) {
            currentPage = 1;
        }
        if (pageSize == null) {
            pageSize = 20;
        }
        int offset = (currentPage - 1) * pageSize;
        int max = pageSize;
        Page<ElasticGiftsListDto> page = new Page<>(pageSize, currentPage);
        List<ElasticGiftsEntity> entities = elasticGiftsDao.findPage(offset, max, bizCode.value(), elasticGiftsId, title4admin);
        Integer count = elasticGiftsDao.findPageCount(bizCode.value(), elasticGiftsId, title4admin);
        page.setList(convertPageListDto(entities));
        page.setTotalCount(count == null ? 0 : count);
        return page;
    }

    /**
     * 获取分页用的 page
     * @param entities
     * @return
     */
    private List<ElasticGiftsListDto> convertPageListDto(List<ElasticGiftsEntity> entities) {
        if (CollectionUtils.isEmpty(entities)) {
            return Collections.emptyList();
        }
        List<ElasticGiftsListDto> result = new ArrayList<>();
        for (ElasticGiftsEntity it : entities) {
            ElasticGiftsListDto dto = new ElasticGiftsListDto();
            dto.setId(it.getId());
            dto.setTitle4admin(it.getTitle4admin());
            dto.setBizCode(ElasticGiftsBizCodeEnum.fromValue(it.getBizCode()));
            dto.setStatus(ElasticGiftsStatusEnum.fromValue(it.getStatus()));
            dto.setAppCount(getAppCountByElId(it.getId()));
            dto.setTermCount(getTermCountByElId(it.getId()));
            result.add(dto);
        }
        return result;
    }

    /**
     * 获取弹层活动的开启 app 数量,不包含共享库
     * @param elasticGiftsId
     * @return
     */
    private int getAppCountByElId(Long elasticGiftsId) {
        List<Long> appIds = elasticGiftsAppDao.getAppIdsByElasticGiftsId(elasticGiftsId);
        return appIds.size();
    }

    /**
     * 获取弹层活动的礼包项数量
     * @param elasticGiftsId
     * @return
     */
    private int getTermCountByElId(Long elasticGiftsId) {
        List<ElasticGiftsTermEntity> terms = elasticGiftsTermDao.findAllByElId(elasticGiftsId);
        return terms.size();
    }

    @Override
    public Boolean updateStatus(Long elasticGiftsId, ElasticGiftsStatusEnum targetStatus) {
        Preconditions.checkNotNull(elasticGiftsId, "活动 id 不能为 null");
        Preconditions.checkNotNull(targetStatus, "目标状态不能为 null");
        return elasticGiftsDao.updateStatus(elasticGiftsId, targetStatus.value()) > 0;
    }

    @Override
    public List<Long> getAppIdsByElasticGiftsId(Long elasticGiftsId) {
        return elasticGiftsAppDao.getAppIdsByElasticGiftsId(elasticGiftsId);
    }

    @Override
    public Boolean delete(Long elasticGiftsId) {
        elasticGiftsTermDao.deleteByElId(elasticGiftsId);   // 删除对应礼包项
        return elasticGiftsDao.delete(elasticGiftsId) > 0;
    }

    @Override
    public Boolean save(Long elasticGiftsId, String title4admin, ElasticGiftsBizCodeEnum bizCode, List<Long> termIds) {
        Preconditions.checkNotNull(bizCode, "业务编号不能为 null");
        Long elId = saveEl(elasticGiftsId, title4admin, bizCode);
        if (isCreateOrCpoy(elasticGiftsId) && CollectionUtils.isEmpty(termIds)) { // 如果是创建,并且礼包项没有,那就返回成功
            return true;
        }
        List<ElasticGiftsTermEntity> terms = getSortedTerms(termIds);
        if (isCreateOrCpoy(elasticGiftsId)) {   // 表示是复制
            copyTerms(terms, elId);
            return true;
        }
        // 表示更新
        List<ElasticGiftsTermEntity> oldTerms = elasticGiftsTermDao.findAllByElId(elId);
        deleteTerms(oldTerms, terms);   // 删除
        return true;
    }

    /**
     * 判断是否是创建或者复制操作
     * @param elasticGiftsId
     * @return
     */
    private boolean isCreateOrCpoy(Long elasticGiftsId) {
        return elasticGiftsId == null;
    }

    /**
     * 删除礼包项
     * @param oldTerms
     * @param terms
     */
    private void deleteTerms(List<ElasticGiftsTermEntity> oldTerms, List<ElasticGiftsTermEntity> terms) {
        if (CollectionUtils.isEmpty(oldTerms)) {    // 原来就没有
            return;
        }
        oldTerms.removeAll(terms);  // 做差集
        for (ElasticGiftsTermEntity it : oldTerms) {
            elasticGiftsTermDao.delete(it.getId());
        }
    }

    /**
     * 获取排序好的礼包项
     * @param termIds
     * @return
     */
    private List<ElasticGiftsTermEntity> getSortedTerms(final List<Long> termIds) {
        if (CollectionUtils.isEmpty(termIds)) {
            return Collections.emptyList();
        }
        List<ElasticGiftsTermEntity> terms = elasticGiftsTermDao.findAllByIds(termIds);
        Collections.sort(terms, new Comparator<ElasticGiftsTermEntity>() {

            @Override
            public int compare(ElasticGiftsTermEntity o1, ElasticGiftsTermEntity o2) {
                if (termIds.indexOf(o1.getId()) > termIds.indexOf(o2.getId())) {
                    return 1;
                } else if (termIds.indexOf(o1.getId()) < termIds.indexOf(o2.getId())) {
                    return -1;
                } else {
                    return 0;
                }
            }

        });
        return terms;
    }

    /**
     * 根据已有的礼包项,复制一份到数据库
     * @param terms
     * @return
     */
    private void copyTerms(List<ElasticGiftsTermEntity> terms, Long elasticGiftsId) {
        if (CollectionUtils.isEmpty(terms)) {   // 没有,默认复制成功
            return;
        }
        int payload = 1;
        for (ElasticGiftsTermEntity it : terms) {
            ElasticGiftsTermEntity e4i = new ElasticGiftsTermEntity();
            BeanUtils.copyProperties(it, e4i, "id");
            e4i.setDeleted(DeletedEnum.UNDELETED.value());
            e4i.setElasticGiftsId(elasticGiftsId);
            e4i.setPayload(payload++);
            elasticGiftsTermDao.insert(e4i);
        }
    }

    /**
     * 保存活动
     * @param elasticGiftsId
     * @param title4admin
     * @param bizCode
     * @return
     */
    private Long saveEl(Long elasticGiftsId, String title4admin, ElasticGiftsBizCodeEnum bizCode) {
        // 编辑的时候 bizCode 不允许更新 有问题找郭燕飞
        if (isCreateOrCpoy(elasticGiftsId)) {   // 创建或者复制
            ElasticGiftsEntity e4i = new ElasticGiftsEntity();
            e4i.setTitle4admin(title4admin);
            e4i.setTitle(null);
            e4i.setStatus(ElasticGiftsStatusEnum.CLOSE_AND_UNVIEW.value());
            e4i.setBizCode(bizCode.value());
            e4i.setDeleted(DeletedEnum.UNDELETED.value());
            int ret = elasticGiftsDao.insert(e4i);
            if (ret <= 0) {
                throw new RuntimeException("保存活动异常");
            }
            return e4i.getId();
        } else {    // 更新
            ElasticGiftsEntity e4u = new ElasticGiftsEntity();
            e4u.setId(elasticGiftsId);
            e4u.setTitle4admin(title4admin);
            int ret = elasticGiftsDao.update(e4u);
            if (ret <= 0) {
                throw new RuntimeException("更新活动异常");
            }
            return e4u.getId();
        }

    }

    @Override
    public ElasticGiftsEditDto getElasticGiftsById(Long elasticGiftsId) {
        Preconditions.checkNotNull(elasticGiftsId, "活动 id 不能为 null");
        ElasticGiftsEntity entity = elasticGiftsDao.find(elasticGiftsId);
        List<ElasticGiftsTermEntity> termEntities = elasticGiftsTermDao.findAllByElId(entity.getId());
        List<ElasticGiftsTermSimpleDto> terms = new ArrayList<>();
        for (ElasticGiftsTermEntity it : termEntities) {
            ElasticGiftsTermSimpleDto term = new ElasticGiftsTermSimpleDto(it.getId(), it.getTermId(), it.getPayload());
            terms.add(term);
        }
        ElasticGiftsEditDto dto = new ElasticGiftsEditDto();
        dto.setId(entity.getId());
        dto.setBizCode(ElasticGiftsBizCodeEnum.fromValue(entity.getBizCode()));
        dto.setTitle4admin(entity.getTitle4admin());
        dto.setTerms(terms);
        return dto;
    }

    @Override
    public List<ElasticGiftsDto> getElasticGiftsByText(ElasticGiftsBizCodeEnum bizCode, String text) {
        Preconditions.checkNotNull(bizCode, "业务编号不能为 null");
        ElasticGiftsEntity entity4id = null;
        if (StringUtils.isNotBlank(text) && StringUtils.isNumeric(text)) {
            Long id = Long.valueOf(text);
            entity4id = elasticGiftsDao.findByIdAndBizCode(id, bizCode.value());
        }
        List<ElasticGiftsEntity> entities = elasticGiftsDao.findAllByTitle4adminAndBizCode(text, bizCode.value());
        if (!entities.contains(entity4id)) {
            entities.add(entity4id);
        }
        return cn.com.duiba.wolf.utils.BeanUtils.copyList(entities, ElasticGiftsDto.class);
    }

    @Override
    public Tuple.Tuple2<ElasticGiftsDto, List<ElasticGiftsTermDto>> getElasticGiftsForMobile(ElasticGiftsBizCodeEnum bizCode, Long appId) throws Exception {
        Preconditions.checkNotNull(appId, "appId不能为null");
        Preconditions.checkNotNull(bizCode, "bizCode不能为null");
        Long elasticGiftsId = getElasticGiftsId(bizCode.value(), appId);
        if (elasticGiftsId == null) {
            throw new Exception("礼包 id 不存在");
        }
        ElasticGiftsEntity gift = elasticGiftsDao.find(elasticGiftsId);
        if (gift == null || DeletedEnum.DELETED.value() == gift.getDeleted()) {
            throw new Exception("礼包不存在");
        }
        if (ElasticGiftsStatusEnum.OPEN.value() != gift.getStatus()) {
            throw new Exception("礼包未启用");
        }
        List<ElasticGiftsTermEntity> terms = elasticGiftsTermDao.findAllByElId(gift.getId());
        List<ElasticGiftsTermDto> dtos = new ArrayList<>(terms.size());
        for (ElasticGiftsTermEntity it : terms) {
            ElasticGiftsTermDto dto = new ElasticGiftsTermDto();
            BeanUtils.copyProperties(it, dto, "termType", "deleted");
            dto.setDeleted(DeletedEnum.fromValue(it.getDeleted()));
            dto.setTermType(ElasticGiftsTermTypeEnum.fromValue(it.getTermType()));
            dtos.add(dto);
        }
        ElasticGiftsDto eldto = cn.com.duiba.wolf.utils.BeanUtils.copy(gift, ElasticGiftsDto.class);
        return Tuple.tuple(eldto, dtos);
    }

    /**
     * 获取对应的礼包 id
     * @param bizCode
     * @param appId
     * @return
     * @throws Exception
     */
    private Long getElasticGiftsId(Integer bizCode, Long appId) throws Exception {
        ElasticGiftsAppEntity elApp = elasticGiftsAppDao.findByBizCodeAndAppId(bizCode, appId);
        if (elApp != null) {
            return elApp.getElasticGiftsId();
        }
        ElasticGiftsAppEntity commonApp = elasticGiftsAppDao.findByBizCodeAndAppId(bizCode, 0L);
        if (commonApp != null) {
            return commonApp.getElasticGiftsId();
        }
        throw new Exception("共享库不存在");
    }

    @Override
    public ElasticTakeOrderDto checkElasticTakeOrder(Long egTermId, Long appId) throws Exception {
        Preconditions.checkNotNull(egTermId, "礼包项参数不全");
        Preconditions.checkNotNull(appId, "appId 不能的空");
        ElasticGiftsTermEntity term = elasticGiftsTermDao.find(egTermId);
        if (term == null || DeletedEnum.DELETED.value() == term.getDeleted()) {
            throw new Exception("礼包项不存在");
        }
        ElasticGiftsEntity gift = elasticGiftsDao.find(term.getElasticGiftsId());
        if (gift == null || DeletedEnum.DELETED.value() == gift.getDeleted()) {
            throw new Exception("礼包不存在");
        }

        ElasticGiftsAppEntity giftsApp = elasticGiftsAppDao.findByBizCodeAndAppId(gift.getBizCode(), appId);
        if (giftsApp == null) {
            giftsApp = elasticGiftsAppDao.findByBizCodeAndAppId(gift.getBizCode(), ElasticGiftsAppServiceImpl.APP_COMMON_LIST.get(0));
        }
        if (!giftsApp.getElasticGiftsId().equals(gift.getId())) {
            throw new Exception("非法的礼包项");
        }
        ElasticGiftsDto giftsDto = new ElasticGiftsDto();
        BeanUtils.copyProperties(gift, giftsDto);
        return new ElasticTakeOrderDto(giftsDto, term.getId(), term.getTermId());
    }

    /**
     * corePoolSize = 5
     * maximumPoolSize = 10
     */
    private ExecutorService executorService = new ThreadPoolExecutor(5, 10, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>());

    @Override
    public String takeOrder(ElasticGiftsDto dto, final Long itemId, final Long appId, final Long consumerId, RequestParamsDto requestParamsDto) throws Exception {
        ConsumerDO consumer = remoteConsumerService.find(consumerId);
        final ActivityOrderDto orderDto = new ActivityOrderDto();
        orderDto.setConsumerId(consumer.getId());
        orderDto.setAppId(consumer.getAppId());
        orderDto.setPartnerUserId(consumer.getPartnerUserId());
        orderDto.setDuibaActivityId(dto.getId());
        orderDto.setActivityType(ActivityOrderDto.TypeElasticGifts);
        orderDto.setConsumeCredits(0L);
        orderDto.setExchangeStatus(ActivityOrderDto.ExchangeWait);
        orderDto.setConsumeCreditsStatus(ActivityOrderDto.ConsumeCreditsProcessing);   // 没有扣积分,后面直接置为成功
        orderDto.setIp(requestParamsDto.getIp());
        orderDto.setItemId(itemId);
        orderDto.setActivityOptionType(ItemDO.TypeCoupon);
        orderDto.setGid(itemId);
        orderDto.setGtype(GoodsTypeEnum.DUIBA.getGtype() + "");
        Date now = new Date();
        orderDto.setGmtCreate(now);
        orderDto.setGmtModified(now);
        final String orderNum;
        try {
            orderNum = remoteActivityOrderService.createOrder(orderDto).getResult();
            this.elasticGiftsFlowwork(orderNum);
        } catch (Exception e) {
            LOGGER.error("创建订单失败", e);
            throw e;
        }
        return orderNum;
    }

    /**
     * 弹层礼包流程
     * @param orderDto
     * @param orderNum
     */
    private void elasticGiftsFlowwork(final String orderNum) {
        Preconditions.checkNotNull(orderNum, "订单未创建");
        executorService.submit(new Runnable() {
            @Override
            public void run() {
                // 获取锁,重试5次
                AtomicInteger ai = new AtomicInteger(5);
                try {
                    process(ai);
                } catch (Exception e) {
                    LOGGER.error("弹层礼包流程异常, orderNum = " + orderNum, e);
                }
            }

            private void process(AtomicInteger tryCount) throws Exception {
                ActivityOrderDto order = remoteActivityOrderService.findByOrderNum(orderNum).getResult();
                // 建一个假的
                ActivityPrizeOptionDto option = new ActivityPrizeOptionDto();
                option.setItemId(order.getItemId());
                option.setActivityType(order.getActivityType());
                option.setPrizeType(ItemDO.TypeCoupon);
                ActivityOrderSyncEvent.ActivityOrderPluginContext context = new ActivityOrderSyncEvent.ActivityOrderPluginContext();

                MemLock lock = new MemLock();
                lock.setLocked(false);
                try {
                    lock = frequentExchangeLimitService.lockPrize(order.getDuibaActivityId(), order.getActivityType(), order.getItemId());
                    Long couponId = null;
                    if(lock.isLocked()) {
                        DuibaEventsDispatcher.get().triggerBeforeActivityOrderComplete(order, option, context);
                        Object obj = context.getAttribute("couponId");
                        if(null != obj){
                            couponId = (Long)obj;
                        }
                    } else {
                        if (tryCount.decrementAndGet() <= 0) {
                            throw new Exception("无法获取锁");
                        }
                        process(tryCount);
                    }
                    // 默认扣积分成功
                    DubboResult<Boolean> result = remoteActivityOrderService.consumeCreditsSuccess(orderNum, null, null, null, null, null, null, null, null, couponId);
                    if (!result.isSuccess() || result.getResult() == null || !result.getResult()) {
                        throw new Exception("更新扣积分和 couponId失败");
                    }
                } catch (Exception e) {
                    if (lock.isLocked()) {
                        DuibaEventsDispatcher.get().triggerOnActivityOrderCompleteException(order, option, e, context);
                    }
                    throw e;
                } finally {
                    if(lock.isLocked()){
                        frequentExchangeLimitService.unlockPrize(order.getDuibaActivityId(), order.getActivityType(), order.getItemId());
                    }
                }

                this.complete(order);
            }

            /**
             * 完成订单,自动领奖,产生主订单,此处失败已经不会影响用户领券
             */
            private void complete(ActivityOrderDto order) {
                // 增加兑换记录
                activityLotteryFlowworkService.insertCosumerExchanageRecord(order.getConsumerId(), order.getOrderNum(), order.getActivityType(), order.getActivityOptionType());
                // 自动领奖
                activityLotteryFlowworkService.autoTakePrize(orderNum, order.getConsumerId(), order.getAppId(), order.getIp(), null);

                // 没有扣积分,理论上只有扣积分成功操作
                if (order.getConsumeCreditsStatus() == ActivityOrderDto.ConsumeCreditsSuccess) {
                    DuibaEventsDispatcher.get().dispatchEvent(new ActivityOrdersEvent(ActivityOrdersEvent.ActivityOrdersEventType.OnOrderSuccess, order));
                } else if (order.getConsumeCreditsStatus() == ActivityOrderDto.ConsumeCreditsFail) {
                    DuibaEventsDispatcher.get().dispatchEvent(new ActivityOrdersEvent(ActivityOrdersEvent.ActivityOrdersEventType.OnOrderFail, order));
                }

            }
        });
    }
}
