/**
 * Project Name:activity-center-biz File Name:BaseSerice.java Package
 * Name:cn.com.duiba.activity.center.biz.service.gamecenter.impl Date:2016年11月9日下午5:42:00 Copyright (c) 2016,
 * duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.activity.center.biz.service.gamecenter.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.activity.center.api.dto.direct.ActivityBlackList4DeveloperDto;
import cn.com.duiba.activity.center.api.dto.direct.DuibaActivityAppSpecifyNewDto;
import cn.com.duiba.activity.center.api.dto.gamecenter.DefinedActivityTypesDto;
import cn.com.duiba.activity.center.api.dto.gamecenter.LittleGameResourceDto;
import cn.com.duiba.activity.center.biz.constant.RedisKeys;
import cn.com.duiba.activity.center.biz.dao.gamecenter.GameCenterResourceLocationDao;
import cn.com.duiba.activity.center.biz.entity.gamecenter.ActivityShieldEntity;
import cn.com.duiba.activity.center.biz.entity.gamecenter.GameCenterResourceLocationEntity;
import cn.com.duiba.activity.center.biz.entity.littlegame.LittleGameEntity;
import cn.com.duiba.activity.center.biz.service.direct.DeveloperActBlackService;
import cn.com.duiba.activity.center.biz.service.direct.DuibaActivityAppSpecifyNewService;
import cn.com.duiba.activity.center.biz.service.gamecenter.ActivityShieldService;
import cn.com.duiba.activity.center.biz.service.gamecenter.GameCenterException;
import cn.com.duiba.activity.center.biz.service.littlegame.LittleGameService;
import cn.com.duiba.service.domain.dataobject.AppDO;
import cn.com.duiba.service.remoteservice.RemoteAppService;
import redis.clients.jedis.JedisCommands;

/**
 * ClassName:BaseSerice <br/>
 * Function: TODO ADD FUNCTION. <br/>
 * Reason: TODO ADD REASON. <br/>
 * Date: 2016年11月9日 下午5:42:00 <br/>
 *
 * @author Zhuzhiyong
 * @version
 * @see
 * @since JDK 1.6
 */
@Service
class GameCenterCommonSerice {

    /** The Constant log. */
    private static final Logger               log = LoggerFactory.getLogger(GameCenterCommonSerice.class);

    /** The remote app service. */
    @Autowired
    private RemoteAppService                  remoteAppService;

    /** The developer act black service. */
    @Autowired
    private DeveloperActBlackService          developerActBlackService;

    /** The duiba activity app specify new service. */
    @Autowired
    private DuibaActivityAppSpecifyNewService duibaActivityAppSpecifyNewService;

    /** The game center shield service. */
    @Autowired
    private ActivityShieldService             gameCenterShieldService;

    /** The game center resource location dao. */
    @Autowired
    private GameCenterResourceLocationDao     gameCenterResourceLocationDao;

    /** The little game service. */
    @Autowired
    private LittleGameService                 littleGameService;

    /** The jedis commands. */
    @Autowired
    private JedisCommands                     jedisCommands;

    /**
     * 根据APP ID查询developer ID.<br>
     * 先查询缓存，查询不到再调用dubbo接口.<br>
     *
     * @param appId the app id
     * @return the long
     */
    public Long findDeveloperIdByAppId(Long appId) {
        String redisKey = RedisKeys.ACTIVITY_CENTER_GAMECENTER_APPDO_DID + String.valueOf(appId);
        String didStr = getCache(redisKey);
        if (StringUtils.isNotEmpty(didStr)) {
            return Long.parseLong(didStr);
        }
        AppDO appDO = remoteAppService.find(appId);
        if (appDO == null) {
            log.error("appId is invalid: no record found: {}", appId);
            return null;
        }
        Long did = appDO.getDeveloperId();
        putCache(redisKey, String.valueOf(did));
        return did;
    }

    /**
     * 开启了黑名单，并且在黑名单中.
     *
     * @param littleGameEntity the little game entity
     * @param developerId the developer id
     * @return true, if successful
     */
    public boolean filterByActivityBlacklist(LittleGameEntity littleGameEntity, Long developerId) {
        return littleGameEntity.isOpenSwitch(LittleGameEntity.SWITCHES_DEV_BLACKLIST)
               && developerActBlackService.isExist(littleGameEntity.getId(),
                                                   ActivityBlackList4DeveloperDto.TypeDuibaLittleGame, developerId).booleanValue();
    }

    /**
     * 开启了定向,但app不在定向范围内.
     *
     * @param littleGameEntity the little game entity
     * @param appId the app id
     * @return true, if successful
     */
    public boolean filterBySpecify(LittleGameEntity littleGameEntity, Long appId) {
        if (!littleGameEntity.isOpenSwitch(LittleGameEntity.SWITCHES_DIRECT)) {
            return false;
        }

        String key = RedisKeys.ACTIVITY_CENTER_GAMECENTER_APPSPECIFY_LITTLEGAME
                     + String.valueOf(littleGameEntity.getId()) + "." + String.valueOf(appId);
        String resultStr = getCache(key);
        if (StringUtils.isNotEmpty(resultStr)) {
            return Boolean.valueOf(resultStr);
        }
        // 不存在定向记录
        boolean result = duibaActivityAppSpecifyNewService.findAppSpecifyByActivityIdAndAppIdAndActivityType(littleGameEntity.getId(),
                                                                                                             appId,
                                                                                                             DuibaActivityAppSpecifyNewDto.ACTIVITY_LITTLE_GAME) == null;
        putCache(key, String.valueOf(result));
        return result;
    }

    /**
     * 被开发者屏蔽.
     *
     * @param appId the app id
     * @param littleGameId the little game id
     * @return true, if successful
     */
    public boolean filterByShield(Long appId, Long littleGameId) {
        return gameCenterShieldService.isExist(appId, DefinedActivityTypesDto.LITTLE_GAME, littleGameId);
    }

    /**
     * 根据locationName查询id.
     *
     * @param name the name
     * @return the long
     * @throws GameCenterException the game center exception
     */
    public Long findLocationIdByName(String name) throws GameCenterException {

        String key = RedisKeys.ACTIVITY_CENTER_GAMECENTER_LOCNAME + name;
        String resultStr = getCache(key);
        if (StringUtils.isNotEmpty(resultStr)) {
            return Long.valueOf(resultStr);
        }

        GameCenterResourceLocationEntity entity = gameCenterResourceLocationDao.findResourceLocationByName(name);
        if (entity == null) {
            throw new GameCenterException("No this location found:" + name);
        }
        Long result = entity.getId();
        putCache(key, String.valueOf(result));
        return result;
    }

    /**
     * Builds the little game entity map.
     *
     * @param littleGameIds the little game ids
     * @return the map
     */
    public Map<Long, LittleGameEntity> buildLittleGameEntityMap(Set<Long> littleGameIds) {
        List<LittleGameEntity> littleGameEntities = littleGameService.selectByIds(littleGameIds);
        if (CollectionUtils.isEmpty(littleGameEntities)) {
            return Collections.emptyMap();
        }
        Map<Long, LittleGameEntity> map = new HashMap<Long, LittleGameEntity>();
        for (LittleGameEntity entity : littleGameEntities) {
            map.put(entity.getId(), entity);
        }
        return map;
    }

    /**
     * Builds the activity shield entity map.
     *
     * @param appId the app id
     * @param littleGameIds the little game ids
     * @param useCache 是否使用缓存
     * @return the map
     */
    public Map<Long, ActivityShieldEntity> buildActivityShieldEntityMap(Long appId, Set<Long> littleGameIds,
                                                                        boolean useCache) {

        List<ActivityShieldEntity> activityShieldEntityList = gameCenterShieldService.findShields(appId,
                                                                                                  DefinedActivityTypesDto.LITTLE_GAME,
                                                                                                  littleGameIds,
                                                                                                  useCache);
        if (CollectionUtils.isEmpty(activityShieldEntityList)) {
            return Collections.emptyMap();
        }
        Map<Long, ActivityShieldEntity> map = new HashMap<Long, ActivityShieldEntity>();
        for (ActivityShieldEntity entity : activityShieldEntityList) {
            map.put(entity.getActivityId(), entity);
        }
        return map;
    }

    public List<LittleGameResourceDto> batchFindAvailableLittleGames(List<LittleGameResourceDto> littleGameResourceDtoList,
                                                                     Set<Long> littleGameIds, Long appId,
                                                                     Long developerId, boolean needCheckShield,
                                                                     boolean shieldUseCache) {
        // 批量查询
        Map<Long, LittleGameEntity> littleGameEntityMap = buildLittleGameEntityMap(littleGameIds);
        if (littleGameEntityMap.isEmpty()) {
            return Collections.emptyList();
        }
        Map<Long, ActivityShieldEntity> activityShieldEntityMap = null;
        if (needCheckShield) {
            activityShieldEntityMap = buildActivityShieldEntityMap(appId, littleGameIds, shieldUseCache);
        }
        List<LittleGameResourceDto> finalLittleGameResourceDtoList = new ArrayList<LittleGameResourceDto>();
        LittleGameEntity littleGameEntity;
        for (LittleGameResourceDto littleGameResourceDto : littleGameResourceDtoList) {
            littleGameEntity = littleGameEntityMap.get(littleGameResourceDto.getActivityId());
            if (Utils.filterLittleGameByStatus(littleGameEntity)) {
                continue;
            }
            // 开启了黑名单，并且在黑名单中
            if (filterByActivityBlacklist(littleGameEntity, developerId)) {
                continue;
            }
            // 开启了定向,但app不在定向范围内
            if (filterBySpecify(littleGameEntity, appId)) {
                continue;
            }
            if (MapUtils.isNotEmpty(activityShieldEntityMap)) {
                if (null != activityShieldEntityMap.get(littleGameResourceDto.getActivityId())) {
                    continue;
                }
            }
            littleGameResourceDto.setTitle(littleGameEntity.getLittleGameTitle());
            littleGameResourceDto.setBannerImg(littleGameEntity.getLittleGameBannerImg());
            littleGameResourceDto.setIconImg(littleGameEntity.getLittleGameIconImg());
            littleGameResourceDto.setSmallImg(littleGameEntity.getLittleGameSmallImg());
            littleGameResourceDto.setRecomImg(littleGameEntity.getLittleGameRecomImg());
            finalLittleGameResourceDtoList.add(littleGameResourceDto);
        }
        return finalLittleGameResourceDtoList;
    }

    /**
     * Gets the cache.
     *
     * @param key the key
     * @return the cache
     */
    private String getCache(String key) {
        return jedisCommands.get(key);
    }

    /**
     * Put cache.
     *
     * @param key the key
     * @param value the value
     */
    private void putCache(String key, String value) {
        jedisCommands.set(key, value, "NX", "EX", 60L);
    }
}
