/**
 * Project Name:activity-center-biz File Name:TodayRobSeckillBoImpl.java Package
 * Name:cn.com.duiba.activity.center.biz.bo.rob Date:2016年9月30日上午9:43:28 Copyright (c) 2016, duiba.com.cn All Rights
 * Reserved.
 */

package cn.com.duiba.activity.center.biz.bo.rob.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.com.duiba.activity.center.api.dto.PaginationDto;
import cn.com.duiba.activity.center.api.dto.activity.OperatingActivityDto;
import cn.com.duiba.activity.center.api.dto.game.DuibaQuestionAnswerDto;
import cn.com.duiba.activity.center.api.dto.rob.TodayRobSeckillAppDto;
import cn.com.duiba.activity.center.api.dto.rob.TodayRobSeckillConfigDto;
import cn.com.duiba.activity.center.api.dto.rob.TodayRobSeckillConfigListDto;
import cn.com.duiba.activity.center.api.dto.rob.TodayRobSeckillDto;
import cn.com.duiba.activity.center.api.dto.rob.TodayRobSeckillListDto;
import cn.com.duiba.activity.center.api.enums.ActivityStatusEnum;
import cn.com.duiba.activity.center.api.enums.PushStatusEnum;
import cn.com.duiba.activity.center.biz.bo.rob.TodayRobSeckillBo;
import cn.com.duiba.activity.center.biz.dao.DsConstants;
import cn.com.duiba.activity.center.biz.domain.PushAppsConfig;
import cn.com.duiba.activity.center.biz.entity.config.CenterConfigEntity;
import cn.com.duiba.activity.center.biz.entity.rob.TodayRobSeckillConfigEntity;
import cn.com.duiba.activity.center.biz.entity.rob.TodayRobSeckillEntity;
import cn.com.duiba.activity.center.biz.entity.seckill.DuibaSeckillEntity;
import cn.com.duiba.activity.center.biz.service.activity.OperatingActivityService;
import cn.com.duiba.activity.center.biz.service.config.CenterConfigService;
import cn.com.duiba.activity.center.biz.service.direct.DeveloperActBlackService;
import cn.com.duiba.activity.center.biz.service.rob.TodayRobSeckillService;
import cn.com.duiba.activity.center.biz.service.seckill.DuibaSeckillAppSpecifyService;
import cn.com.duiba.activity.center.biz.service.seckill.DuibaSeckillService;
import cn.com.duiba.activity.center.biz.service.seckill.SeckillService;
import cn.com.duiba.activity.center.common.util.AppLogUtil;
import cn.com.duiba.activity.center.common.util.DateUtil;
import cn.com.duiba.service.domain.dataobject.AppDO;
import cn.com.duiba.service.domain.dataobject.DuibaSeckillDO;
import cn.com.duiba.service.domain.vo.ItemKey;
import cn.com.duiba.service.item.remoteservice.RemoteItemKeyService;
import cn.com.duiba.service.remoteservice.RemoteAppService;
import cn.com.duiba.service.remoteservice.RemoteDuibaSeckillService;
import cn.com.duiba.service.remoteservice.RemotePreStockService;
import cn.com.duiba.wolf.redis.RedisClient;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.ObjectUtil;

import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * ClassName:TodayRobSeckillBoImpl <br/>
 * Function: TODO ADD FUNCTION. <br/>
 * Reason: TODO ADD REASON. <br/>
 * Date: 2016年9月30日 上午9:43:28 <br/>
 * 
 * @author zhanglihui
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
public class TodayRobSeckillBoImpl implements TodayRobSeckillBo {

    private static Logger                 log                         = LoggerFactory.getLogger(TodayRobSeckillBoImpl.class);

    // 秒杀活动推送的分布式锁前缀名称
    private static final String           REDIS_KEY_PREFIX            = "TODAY_ROB_SECKILL";

    // 子活动不会在开发者端展示，推送给开发者的秒杀活动设置一个默认的父活动ID
    private static final Long             TODAY_ROB_SECKILL_PARENT_ID = 0L;

    @Autowired
    private TodayRobSeckillService        todayRobSeckillService;

    @Autowired
    private CenterConfigService           centerConfigService;

    @Autowired
    private DuibaSeckillService           duibaSeckillService;

    @Autowired
    private RedisClient                   redisClient;

    @Autowired
    private OperatingActivityService      operatingActivityService;

    @Autowired
    private RemoteItemKeyService          remoteItemKeyService;

    @Autowired
    private RemotePreStockService         remotePreStockService;

    @Autowired
    private RemoteDuibaSeckillService     remoteDuibaSeckillService;

    @Autowired
    private RemoteAppService              remoteAppService;

    @Autowired
    private DeveloperActBlackService      developerActBlackService;

    @Autowired
    private DuibaSeckillAppSpecifyService duibaSeckillAppSpecifyService;

    @Autowired
    private SeckillService                seckillService;

    @Autowired
    @Qualifier("seckillPushThreadPool")
    private ThreadPoolTaskExecutor        seckillPushThreadPool;

    @Override
    public void pushSeckillActivity(List<TodayRobSeckillConfigDto> activityList, final List<Long> appIdList) {
        if (CollectionUtils.isEmpty(activityList) || CollectionUtils.isEmpty(appIdList)) {
            AppLogUtil.warn(log, "今日必抢-秒杀免单活动推送失败，参数错误，activityList={}， appIdList={}", activityList, appIdList);
            return;
        }
        CenterConfigEntity config = centerConfigService.findPushAppsConfig();
        for (TodayRobSeckillConfigDto dto : activityList) {
            TodayRobSeckillConfigEntity entity = todayRobSeckillService.findPushedActivity(dto.getActivityId());
            if (entity != null && entity.getPushTime() != null
                && StringUtils.equals(entity.getPushStatus(), PushStatusEnum.SUCC.getCode())) {
                // 该活动推送过，判断app推送列表版本是否更新过
                if (Objects.equal(entity.getPushAppsVersion(), config.getVersion())) {
                    AppLogUtil.warn(log, "活动不需要推送，activityId={}", dto.getActivityId());
                    continue;
                }
            }
            // 判断活动状态是否开启且可见
            final DuibaSeckillEntity duibaSeckill = duibaSeckillService.find(dto.getActivityId());
            if (duibaSeckill == null) {
                AppLogUtil.error(log, "兑吧秒杀活动不存在，activityId={}", dto.getActivityId());
                continue;
            }
            if (!Objects.equal(duibaSeckill.getStatus(), ActivityStatusEnum.STATUS_OPEN.getIntCode())) {
                AppLogUtil.warn(log, "活动未开启，不需要推送，activityId={}，status={}", dto.getActivityId(),
                                duibaSeckill.getStatus());
                continue;
            }

            // 开启异步线程推送活动至开发者库
            final Long todayRobSeckillConfigId = entity == null ? dto.getId() : entity.getId();
            seckillPushThreadPool.execute(new Runnable() {

                @Override
                public void run() {
                    pushToApps(todayRobSeckillConfigId, duibaSeckill, appIdList);
                }});
        }

    }

    private void pushToApps(Long todayRobSeckillConfigId, DuibaSeckillEntity duibaSeckill, List<Long> appIdList) {
        Long reply = redisClient.setnx(REDIS_KEY_PREFIX + duibaSeckill.getId(), duibaSeckill.getId().toString());
        if (reply == 0) {
            AppLogUtil.warn(log, "获取分布式锁失败，activityId={}", duibaSeckill.getId());
            return;
        }
        // 设置失效时间，防止系统崩溃导致不能释放锁
        redisClient.expire(REDIS_KEY_PREFIX + duibaSeckill.getId(), 120);
        // 更新活动推送时间和推送状态
        todayRobSeckillService.updatePushStatus(todayRobSeckillConfigId, PushStatusEnum.EXEC);
        PushStatusEnum pushStatus = PushStatusEnum.SUCC;
        try {
            for (Long appId : appIdList) {
                doPush(duibaSeckill, appId);
            }
        } catch (Exception e) {
            AppLogUtil.error(log, "秒杀活动推送失败, activityId={}", duibaSeckill.getId(), e);
            pushStatus = PushStatusEnum.FAILED;
        } finally {
            redisClient.del(REDIS_KEY_PREFIX + duibaSeckill.getId());
        }
        todayRobSeckillService.updatePushStatus(todayRobSeckillConfigId, pushStatus);
    }

    private void doPush(DuibaSeckillEntity duibaSeckill, Long appId) {
        OperatingActivityDto operatingActivity = operatingActivityService.findByAppIdAndDuibaSeckillIdAndDeleted(appId,
                                                                                                                 duibaSeckill.getId(),
                                                                                                                 null);
        if (operatingActivity != null) {
            OperatingActivityDto o4u = new OperatingActivityDto(operatingActivity.getId());
            o4u.setDeleted(Boolean.FALSE);
            setInsertAndUpdateChildCommonProperties(duibaSeckill, appId, o4u);
            o4u.setStatus(OperatingActivityDto.StatusIntOpen);
            operatingActivityService.update(o4u);
        } else {
            OperatingActivityDto o4i = new OperatingActivityDto(Boolean.TRUE);
            o4i.setAppId(appId);
            setInsertAndUpdateChildCommonProperties(duibaSeckill, appId, o4i);
            o4i.setStatus(OperatingActivityDto.StatusIntOpen);
            o4i.setParentActivityId(TODAY_ROB_SECKILL_PARENT_ID);
            operatingActivityService.insert(o4i);
        }

    }

    private void setInsertAndUpdateChildCommonProperties(DuibaSeckillEntity duibaSeckill, Long appId,
                                                         OperatingActivityDto entity) {
        entity.setType(OperatingActivityDto.TypeDuibaSeckill);
        entity.setTitle(duibaSeckill.getTitle());
        ItemKey itemKey = remoteItemKeyService.getItemKey(duibaSeckill.getItemId(), null, appId);
        Long credits = remotePreStockService.calculateCreditsByItemKeyAndDegree(itemKey, null);
        entity.setCredits(credits);
        entity.setRule(duibaSeckill.getRule());
        entity.setActivityId(duibaSeckill.getId());
        entity.setImage(duibaSeckill.getMultiImage());
        entity.setSmallImage(duibaSeckill.getSmallImage());
    }

    @Override
    public PaginationDto<TodayRobSeckillListDto> findTodayRobSeckillList(int pageNum, int pageSize) {
        // 查询秒杀区间
        List<TodayRobSeckillListDto> trd = new ArrayList<TodayRobSeckillListDto>();
        List<TodayRobSeckillEntity> ret = todayRobSeckillService.findTodayRobSeckillList(pageNum, pageSize);
        ObjectUtil.convertList(ret, trd, TodayRobSeckillListDto.class);
        List<Long> todayRobSeckillIds = new ArrayList<Long>();
        for (TodayRobSeckillListDto trs : trd) {
            todayRobSeckillIds.add(trs.getId());
        }
        if (todayRobSeckillIds == null || todayRobSeckillIds.isEmpty()) {
            return new PaginationDto<TodayRobSeckillListDto>();
        }
        // 查询秒杀区间配置
        Map<Long, List<TodayRobSeckillConfigListDto>> configMap = Maps.newHashMap();
        List<TodayRobSeckillConfigListDto> trsd = new ArrayList<TodayRobSeckillConfigListDto>();
        List<Long> todayRobSeckillConfigIds = new ArrayList<Long>();// 秒杀区间配置id
        List<Long> seckillIds = new ArrayList<Long>();// 秒杀id
        List<TodayRobSeckillConfigEntity> seckillConfigs = todayRobSeckillService.findSeckillConfigByIdss(todayRobSeckillIds);
        ObjectUtil.convertList(seckillConfigs, trsd, TodayRobSeckillConfigListDto.class);
        for (TodayRobSeckillConfigListDto trs : trsd) {
            todayRobSeckillConfigIds.add(trs.getId());
            seckillIds.add(trs.getActivityId());
        }
        // 查询秒杀
        List<DuibaSeckillDO> seckills = remoteDuibaSeckillService.findAllByIds(seckillIds);
        Map<Long, DuibaSeckillDO> seckillMap = Maps.newHashMap();
        for (DuibaSeckillDO d : seckills) {
            seckillMap.put(d.getId(), d);
        }
        // set秒杀字段
        for (TodayRobSeckillConfigListDto trld : trsd) {
            DuibaSeckillDO dsd = seckillMap.get(trld.getActivityId());
            if (dsd != null) {
                trld.setStartDate(DateUtil.getDayStr(dsd.getDateStart()));
                trld.setEndDate(DateUtil.getDayStr(dsd.getDateEnd()));
                trld.setStartTime(DateUtil.getMinuteOnlyStr(dsd.getTimeStart()));
                trld.setEndTime(DateUtil.getMinuteOnlyStr(dsd.getTimeEnd()));
                trld.setTitle(dsd.getTitle());
                // 打上失效标识：已失效，如果秒杀活动处于“关闭且可见”或者“关闭且不可见”状态，或者已删除，则在页面上标记“已失效”；
                if (dsd.getDeleted() || DuibaQuestionAnswerDto.STATUS_OPEN != dsd.getStatus()) {
                    trld.setInvalidFlag(true);
                }
            }

            List<TodayRobSeckillConfigListDto> l = (List<TodayRobSeckillConfigListDto>) configMap.get(trld.getTodayRobSeckillId());
            if (l == null) {
                l = new ArrayList<TodayRobSeckillConfigListDto>();
                configMap.put(trld.getTodayRobSeckillId(), l);
            }
            l.add(trld);
        }
        // set秒杀区间配置
        for (TodayRobSeckillListDto trs : trd) {
        	if(configMap.get(trs.getId()) != null && !configMap.get(trs.getId()).isEmpty()){
        		Collections.sort(configMap.get(trs.getId()), new Comparator<TodayRobSeckillConfigListDto>() {
        			
        			@Override
        			public int compare(TodayRobSeckillConfigListDto o1, TodayRobSeckillConfigListDto o2) {
        				return o2.getPayload().compareTo(o1.getPayload());
        			}
        		});
        		trs.setTodayRobSeckillConfigs(configMap.get(trs.getId()));
        		
        	}

        }
        PaginationDto<TodayRobSeckillListDto> page = new PaginationDto<TodayRobSeckillListDto>();
        page.setRows(trd);
        page.setTotalCount(todayRobSeckillService.findTodayRobSeckillCount());
        return page;

    }
    
    

    @Transactional(value = DsConstants.DATABASE_CREDITS_ACTIVITY)
    @Override
    public Boolean seveSeckillConfigs(List<TodayRobSeckillConfigDto> list, TodayRobSeckillDto todayRobSeckillDto) {
        List<Long> delIdList = new LinkedList<Long>();
        List<TodayRobSeckillConfigEntity> updateList = new ArrayList<TodayRobSeckillConfigEntity>();
        List<TodayRobSeckillConfigEntity> insertList = new ArrayList<TodayRobSeckillConfigEntity>();
        boolean flag = false;//秒杀区间是否是修改标识，如果是调用推送方法
        TodayRobSeckillEntity trce = new TodayRobSeckillEntity();
        ObjectUtil.convert(todayRobSeckillDto,trce );
        if (trce.getId() == null) {
            todayRobSeckillService.saveSeckill(trce);
        } else {
        	flag = true;
            todayRobSeckillService.updateSeckill(trce);
        }

        for (TodayRobSeckillConfigDto trc : list) {
            // 拼接删除id
            if (trc.getDeleted()) {
                delIdList.add(trc.getId());
                continue;
            }
            trc.setTodayRobSeckillId(trce.getId());
            TodayRobSeckillConfigEntity pTrc = new TodayRobSeckillConfigEntity();
            ObjectUtil.convert(trc, pTrc);
            // 拼接插入
            if (trc.getId() == null) {
                insertList.add(pTrc);
                continue;
                // 拼接更新
            } else {
                updateList.add(pTrc);
                continue;

            }
        }
        if (delIdList != null && !delIdList.isEmpty()) {
        	todayRobSeckillService.delSeckillConfigBatch(delIdList);
        }
        if (insertList != null && !insertList.isEmpty()) {
            todayRobSeckillService.saveSeckillConfigBatch(insertList);
        }else{
        	insertList = new ArrayList<TodayRobSeckillConfigEntity>();
        }
        if (updateList != null && !updateList.isEmpty()) {
            todayRobSeckillService.updateSeckillConfigBatch(updateList);
        }else{
        	updateList = new ArrayList<TodayRobSeckillConfigEntity>();
        }
        
        //推送
        if(flag){
        	List<TodayRobSeckillConfigDto> pushInsertDtoList = new ArrayList<TodayRobSeckillConfigDto>();
        	List<TodayRobSeckillConfigDto> pushUpdateDtoList = new ArrayList<TodayRobSeckillConfigDto>();
        	ObjectUtil.convertList(insertList, pushInsertDtoList, TodayRobSeckillConfigDto.class);
        	ObjectUtil.convertList(updateList, pushUpdateDtoList, TodayRobSeckillConfigDto.class);
        	CenterConfigEntity entity = centerConfigService.findPushAppsConfig();
            PushAppsConfig pushAppsConfig = new PushAppsConfig(entity);
            pushInsertDtoList.addAll(pushUpdateDtoList);
        	pushSeckillActivity(pushInsertDtoList, pushAppsConfig.getAppIdList());
        }
        return true;
    }

    @Override
    public List<TodayRobSeckillAppDto> findSeckillListByApp(Long appId) {
        CenterConfigEntity entity = centerConfigService.findPushAppsConfig();
        PushAppsConfig pushAppsConfig = new PushAppsConfig(entity);
        if (!pushAppsConfig.getAppIdList().contains(appId)) {
            AppLogUtil.info(log, "app没有开启秒杀区间，appId={}", appId);
            return Lists.newArrayList();
        }
        List<TodayRobSeckillAppDto> list = Lists.newArrayList();
        // 查询上一个秒杀区间
        TodayRobSeckillEntity beforeEntity = todayRobSeckillService.findSeckillBeforeNow();
        TodayRobSeckillAppDto beforeAppDto = null;
        AppDO appDO = remoteAppService.find(appId);
        // 查询被加入黑名单的秒杀活动
        List<Long> blackActivityIds = developerActBlackService.queryActivityIdsByDeveloperId(appDO.getDeveloperId(),
                                                                                             OperatingActivityDto.TypeDuibaSeckill);
        // 查询被定向的活动列表
        List<Long> specifyActivityIds = duibaSeckillAppSpecifyService.findActivityIdsByAppId(appId);
        if (beforeEntity != null) {
            beforeAppDto = convertDto(appId, beforeEntity, blackActivityIds, specifyActivityIds);
            if (beforeAppDto.isActivityEnd()) {// 是否区间内所有活动已结束
                if (new Date().before(DateUtil.minutesAddOrSub(beforeAppDto.getStartTime(), 15))) {
                    // 结束的活动保留15分钟
                    list.add(beforeAppDto);
                }
            } else {
                // 未结束的活动继续展示
                list.add(beforeAppDto);
            }
        }
        List<TodayRobSeckillEntity> entityList = Lists.newArrayList();
        if (list.size() > 0) {
            entityList = todayRobSeckillService.findSeckillAfterNow(1);
        } else {
            entityList = todayRobSeckillService.findSeckillAfterNow(2);
        }
        if (CollectionUtils.isNotEmpty(entityList)) {
            for (TodayRobSeckillEntity afterEntity : entityList) {
                TodayRobSeckillAppDto afterAppDto = convertDto(appId, afterEntity, blackActivityIds, specifyActivityIds);
                if (CollectionUtils.isNotEmpty(afterAppDto.getOperatingActivityList())) {
                    list.add(afterAppDto);
                }
            }
        }
        if (CollectionUtils.isEmpty(list) && beforeAppDto != null) {
            // 如果后面没有配置秒杀活动，持续展示今天最后一个秒杀区间
            list.add(beforeAppDto);
        }
        return list;
    }

    private TodayRobSeckillAppDto convertDto(Long appId, TodayRobSeckillEntity todayRobSeckillEntity,
                                             List<Long> blackActivityIds, List<Long> specifyActivityIds) {
        TodayRobSeckillAppDto todayRobSeckillAppDto = BeanUtils.copy(todayRobSeckillEntity, TodayRobSeckillAppDto.class);
        List<TodayRobSeckillConfigEntity> seckillList = todayRobSeckillService.findSeckillConfigs(todayRobSeckillAppDto.getId());
        List<Long> seckillIds = Lists.newArrayList();
        for (TodayRobSeckillConfigEntity seckill : seckillList) {
            seckillIds.add(seckill.getActivityId());
        }
        // 查询秒杀活动：开启且可见，未删除
        List<DuibaSeckillEntity> duibaSeckillList = duibaSeckillService.findAllByIds(seckillIds);
        // 根据管理员配置的顺序排列
        duibaSeckillList = orderBySeckillIds(duibaSeckillList, seckillIds);
        List<Long> activityIds = Lists.newArrayList();
        int num = 0;
        for (DuibaSeckillEntity duibaSeckill : duibaSeckillList) {
            // 1. 判断开发者是否在活动黑名单中
            if (blackActivityIds.contains(duibaSeckill.getId())) {
                AppLogUtil.info(log, "开发者在活动黑名单中，appId={}, activityId={}", appId, duibaSeckill.getId());
                continue;
            }
            // 2. 判断app是否在定向列表中
            if (duibaSeckill.isOpenSwitch(DuibaSeckillEntity.SWITCHES_DIRECT)
                && !specifyActivityIds.contains(duibaSeckill.getId())) {
                AppLogUtil.info(log, "app没有在活动定向列表中，appId={}, activityId={}", appId, duibaSeckill.getId());
                continue;
            }
            /* 以下情况活动需要展示 */
            activityIds.add(duibaSeckill.getId());
            // 3. 判断活动是否已结束
            Date now = new Date();
            if (now.after(duibaSeckill.getDateEnd()) || DateUtil.compareTime(now, duibaSeckill.getTimeEnd()) > 0) {
                AppLogUtil.info(log, "活动已结束，appId={}, activityId={}, dateEnd={}, timeEnd={}", appId,
                                duibaSeckill.getId(), duibaSeckill.getDateEnd(), duibaSeckill.getTimeEnd());
                continue;
            }
            // 4. 判断秒杀库存是否耗尽
            Long stock = seckillService.getStock(appId, duibaSeckill);
            if (stock < 1) {
                AppLogUtil.info(log, "秒杀库存不足，appId={}, activityId={}", appId, duibaSeckill.getId());
                continue;
            }
            num++;
        }
        todayRobSeckillAppDto.setActivityEnd(num == 0);
        List<OperatingActivityDto> operatingActivityList = operatingActivityService.findDuibaSeckillByAppIdAndActivityId(appId,
                                                                                                                         activityIds,
                                                                                                                         OperatingActivityDto.TypeDuibaSeckill);
        todayRobSeckillAppDto.setOperatingActivityList(operatingActivityList);
        return todayRobSeckillAppDto;
    }

	private List<DuibaSeckillEntity> orderBySeckillIds(List<DuibaSeckillEntity> duibaSeckillList, List<Long> seckillIds) {
        Map<Long, DuibaSeckillEntity> map = Maps.uniqueIndex(duibaSeckillList, new Function<DuibaSeckillEntity, Long>() {

            @Override
            public Long apply(DuibaSeckillEntity input) {
                return input.getId();
            }
        });
        List<DuibaSeckillEntity> list = Lists.newArrayList();
        for(Long seckillId : seckillIds){
            list.add(map.get(seckillId));
        }
        return list;
    }

    @Override
	public Boolean enableTodayRobSeckill(long todayRobSeckillId, boolean enable) {
		//开启状态，推送活动
		if(enable){
			List<TodayRobSeckillConfigDto> pushList = new ArrayList<TodayRobSeckillConfigDto>();
			List<TodayRobSeckillConfigEntity> seckillList = todayRobSeckillService.findSeckillConfigs(todayRobSeckillId);
        	ObjectUtil.convertList(seckillList, pushList, TodayRobSeckillConfigDto.class);
        	CenterConfigEntity entity = centerConfigService.findPushAppsConfig();
            PushAppsConfig pushAppsConfig = new PushAppsConfig(entity);
        	pushSeckillActivity(pushList, pushAppsConfig.getAppIdList());
		}
		todayRobSeckillService.enableTodayRobSeckill(todayRobSeckillId, enable);
		return true;
	}
}
