package cn.com.duiba.activity.center.biz.service.seckill.impl;

import cn.com.duiba.activity.center.api.dto.seckill.SeckillStockDto;
import cn.com.duiba.activity.center.biz.dao.seckill.DuibaSeckillDao;
import cn.com.duiba.activity.center.biz.entity.seckill.DuibaSeckillEntity;
import cn.com.duiba.activity.center.biz.entity.seckill.DuibaSeckillStockConfigEntity;
import cn.com.duiba.activity.center.biz.service.seckill.SeckillDateUtilService;
import cn.com.duiba.activity.center.biz.service.seckill.SeckillMemcacheService;
import cn.com.duiba.activity.center.biz.service.seckill.SeckillService;
import cn.com.duiba.activity.center.biz.service.seckill.SeckillStockService;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

@Service("seckillService")
public class SeckillServiceImpl implements SeckillService {

    private static Logger LOGGER = LoggerFactory.getLogger(SeckillServiceImpl.class);

    @Autowired
    private SeckillMemcacheService seckillMemcacheService;

    @Autowired
    private DuibaSeckillDao duibaSeckillDao;

    @Autowired
    private SeckillStockService seckillStockService;

    @Autowired
    private SeckillDateUtilService seckillDateUtilService;

    private static Cache<String, SeckillStockCache> stockCache = CacheBuilder.newBuilder().expireAfterWrite(1, TimeUnit.DAYS).build();
    private static Cache<Long, DuibaSeckillEntity> seckillCache = CacheBuilder.newBuilder().expireAfterWrite(1, TimeUnit.MINUTES).build();

    private static String getStockCacheKey(Long appId, Long duibaSeckillId) {
        if (appId == null || duibaSeckillId == null) {
            return null;
        }
        return duibaSeckillId + "-" + appId;
    }

    @Override
    public Long getStock(Long appId, DuibaSeckillEntity duibaSeckill) {
        Long stockId = null;
        try {
            stockId = getStockId(appId, duibaSeckill);
        } catch (Exception e) {
            LOGGER.error("getStockId error，库存作为0处理", e);
            return 0L;
        }
        return getStockByMemchache(stockId, duibaSeckill, appId);
    }

    @Override
    public List<SeckillStockDto> batchGetStock(Long appId, List<Long> duibaSeckillIds) throws Exception {
        if (CollectionUtils.isEmpty(duibaSeckillIds)) {
            throw new Exception("参数不全");
        }
        if (appId == null) {
            throw new Exception("参数不全");
        }
        List<SeckillStockDto> result = new ArrayList<>();
        for (final Long it : duibaSeckillIds) {
            if (it == null) {
                continue;
            }
            DuibaSeckillEntity entity = seckillCache.get(it, new Callable<DuibaSeckillEntity>() {
                @Override
                public DuibaSeckillEntity call() {
                    return duibaSeckillDao.find(it);
                }
            });
            if (entity == null) {
                continue;
            }
            Long stock = this.getStock(appId, entity);
            result.add(new SeckillStockDto(it, appId, stock));
        }
        return result;
    }

    private long getStockByMemchache(Long stockId, DuibaSeckillEntity duibaSeckill, Long appId) {
        if (!seckillDateUtilService.isNowBeforeAfter5Min(duibaSeckill, new Date())) {
            seckillMemcacheService.invalidateStockCacheById(stockId);
            DuibaSeckillStockConfigEntity stockConfig = seckillStockService.findStockByUnique(duibaSeckill.getId(), appId);
            return stockConfig.getStock();
        }
        return seckillMemcacheService.getMemcachedStockById(stockId, duibaSeckill.getId(), appId);
    }

    private Long getStockId(final Long appId, final DuibaSeckillEntity duibaSeckill) throws Exception {
        String key = getStockCacheKey(appId, duibaSeckill.getId());
        if (!seckillDateUtilService.isNowBeforeAfter5Min(duibaSeckill, new Date())) {
            stockCache.invalidate(key);
            DuibaSeckillStockConfigEntity stockConfig = seckillStockService.findStockByUnique(duibaSeckill.getId(), appId);
            return stockConfig.getId();
        }
        SeckillStockCache sCache = stockCache.get(key, new Callable<SeckillStockCache>() {
            @Override
            public SeckillStockCache call() throws Exception {
                DuibaSeckillStockConfigEntity stockConfig = seckillStockService.findStockByUnique(duibaSeckill.getId(), appId);
                SeckillStockCache stockCache = new SeckillStockCache();
                stockCache.setId(stockConfig.getId());
                stockCache.setDuibaSeckillId(stockConfig.getSeckillId());
                stockCache.setAppId(appId);
                stockCache.setRemaining(stockConfig.getStock().longValue());
                return stockCache;
            }
        });
        return sCache.getId();
    }
}
