/**
 * Project Name:activity-center-biz
 * File Name:SignConfigServiceImpl.java
 * Package Name:cn.com.duiba.activity.center.biz.service.sign.impl
 * Date:2016年10月22日下午1:56:25
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.activity.center.biz.service.sign.impl;

import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.activity.center.api.dto.sign.SignConfigDto;
import cn.com.duiba.activity.center.biz.dao.sign.SignConfigDao;
import cn.com.duiba.activity.center.biz.entity.sign.SignConfigEntity;
import cn.com.duiba.activity.center.biz.plugin.buckle.ConsumerSignService;
import cn.com.duiba.activity.center.biz.service.sign.SignConfigService;
import cn.com.duiba.activity.center.biz.support.CacheConstants;
import cn.com.duiba.order.center.api.dto.ActivityOrderDto;
import cn.com.duiba.order.center.api.remoteservice.RemoteActivityOrderService;
import cn.com.duiba.service.domain.dataobject.ConsumerDO;
import cn.com.duiba.service.remoteservice.RemoteConsumerService;
import cn.com.duiba.wolf.cache.CacheClient;
import cn.com.duiba.wolf.cache.CacheLoader;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.utils.BeanUtils;


/**
 * ClassName:SignConfigServiceImpl <br/>
 * Date:     2016年10月22日 下午1:56:25 <br/>
 * @author   liyang
 * @version  
 * @since    JDK 1.6
 * @see 	 
 */
@Service
public class SignConfigServiceImpl implements SignConfigService {
    
    private static Logger log = LoggerFactory.getLogger(SignConfigServiceImpl.class);
    
    @Resource
    private SignConfigDao signConfigDao;
    @Resource
    private CacheClient cacheClient;
    @Autowired
    private RemoteConsumerService remoteConsumerService;
    @Autowired 
    private RemoteActivityOrderService remoteActivityOrderService;
    @Autowired
    private ConsumerSignService consumerSignService;
    
    private String getCacheKeyById(Long id){
        return CacheConstants.KEY_SIGN_CONFIG_BY_ID + id;
    }

    @Override
    public List<SignConfigDto> findAllByIds(List<Long> ids) {
        return BeanUtils.copyList(signConfigDao.findAllByIds(ids), SignConfigDto.class);
    }

    @Override
    public List<SignConfigDto> findByPage(Long appId, Integer offset, Integer max, String title, Integer id) {
        return BeanUtils.copyList(signConfigDao.findByPage(appId, offset, max, title, id), SignConfigDto.class);
    }

    @Override
    public Long findByPageCount(Long appId, String title, Integer id) {
        return signConfigDao.findByPageCount(appId, title, id);
    }

    @Override
    public SignConfigDto find(final Long id) {
        return cacheClient.getWithCacheLoader(getCacheKeyById(id), 30, TimeUnit.MINUTES, new CacheLoader<SignConfigDto>() {
            @Override
            public SignConfigDto load() {
                return BeanUtils.copy(signConfigDao.find(id), SignConfigDto.class);
            }
        });
    }

    @Override
    public SignConfigDto add(SignConfigDto sign) {
        SignConfigEntity e = BeanUtils.copy(sign, SignConfigEntity.class);
        signConfigDao.add(e);
        sign.setId(e.getId());
        return sign;
    }

    @Override
    public void delete(Long id) {
        signConfigDao.delete(id);
        cacheClient.remove(getCacheKeyById(id));
    }

    @Override
    public void update(SignConfigDto sign) {
        SignConfigEntity e = BeanUtils.copy(sign, SignConfigEntity.class);
        signConfigDao.update(e);
        cacheClient.remove(getCacheKeyById(sign.getId()));
    }

    @Override
    public int updateStatus(Long id, Integer status) {
        int ret = signConfigDao.updateStatus(id, status);
        cacheClient.remove(getCacheKeyById(id));
        return ret ;
    }

    @Override
    public String createOrder(Long consumerId, String transfer, String ip, String description, String ua,
                                           String os, SignConfigDto signDto, Integer signDays, Long addCredits) throws Exception {
        //查询用户信息
        ConsumerDO consumer = remoteConsumerService.find(consumerId);
        //产生订单号
        String orderNum = createOrder(consumer, signDto, ip, addCredits);
        consumerSignService.asyncConsumerPlug(consumer.getAppId(), consumerId, orderNum,transfer,ip,ua,os,signDto, signDays);
        return orderNum;
    }
    
    @Override
    public String addCreditsAgain(String orderNum,Long appId, Long consumerId, String transfer, String ip, String description, String ua,
                                           String os, SignConfigDto signDto, Integer signDays, Long addCredits) throws Exception {
        // 首先设置订单状态为：加积分处理中
        DubboResult<Boolean> result = remoteActivityOrderService.addCreditsStatusToProcessing(orderNum);
        if(result.isSuccess() && result.getResult()) {
            consumerSignService.asyncConsumerPlug(appId, consumerId, orderNum,transfer,ip,ua,os,signDto, signDays);
            return orderNum;
        } else {
            throw new Exception("交易中心修改加积分状态出错了");
        }
    }
    
    //生成订单
    private String createOrder(ConsumerDO consumer, SignConfigDto signDto,String ip, Long addCredits) {
        ActivityOrderDto orderDto = new ActivityOrderDto();
        orderDto.setConsumerId(consumer.getId());
        orderDto.setAppId(consumer.getAppId());
        orderDto.setPartnerUserId(consumer.getPartnerUserId());
        orderDto.setDuibaActivityId(signDto.getId());
        orderDto.setActivityType(ActivityOrderDto.TypeSign);
        orderDto.setConsumeCredits(0L);
        orderDto.setAddCredits(addCredits);
        orderDto.setExchangeStatus(ActivityOrderDto.ExchangeWait);
        orderDto.setAddCreditsStatus(ActivityOrderDto.AddCreditsProcessing);
        orderDto.setIp(ip);
        Date now = new Date();
        orderDto.setGmtCreate(now);
        orderDto.setGmtModified(now);
        String orderNum = null;
        try{
            orderNum = remoteActivityOrderService.createOrder(orderDto).getResult();
        }catch(Exception e){
            log.error("创建订单失败",e);
            throw e;
        }
        return orderNum;
    }
}

