package cn.com.duiba.rank.api.constants;

import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;
import java.util.Objects;

import cn.com.duiba.api.tools.DateUtils;
import cn.com.duiba.rank.api.dto.rank.RankConfigDto;
import cn.com.duiba.rank.api.enums.RankCycleTypeEnum;

public class RankConfigConstants {

	/** 排名限制数 */
	public static final int RANK_LIMIT_NUM = 5000;
	
	/** 每日排名redis缓存时间*/
	private static final int DAILY_RANK_EXPIRE_DAYS = 3;

	/** 每周排名redis缓存时间 */
	private static final int WEEKLY_RANK_EXPIRE_DAYS = 14;
	
	/** 每月排名redis缓存时间 */
	private static final int MONTH_RANK_EXPIRE_DAYS = 45;
	
	/** 赛季中心key */
	private String rankId;
	
	/** 失效天数 */
	private int expireDay;

	/** 开奖时间 */
	private Date openTime;

	//获取 排行榜redis 中key
	public RankConfigConstants(RankConfigDto rank, Long separatorId) {
		String rankIdtmp = rank.getId().toString()+"_"+separatorId +"_";
		//日榜yyyyMMdd
		if(Objects.equals(rank.getRankCycle(), RankCycleTypeEnum.DAY.getCode())) {
			this.rankId = rankIdtmp +DateUtils.getDayStr(new Date());
			this.expireDay = DAILY_RANK_EXPIRE_DAYS;
			this.openTime = getNextDay(new Date());
		//年终 第几周
		}else if (Objects.equals(rank.getRankCycle(), RankCycleTypeEnum.WEEK.getCode())) {
			this.rankId = rankIdtmp +DateUtils.getWeekOfYear(new Date());
			this.expireDay = WEEKLY_RANK_EXPIRE_DAYS;
			this.openTime = getNextWeekMonday(new Date());
		//yyyyMM
		}else if (Objects.equals(rank.getRankCycle(), RankCycleTypeEnum.MONTH.getCode())) {
			this.rankId = rankIdtmp +DateUtils.getYYYYMM(new Date());
			this.expireDay = MONTH_RANK_EXPIRE_DAYS;
			this.openTime = getNextMonthFirstDay();
		//无周期  暂时 不支持 自定义周期	  无周期  过期时间 根据配置的  排行榜 开奖时间  +7天
		}else if (Objects.equals(rank.getRankCycle(), RankCycleTypeEnum.NONE.getCode())) {
			this.rankId = rankIdtmp + "0";
			this.expireDay = daysBetween(new Date(),rank.getOpenPrizeTime());
			this.openTime = rank.getOpenPrizeTime();
		}
	}

	public String getRankId() {
		return rankId;
	}

	public void setRankId(String rankId) {
		this.rankId = rankId;
	}

	public int getExpireDay() {
		return expireDay;
	}

	public void setExpireDay(int expireDay) {
		this.expireDay = expireDay;
	}
	
	public Date getOpenTime() {
		return openTime;
	}

	public void setOpenTime(Date openTime) {
		this.openTime = openTime;
	}

	/**
	 * 计算两个日期之间相差的天数
	 * @param smdate 较小的时间
	 * @param bdate  较大的时间
	 * @return 相差天数
	 * @throws ParseException
	 */
	public static int daysBetween(Date smdate, Date bdate) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(smdate);
		long time1 = cal.getTimeInMillis();
		cal.setTime(bdate);
		long time2 = cal.getTimeInMillis();
		long betweenDays = (time2 - time1) / (1000 * 3600 * 24);
		return Integer.parseInt(String.valueOf(betweenDays));
	}
	
	/**
	 * 获取下一月第一天
	 * @return
	 */
	public static Date getNextMonthFirstDay(){
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(new Date());
		calendar.add(Calendar.MONTH, 1);
		calendar.set(Calendar.DAY_OF_MONTH, 1);
		calendar.set(Calendar.HOUR_OF_DAY, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();
	}
	
	private Date getNextDay(Date date) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		cal.add(Calendar.DATE, 1);
		return cal.getTime();
	}
	
	/**
	 * 获取下周第一天
	 * @return
	 */
	public static Date getNextWeekMonday(Date date) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(getThisWeekMonday(date));
		cal.add(Calendar.DATE, 7);
		return cal.getTime();
	}
	
	public static Date getThisWeekMonday(Date date) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		// 获得当前日期是一个星期的第几天
		int dayWeek = cal.get(Calendar.DAY_OF_WEEK);
		if (1 == dayWeek) {
			cal.add(Calendar.DAY_OF_MONTH, -1);
		}
		// 设置一个星期的第一天，按中国的习惯一个星期的第一天是星期一
		cal.setFirstDayOfWeek(Calendar.MONDAY);
		// 获得当前日期是一个星期的第几天
		int day = cal.get(Calendar.DAY_OF_WEEK);
		// 根据日历的规则，给当前日期减去星期几与一个星期第一天的差值
		cal.add(Calendar.DATE, cal.getFirstDayOfWeek() - day);
		return cal.getTime();
	}
}