package cn.com.duiba.anticheat.center.biz.remoteservice.activity.impl;

import cn.com.duiba.anticheat.center.api.constant.StaticListConstant;
import cn.com.duiba.anticheat.center.api.domain.activity.LotteryConsumerParams;
import cn.com.duiba.anticheat.center.api.domain.activity.LotteryOrderParams;
import cn.com.duiba.anticheat.center.api.domain.activity.LotteryRequestParams;
import cn.com.duiba.anticheat.center.api.remoteservice.activity.RemoteAnticheatLotteryCheckService;
import cn.com.duiba.anticheat.center.api.result.activity.ALCResultDto;
import cn.com.duiba.anticheat.center.biz.service.tongdun.FraudApiResponse;
import cn.com.duiba.anticheat.center.biz.service.tongdun.TongdunClient;
import cn.com.duiba.anticheat.center.biz.service.tongdun.TongdunThreadLocal;
import cn.com.duiba.anticheat.center.biz.strategy.activity.AnticheatLotteryStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.activity.impl.AnticheatLotteryBlackConsumerStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.activity.impl.AnticheatLotteryLessOneSecStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.activity.impl.AnticheatLotteryNotMobileStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.activity.impl.AnticheatLotterySameDeapStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.activity.impl.AnticheatLotterySameIpStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.activity.impl.AnticheatLotterySameUaCreditsStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.activity.impl.AnticheatLotteryTongDunHighStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.activity.impl.AnticheatLotteryWhiteAppStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.activity.impl.AnticheatLotteryWhiteIpStrategy;
import cn.com.duiba.anticheat.center.common.tool.ThreadTool;
import cn.com.duiba.wolf.dubbo.DubboResult;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Transaction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * 活动防作弊检查
 */
@RestController("remoteAnticheatLotteryCheckService")
public class RemoteAnticheatLotteryCheckServiceImpl implements InitializingBean, RemoteAnticheatLotteryCheckService {

    private static final Logger LOGGER = LoggerFactory.getLogger(RemoteAnticheatLotteryCheckServiceImpl.class);
    @Autowired
    private AnticheatLotteryLessOneSecStrategy anticheatLotteryLessOneSecStrategy;
    @Autowired
    private AnticheatLotterySameDeapStrategy anticheatLotterySameDeapStrategy;
    @Autowired
    private AnticheatLotterySameIpStrategy anticheatLotterySameIpStrategy;
    @Autowired
    private AnticheatLotteryBlackConsumerStrategy anticheatLotteryBlackConsumerStrategy;
    @Autowired
    private AnticheatLotteryNotMobileStrategy anticheatLotteryNotMobileStrategy;
    @Autowired
    private AnticheatLotterySameUaCreditsStrategy anticheatLotterySameUaCreditsStrategy;
    @Autowired
    private AnticheatLotteryTongDunHighStrategy anticheatLotteryTongDunHighStrategy;
    @Autowired
    private AnticheatLotteryWhiteIpStrategy anticheatLotteryWhiteIpStrategy;
    @Autowired
    private AnticheatLotteryWhiteAppStrategy anticheatLotteryWhiteAppStrategy;
    @Resource
    private ExecutorService executorService;

    /**
     * 超时时间 80毫秒
     */
    private static final int TIMEOUT_TIME = 80;

    /**
     * 信号量
     */
    private static final Semaphore SEMAPHORE = new Semaphore(100);

    private List<AnticheatLotteryStrategy> tongDunStrategies = new ArrayList<>();
    private List<AnticheatLotteryStrategy> blackStrategies = new ArrayList<>();
    private List<AnticheatLotteryStrategy> whiteStrategies = new ArrayList<>();

    @Override
    public void afterPropertiesSet() throws Exception {
        tongDunStrategies.add(anticheatLotteryTongDunHighStrategy);

        blackStrategies.add(anticheatLotteryLessOneSecStrategy);
        blackStrategies.add(anticheatLotterySameDeapStrategy);
        blackStrategies.add(anticheatLotteryBlackConsumerStrategy);
        blackStrategies.add(anticheatLotterySameIpStrategy);
        blackStrategies.add(anticheatLotteryNotMobileStrategy);
        blackStrategies.add(anticheatLotterySameUaCreditsStrategy);

        whiteStrategies.add(anticheatLotteryWhiteAppStrategy);
        whiteStrategies.add(anticheatLotteryWhiteIpStrategy);
    }

    @Override
    public DubboResult<ALCResultDto> checkLottery(final LotteryConsumerParams consumer, final LotteryOrderParams order, final LotteryRequestParams request) {
        //命中APP白名单直接通过
        if(StaticListConstant.matchWhiteApp(consumer.getAppId())){
            return DubboResult.successResult(new ALCResultDto(true, null));
        }
        Callable<DubboResult<ALCResultDto>> callable = new Callable<DubboResult<ALCResultDto>>() {
            @Override
            public DubboResult<ALCResultDto> call() throws Exception {
                ALCResultDto result = innerCheckLottery(consumer, order, request);
                return DubboResult.successResult(result);
            }
        };
        DubboResult<ALCResultDto> successResult = DubboResult.successResult(new ALCResultDto(true, null));
        if(SEMAPHORE.tryAcquire()) {
            Future<DubboResult<ALCResultDto>> future = null;
            try {
                future = executorService.submit(callable);
                DubboResult<ALCResultDto> result = future.get(TIMEOUT_TIME, TimeUnit.MILLISECONDS);
                return result != null ? result : successResult;
            } catch(TimeoutException e){
                LOGGER.info("抽奖防作弊降级,业务超时, errMsg={}", e.getMessage());
                LOGGER.debug("抽奖防作弊降级,业务超时", e);
                ThreadTool.cancelTask(future);
                return successResult;
            }catch (Exception e) {
                LOGGER.warn("抽奖防作弊降级,异常信息", e);
                ThreadTool.cancelTask(future);
                return successResult;
            } finally {
                SEMAPHORE.release();
            }
        } else {
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("抽奖防作弊降级通过,流量超了, lotteryOrderId={}, type={}, appId={}, consumerId={}", order.getLotteryOrderId(), order.getType().desc(), consumer.getAppId(), consumer.getConsumerId());
            }
            return successResult;
        }
    }

    private ALCResultDto innerCheckLottery(LotteryConsumerParams consumer, LotteryOrderParams order, LotteryRequestParams request) {
        ALCResultDto ret = null;

        //白名单验证
        ret = whiteStrategyMatch(consumer, order, request);
        if(ret != null){
            return ret;
        }

        //黑名单策略验证
        ret = blackStrategyMatch(consumer, order, request, blackStrategies);
        if(ret != null){
            return ret;
        }

        //同盾策略验证
        ret = tongDunStrategyMatch(consumer, order, request);
        if(ret != null){
            return ret;
        }

        // 如果都没有命中，默认放行
        return new ALCResultDto(true, null);
    }

    private ALCResultDto whiteStrategyMatch(LotteryConsumerParams consumer, LotteryOrderParams order,
                                             LotteryRequestParams request){
        for (AnticheatLotteryStrategy as : whiteStrategies) {
            Transaction t = Cat.newTransaction("URL", as.getClass().getSimpleName());
            try {
                AnticheatLotteryStrategy.AnticheatLotteryStrategyResult ret = as.checkLotteryFast(consumer, order, request);
                t.setStatus(Transaction.SUCCESS);
                if (judgeMatch(ret, as, consumer)) {
                    return new ALCResultDto(true, "通过验证");
                }
            } catch (Exception e) {
                t.setStatus(e);
                throw e;
            } finally {
                t.complete();
            }
        }
        return null;
    }

    /**
     * 只验证有效的防作弊策略，避免无效的同盾调用
     * @param consumer
     * @param order
     * @param request
     * @return
     */
    private ALCResultDto tongDunStrategyMatch(LotteryConsumerParams consumer, LotteryOrderParams order,
                                               LotteryRequestParams request){
        List<AnticheatLotteryStrategy> targetStrategies = new ArrayList<>();
        for(AnticheatLotteryStrategy strategy : tongDunStrategies){
            if(strategy.needValid(consumer)){
                targetStrategies.add(strategy);
            }
        }

        if(targetStrategies.isEmpty()){
            return null;
        }

        FraudApiResponse response = TongdunClient.checkLotteryTongdun(consumer, order, request);
        TongdunThreadLocal.get().setApiResponse(response);

        return blackStrategyMatch(consumer, order, request, targetStrategies);
    }

    private ALCResultDto blackStrategyMatch(LotteryConsumerParams consumer, LotteryOrderParams order,
                                             LotteryRequestParams request, List<AnticheatLotteryStrategy> strategies){
        for (AnticheatLotteryStrategy as : strategies) {
            Transaction t = Cat.newTransaction("URL", as.getClass().getSimpleName());
            try {
                AnticheatLotteryStrategy.AnticheatLotteryStrategyResult ret = as.checkLotteryFast(consumer, order, request);
                t.setStatus(Transaction.SUCCESS);
                if(judgeMatch(ret, as, consumer)){
                    return new ALCResultDto(false, "抽奖防作弊策略");
                }
            } catch (Exception e) {
                t.setStatus(e);
                throw e;
            } finally {
                t.complete();
            }
        }
        return null;
    }

    private boolean judgeMatch(AnticheatLotteryStrategy.AnticheatLotteryStrategyResult ret,
                                   AnticheatLotteryStrategy as, LotteryConsumerParams consumer){
        if (!ret.isMatch()) {
            return false;
        }
        if(!as.needValid(consumer)){
            return false;
        }
        return true;
    }

}
