package cn.com.duiba.anticheat.center.biz.service.rules.impl;

import cn.com.duiba.anticheat.center.biz.dao.rules.RuleConfigDao;
import cn.com.duiba.anticheat.center.common.exceptions.AnticheatException;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.drools.core.impl.KnowledgeBaseImpl;
import org.kie.api.KieBase;
import org.kie.api.io.ResourceType;
import org.kie.internal.builder.KnowledgeBuilder;
import org.kie.internal.builder.KnowledgeBuilderFactory;
import org.kie.internal.io.ResourceFactory;
import org.kie.internal.utils.KieHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.PostConstruct;

import cn.com.duiba.anticheat.center.api.enums.RuleSceneEnum;
import cn.com.duiba.anticheat.center.api.result.rules.RuleChangeDto;
import cn.com.duiba.anticheat.center.biz.entity.rules.RuleConfigEntity;
import cn.com.duiba.anticheat.center.biz.service.rules.RuleLoadService;

/**
 * Created by sty on 2018/7/24.
 */
@Component
public class KieComponent {
  private static final Logger LOGGER = LoggerFactory.getLogger(KieComponent.class);
  private static final String DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";

  @Autowired
  private RuleLoadService ruleLoadService;
  @Autowired
  private RuleConfigDao ruleConfigDao;

  //KnowledgeBase 缓存(key：场景标识)
  private static Map<Integer, KieBase> ruleMap = new ConcurrentHashMap<>();

  @PostConstruct
  public void init(){
    System.setProperty("drools.dateformat", DATE_FORMAT);

    List<RuleConfigEntity> rules = ruleLoadService.getAllRule();
    Map<Integer,List<RuleConfigEntity>> ruleSceneMap=Maps.newHashMap();

    rules.forEach(one->{
        List<RuleConfigEntity>  curList= ruleSceneMap.get(one.getRuleScene());
        if(curList == null){
          curList = Lists.newArrayList();
        }
        curList.add(one);
        ruleSceneMap.put(one.getRuleScene(),curList);
    });

    //循环所有场景，进行规则统一初始化
    for(RuleSceneEnum ruleSingleEnum : RuleSceneEnum.values()){
      Integer codeForRule = ruleSingleEnum.getType();
      List<RuleConfigEntity> sceneRules = ruleSceneMap.get(codeForRule);

      KieHelper helper = new KieHelper();
      if(CollectionUtils.isNotEmpty(sceneRules)) {
        sceneRules.forEach(one -> helper.addContent(one.getRuleContent(), ResourceType.DRL));
      }
      //放入内存
      ruleMap.put(codeForRule, helper.build());
    }
  }


  public static KieBase getBaseByScene(Integer scene){
    return ruleMap.get(scene);
  }

  public void changeRule(String opType, RuleSceneEnum scene, String ruleName, String ruleContent){
    //获取相对应场景的kbase
    KnowledgeBaseImpl kbase = (KnowledgeBaseImpl) KieComponent.getBaseByScene(scene.getType());
    if(kbase == null){
      return;
    }
    if(RuleChangeDto.DELETE.equals(opType)){
      //删除
      kbase.removeRule(scene.getPkg(),ruleName);
    }
    if(RuleChangeDto.ADD.equals(opType) && StringUtils.isNotBlank(ruleContent) && kbaseAddRule(ruleContent, kbase)){
      //增加
      return;
    }
    if(RuleChangeDto.UPDATE.equals(opType) && StringUtils.isNotBlank(ruleContent)){
      //更新
      updateKbase(scene, ruleName, ruleContent, kbase);
    }
    // 刷新
    if (RuleChangeDto.REFRESH.equals(opType)) {
      refresh(scene);
    }

    // 打开
    if (RuleChangeDto.OPEN.equals(opType)) {
      open(ruleName, kbase);
    }

  }

  private void open(String ruleName, KnowledgeBaseImpl kbase) {
    RuleConfigEntity rule = ruleConfigDao.getByRuleName(ruleName);
    if (rule == null) {
      throw new AnticheatException("规则不存在, " + ruleName);
    }

    KnowledgeBuilder builder = KnowledgeBuilderFactory.newKnowledgeBuilder();
    builder.add(ResourceFactory.newByteArrayResource(rule.getRuleContent().getBytes(StandardCharsets.UTF_8)), ResourceType.DRL);
    kbase.addPackages(builder.getKnowledgePackages());
  }

  private void refresh(RuleSceneEnum scene) {
    List<RuleConfigEntity> rules = ruleLoadService.getAllRule();
    KieHelper helper = new KieHelper();
    for (RuleConfigEntity rule : rules) {
      if (rule.getRuleScene().equals(scene.getType())) {
        helper.addContent(rule.getRuleContent(), ResourceType.DRL);
      }
    }

    KieBase kieBase = helper.build();
    ruleMap.put(scene.getType(), kieBase);

  }

  private boolean kbaseAddRule(String ruleContent, KnowledgeBaseImpl kbase) {
    //增加
    KnowledgeBuilder kb = KnowledgeBuilderFactory.newKnowledgeBuilder();
    //装入规则，可以装入多个
    try {
      kb.add(ResourceFactory.newByteArrayResource(ruleContent.getBytes("utf-8")), ResourceType.DRL);
    } catch (UnsupportedEncodingException e) {
      LOGGER.error("规则转为二进制转换出错", e);
      return true;
    }
    kbase.addPackages(kb.getKnowledgePackages());
    return false;
  }

  private void updateKbase(RuleSceneEnum scene, String ruleName, String ruleContent,
      KnowledgeBaseImpl kbase) {
    KnowledgeBuilder kb = KnowledgeBuilderFactory.newKnowledgeBuilder();

    //装入规则，可以装入多个
    try {
      kb.add(ResourceFactory.newByteArrayResource(ruleContent.getBytes("utf-8")), ResourceType.DRL);
    } catch (UnsupportedEncodingException e) {
      LOGGER.error("规则转为二进制转换出错", e);
      return;
    }
    //知识库移除,并增加
    kbase.removeRule(scene.getPkg(),ruleName);
    kbase.addPackages(kb.getKnowledgePackages());
  }


}
