package cn.com.duiba.anticheat.center.biz.service.risk.impl;

import cn.com.duiba.anticheat.center.api.dto.RiskBlackListDto;
import cn.com.duiba.anticheat.center.api.param.RiskBlackListParam;
import cn.com.duiba.anticheat.center.biz.constant.RedisKeyFactory;
import cn.com.duiba.anticheat.center.biz.dao.risk.RiskBlackListDao;
import cn.com.duiba.anticheat.center.biz.service.risk.RiskBlackListService;
import cn.com.duiba.anticheat.center.common.constants.DBConstants;
import cn.com.duiba.api.bo.page.Page;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

@Service
public class RiskBlackListServiceImpl implements RiskBlackListService {
    private static final Logger LOGGER = LoggerFactory.getLogger(RiskBlackListServiceImpl.class);

    public static final String SPACE = "_";
    @Resource
    private RiskBlackListDao riskBlackListDao;

    @Autowired
    private RedisTemplate redisTemplate;

    @Autowired
    private ExecutorService executorService;

    @Override
    public int deleteByPrimaryKey(Long id) {
        RiskBlackListDto record = this.selectByPrimaryKey(id);
        int i = riskBlackListDao.deleteByPrimaryKey(id);
        if (null != record) {
            deleteKey(record);
        }
        return i;
    }

    private void deleteKey(RiskBlackListDto record) {
        //延时双删保证一致性
        String key = buildRedisKey(record.getAppId(), record.getRiskScene());
        redisTemplate.delete(key);
        executorService.submit(() -> {
            try {
                Thread.sleep(2000L);
            } catch (InterruptedException e) {
                LOGGER.error("系统错误", e);
            }
            redisTemplate.delete(buildRedisKey(record.getAppId(), record.getRiskScene()));
        });
    }

    @Override
    public Long insert(RiskBlackListDto record) {
        int insert = riskBlackListDao.insert(record);
        //延时双删保证一致性
        deleteKey(record);
        return record.getId();
    }

    @Override
    public RiskBlackListDto selectByPrimaryKey(Long id) {
        return riskBlackListDao.selectByPrimaryKey(id);
    }

    @Override
    public int updateByPrimaryKeySelective(RiskBlackListDto record) {
        RiskBlackListDto old = this.selectByPrimaryKey(record.getId());
        int i = riskBlackListDao.updateByPrimaryKeySelective(record);
        if (null != old) {
            deleteKey(old);
        }
        deleteKey(record);
        return i;
    }

    @Override
    @Transactional(value = DBConstants.DATABASE_DEVELOPER, rollbackFor = Exception.class)
    public int batchInsert(List<RiskBlackListDto> list) {
        int i = riskBlackListDao.batchInsert(list);
        Set<String> keys = new HashSet<>();
        for (RiskBlackListDto riskBlackListDto : list) {
            keys.add(buildRedisKey(riskBlackListDto.getAppId(), riskBlackListDto.getRiskScene()));
        }
        redisTemplate.delete(keys);

        executorService.submit(() -> {
            try {
                Thread.sleep(2000L);
            } catch (InterruptedException e) {
                LOGGER.error("系统错误", e);
            }
            redisTemplate.delete(keys);
        });
        return i;
    }

    @Override
    public Page<RiskBlackListDto> listByPage(RiskBlackListParam riskBlackListParam) {
        List<RiskBlackListDto> list = riskBlackListDao.listByCondition(riskBlackListParam);
        Integer count = riskBlackListDao.selectCountByCondition(riskBlackListParam);
        Page<RiskBlackListDto> dtoPage = new Page<>();
        dtoPage.setTotalCount(count);
        dtoPage.setList(list);
        return dtoPage;
    }

    @Override
    public Boolean checkOnly(Long id, Long appId, Integer riskType, Integer riskScene, String riskValue) {
        RiskBlackListDto riskBlackListDto = riskBlackListDao.findByParam(appId, riskType, riskScene, riskValue);
        if (null == riskBlackListDto || Objects.equals(id, riskBlackListDto.getId())) {
            return false;
        }
        return true;
    }

    @Override
    public Boolean riskBlacklistMatching(Long appId, Integer riskScene, String ip, Long customerId) {
        if (StringUtils.isBlank(ip) || customerId == null) {
            return true;
        }
        String redisKey = buildRedisKey(appId, riskScene);
        if (!redisTemplate.hasKey(redisKey)) {
            TimeUnit timeUnit;
            Integer timeout;
            List<String> list = riskBlackListDao.listByAppIdAndRiskScene(appId, riskScene);
            if (list == null) {
                list = new ArrayList<>();
            }
            if (list.size() == 0) {
                //为空缓存十分钟
                timeUnit = TimeUnit.MINUTES;
                timeout = 10;
                list.add("");
            } else {
                //缓存一天
                timeUnit = TimeUnit.DAYS;
                timeout = 1;
            }
            redisTemplate.opsForSet().add(redisKey, list.toArray(new String[list.size()]));
            redisTemplate.expire(redisKey, timeout, timeUnit);
        }
        return redisTemplate.opsForSet().isMember(redisKey, ip) || redisTemplate.opsForSet().isMember(redisKey, String.valueOf(customerId));
    }

    private String buildRedisKey(Long appId, Integer riskScene) {
        return RedisKeyFactory.K034.toString() + "risk" + SPACE + "blackList" + SPACE + appId + SPACE + riskScene;
    }

}
