package cn.com.duiba.anticheat.center.biz.service.risk.impl;

import cn.com.duiba.anticheat.center.api.dto.RiskWhiteListDto;
import cn.com.duiba.anticheat.center.api.enums.ActRiskSenceEnum;
import cn.com.duiba.anticheat.center.api.enums.RiskWhiteListTypeEnum;
import cn.com.duiba.anticheat.center.api.param.RiksWhitelistMatchingParam;
import cn.com.duiba.anticheat.center.api.param.RiskWhiteListParam;
import cn.com.duiba.anticheat.center.biz.constant.RedisKeyFactory;
import cn.com.duiba.anticheat.center.biz.dao.risk.RiskWhiteListDao;
import cn.com.duiba.anticheat.center.biz.entity.risk.RiskWhiteListEntity;
import cn.com.duiba.anticheat.center.biz.service.risk.RiskWhiteListService;
import cn.com.duiba.anticheat.center.common.constants.DBConstants;
import cn.com.duiba.anticheat.center.common.constants.RiskConstant;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.SetOperations;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author JunAngLiu
 * @Title: RiskWhiteListServiceImpl
 * @Description:
 * @date 2019/6/1821:41
 */
@Service
public class RiskWhiteListServiceImpl implements RiskWhiteListService {
    private static final Logger LOGGER = LoggerFactory.getLogger(RiskWhiteListServiceImpl.class);

    private static final String SPACE = "_";

    @Autowired
    private RiskWhiteListDao riskWhiteListDao;
    @Autowired
    private RedisTemplate redisTemplate;






    @Transactional(DBConstants.DATABASE_DEVELOPER)
    @Override
    public int insert(RiskWhiteListDto dto) throws BizException {
        if(Objects.isNull(dto)){
            return 0;
        }
        RiskWhiteListParam param = new RiskWhiteListParam();
        param.setAppIds(Lists.newArrayList(dto.getAppId()));
        param.setType(RiskWhiteListTypeEnum.getEnumByValue(dto.getType()));
        param.setConsumerId(dto.getConsumerId());
        param.setRiskSence(ActRiskSenceEnum.getEnumByValue(dto.getRiskSence()));
        param.setIp(dto.getIp());
        if(this.countByAppConsumerTypeSences(param) > 0){
            throw new BizException("该配置已存在");
        }
        int result = riskWhiteListDao.insert(BeanUtils.copy(dto, RiskWhiteListEntity.class));
        if(result > 0){
            redisTemplate.delete(this.getKey(ActRiskSenceEnum.getEnumByValue(dto.getRiskSence())));
        }
        return result;
    }

    @Transactional(DBConstants.DATABASE_DEVELOPER)
    @Override
    public int updateById(RiskWhiteListDto dto) {

        if(Objects.isNull(dto) || Objects.isNull(dto.getId())){
            return 0;
        }
        RiskWhiteListEntity entity = riskWhiteListDao.getById(dto.getId());
        if(Objects.isNull(entity)){
            return 0;
        }
        int result = riskWhiteListDao.updateById(BeanUtils.copy(dto, RiskWhiteListEntity.class));
        if(result > 0){
            redisTemplate.delete(getKey(ActRiskSenceEnum.getEnumByValue(entity.getRiskSence())));
            if(dto.getRiskSence() != null && !Objects.equals(dto.getRiskSence(),entity.getRiskSence())){
                redisTemplate.delete(getKey(ActRiskSenceEnum.getEnumByValue(dto.getRiskSence())));
            }
        }
        return result;
    }

    @Override
    public RiskWhiteListEntity getById(Long id) {

        return riskWhiteListDao.getById(id);
    }




    @Override
    public List<RiskWhiteListEntity> listByAppConsumerTypeSences(RiskWhiteListParam param) {
        return riskWhiteListDao.listByAppConsumerTypeSences(param);
    }

    @Override
    public int countByAppConsumerTypeSences(RiskWhiteListParam param) {
        return riskWhiteListDao.countByAppConsumerTypeSences(param);
    }

    @Override
    public Boolean riskWhitelistMatching(RiksWhitelistMatchingParam riksWhitelistMatchingParam) throws BizException {
        Long appId = riksWhitelistMatchingParam.getAppId();
        Long consumerId = riksWhitelistMatchingParam.getConsumerId();
        ActRiskSenceEnum riskSence = riksWhitelistMatchingParam.getRiskSence();
        String ip = riksWhitelistMatchingParam.getIp();
        this.checkParam(appId, consumerId, riskSence, ip);
        String redisKey = this.getKey(riskSence);
        if (BooleanUtils.isNotTrue(redisTemplate.hasKey(redisKey))) {
            RiskWhiteListParam param = new RiskWhiteListParam();
            param.setPageSize(null);
            param.setPageNo(null);
            param.setRiskSence(riskSence);
            List<RiskWhiteListEntity> list = riskWhiteListDao.listByAppConsumerTypeSences(param);
            String[] array;
            if(CollectionUtils.isNotEmpty(list)){
                array = new String[list.size()];
                for (int i = 0; i < list.size(); i++) {
                    array[i] = this.getValueByType(list.get(i));
                }
            }else{
                //初始化空值，防止 大部分没有配置白名单的用户 查询数据库
                array = new String[]{"none"};
            }
            redisTemplate.opsForSet().add(redisKey, array);
            redisTemplate.expire(redisKey, 10, TimeUnit.MINUTES);
        }
        SetOperations setOperations = redisTemplate.opsForSet();
        return BooleanUtils.isTrue(redisTemplate.hasKey(redisKey)) &&
                (setOperations.isMember(redisKey,this.buildRedisSetValue(RiskConstant.RISK_TYPE_APP_WHITELIST,String.valueOf(appId)))
                ||setOperations.isMember(redisKey,this.buildRedisSetValue(RiskConstant.RISK_TYPE_APP_USER_WHITELIST,String.valueOf(consumerId)))
                ||setOperations.isMember(redisKey,this.buildRedisSetValue(RiskConstant.RISK_TYPE_IP_WHITELIST,String.valueOf(ip))));

    }

    /**
     * 根据类型构造不同的 元素
     * @param e
     * @return
     * @throws BizException
     */
    private String getValueByType(RiskWhiteListEntity e) throws BizException {
        String result = "";
        switch (e.getType()){
                case RiskConstant.RISK_TYPE_APP_WHITELIST:
                    result = buildRedisSetValue(e.getType(),String.valueOf(e.getAppId()));
                    break;
                case RiskConstant.RISK_TYPE_APP_USER_WHITELIST:
                    result = buildRedisSetValue(e.getType(),String.valueOf(e.getConsumerId()));
                    break;
                case RiskConstant.RISK_TYPE_IP_WHITELIST:
                    result = buildRedisSetValue(e.getType(),String.valueOf(e.getIp()));
                    break;
                default:
                    throw new BizException(String.format("白名单类型异常%s",e.getType()));
        }
        return result;
    }

    /**
     * 根据白名单类型 拼成不同的 值，防止 在set中 不同类型重复
     * @param type
     * @param e
     * @return
     * @throws BizException
     */
    private String buildRedisSetValue(Integer type,String e) throws BizException {
        if(type == null){
            throw new BizException(String.format("白名单类型异常%s",type));
        }
        return String.format("%s%s%s",type,SPACE,e);

    }



    /**
     * 详细参数判断验证
     * @param appId
     * @param consumerId
     * @param riskSence
     * @param ip
     * @throws BizException
     */
    private void checkParam(Long appId, Long consumerId, ActRiskSenceEnum riskSence, String ip) throws BizException{
        if(Objects.isNull(appId)){
            throw new BizException("参数{appId}不允许为空");
        }

        if(Objects.isNull(consumerId)){
            throw new BizException("参数{consumerId}不允许为空");
        }

        if(Objects.isNull(riskSence)){
            throw new BizException("参数{riskSence}不允许为空");
        }

        if(StringUtils.isEmpty(ip)){
            throw new BizException("参数{ip}不允许为空");
        }
    }




    private String getKey(ActRiskSenceEnum riskSence){
        return RedisKeyFactory.K035.toString() + riskSence.getValue();
    }
}
