package cn.com.duiba.anticheat.center.biz.strategy.activity.impl;

import cn.com.duiba.anticheat.center.api.domain.activity.LotteryConsumerParams;
import cn.com.duiba.anticheat.center.api.domain.activity.LotteryOrderParams;
import cn.com.duiba.anticheat.center.api.domain.activity.LotteryOrderParams.LotteryOrderTypeEnum;
import cn.com.duiba.anticheat.center.api.domain.activity.LotteryRequestParams;
import cn.com.duiba.anticheat.center.biz.dao.activity.AnticheatLotteryDebugLogDao;
import cn.com.duiba.anticheat.center.biz.entity.activity.AnticheatLotteryDebugLogEntity;
import cn.com.duiba.anticheat.center.biz.entity.activity.AnticheatLotteryStrategyAppConfigEntity;
import cn.com.duiba.anticheat.center.biz.entity.activity.AnticheatLotteryStrategyConfigEntity;
import cn.com.duiba.anticheat.center.biz.strategy.activity.AnticheatLotteryStrategy;
import com.google.common.base.Objects;
import com.google.common.collect.HashMultimap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.Map;
import java.util.Set;

/**
 * 在抽奖队列中,出现4个抽奖行为使用一个 ip 对应到4个不同账号,那么,从该 ip 的第五个账号的第一次抽奖行为开始,命中本策略
 * 命中前,4个账号无限抽奖,都不在考虑范围内
 * 命中后,从该 ip 的第五个账号第一次抽奖行为开始,全部算命中
 *
 * @author Yanf Guo
 */
@Component
public class AnticheatLotterySameIpStrategy implements AnticheatLotteryStrategy {

    /**
     * key: operatingActivityId - ip
     * value: Set<LotteryRecord>
     */
    private static final HashMultimap<String, LotteryRecord> sameIpMap = HashMultimap.create();

    /**
     * 超过4个用户
     */
    private static final int CONSUMER_IP_LIMIT_COUNT = 4;

    /**
     * map key 的最大个数
     */
    private static final int MAP_SIZE = 10000;

    @Autowired
    private AnticheatLotteryStrategyConfigService anticheatLotteryStrategyConfigService;

    @Autowired
    private AnticheatLotteryDebugLogDao anticheatLotteryDebugLogDao;

    @Override
    public int getEffectMode() {
        return anticheatLotteryStrategyConfigService.getCacheConfig(AnticheatLotteryStrategyConfigEntity.TYPE_SAME_IP).getEffectMode();
    }

    @Override
    public boolean isEnable() {
        return anticheatLotteryStrategyConfigService.getCacheConfig(AnticheatLotteryStrategyConfigEntity.TYPE_SAME_IP).getEnable();
    }

    @Override
    public boolean isAppEnable(Long appId) {
        Map<Long, AnticheatLotteryStrategyAppConfigEntity> map = anticheatLotteryStrategyConfigService.getCacheAppConfig(AnticheatLotteryStrategyConfigEntity.TYPE_SAME_IP);
        return map.containsKey(appId);
    }

    @Override
    public AnticheatLotteryStrategyResult checkLottery(LotteryConsumerParams consumer, LotteryOrderParams order, LotteryRequestParams request) {
        LotteryRecord record = new LotteryRecord();
        record.setIp(request.getIp());
        record.setOperatingActivityId(order.getOperatingActivityId());
        record.setDuibaActivityId(order.getDuibaActivityId());
        record.setTime(new Date());
        record.setType(order.getType());
        record.setLotteryOrderId(order.getLotteryOrderId());
        record.setConsumerId(consumer.getConsumerId());
        record.setAppId(consumer.getAppId());

        String key = getKey(order.getOperatingActivityId(), request.getIp());
        if (sameIpMap.size() > MAP_SIZE) {
            sameIpMap.clear();
        }
        sameIpMap.put(key, record);

        Set<LotteryRecord> records = sameIpMap.get(key);
        if (records == null) {  // 由于并发导致 put 后被另一个线程 clear,则默认不命中
            return new AnticheatLotteryStrategyResult(false);
        }

        if (records.size() > CONSUMER_IP_LIMIT_COUNT) {
            Long debugId = doMatchProcess(consumer, order, request, records.size());
            return new AnticheatLotteryStrategyResult(true, debugId);
        }
        return new AnticheatLotteryStrategyResult(false);
    }

    /**
     * 组装 sameipmap 的 key
     *
     * @param operatingActivityId
     * @param ip
     * @return
     */
    private static String getKey(Long operatingActivityId, String ip) {
        return String.format("%s-%s", operatingActivityId, ip);
    }

    /**
     * 每次进入此策略进行验证
     */
    private class LotteryRecord {
        private LotteryOrderTypeEnum type;
        private Long operatingActivityId;
        private Long duibaActivityId;
        private Date time;    // 当前时间
        private String ip;
        private Long lotteryOrderId;
        private Long consumerId;
        private Long appId;

        @Override
        public int hashCode() {
            return Objects.hashCode(consumerId);
        }

        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof LotteryRecord)) {
                return false;
            }
            LotteryRecord o = (LotteryRecord) obj;
            return this.consumerId.equals(o.consumerId);
        }

        public LotteryOrderTypeEnum getType() {
            return type;
        }

        public void setType(LotteryOrderTypeEnum type) {
            this.type = type;
        }

        public Long getOperatingActivityId() {
            return operatingActivityId;
        }

        public void setOperatingActivityId(Long operatingActivityId) {
            this.operatingActivityId = operatingActivityId;
        }

        public Long getDuibaActivityId() {
            return duibaActivityId;
        }

        public void setDuibaActivityId(Long duibaActivityId) {
            this.duibaActivityId = duibaActivityId;
        }

        public Date getTime() {
            return time;
        }

        public void setTime(Date time) {
            this.time = time;
        }

        public String getIp() {
            return ip;
        }

        public void setIp(String ip) {
            this.ip = ip;
        }

        public Long getLotteryOrderId() {
            return lotteryOrderId;
        }

        public void setLotteryOrderId(Long lotteryOrderId) {
            this.lotteryOrderId = lotteryOrderId;
        }

        public Long getConsumerId() {
            return consumerId;
        }

        public void setConsumerId(Long consumerId) {
            this.consumerId = consumerId;
        }

        public Long getAppId() {
            return appId;
        }

        public void setAppId(Long appId) {
            this.appId = appId;
        }
    }

    private Long doMatchProcess(LotteryConsumerParams consumer, LotteryOrderParams order, LotteryRequestParams request, int count) {
        AnticheatLotteryDebugLogEntity debug = new AnticheatLotteryDebugLogEntity();
        debug.setStrategyType(AnticheatLotteryStrategyConfigEntity.TYPE_SAME_IP);
        debug.setConsumerId(consumer.getConsumerId());
        debug.setPartnerUserId(consumer.getPartnerUserId());
        debug.setIp(request.getIp());
        debug.setLotteryOrderId(order.getLotteryOrderId());
        debug.setRelationId(order.getDuibaActivityId());
        debug.setRelationType(order.getType().value());
        debug.setOperatingActivityId(order.getOperatingActivityId());
        debug.setAppId(order.getAppId());
        debug.setMessage(String.format("此 IP(%s) 连续使用第 %s 个账号进行 %s 活动抽奖(app活动id: %s, duiba活动id(或app-单品抽奖id): %s)", request.getIp(), count, order.getType().desc(), order.getOperatingActivityId(), order.getDuibaActivityId()));
        anticheatLotteryDebugLogDao.insert(debug);

        return debug.getId();
    }
}
