package cn.com.duiba.anticheat.center.biz.strategy.goods.impl;

import cn.com.duiba.anticheat.center.api.domain.goods.BehaviorParams;
import cn.com.duiba.anticheat.center.api.domain.goods.ConsumerParams;
import cn.com.duiba.anticheat.center.api.domain.goods.GoodsParams;
import cn.com.duiba.anticheat.center.api.domain.goods.RequestParams;
import cn.com.duiba.anticheat.center.biz.dao.goods.AnticheatDebugLogDao;
import cn.com.duiba.anticheat.center.biz.dao.goods.ConsumerFootprintDao;
import cn.com.duiba.anticheat.center.biz.entity.goods.*;
import cn.com.duiba.anticheat.center.biz.strategy.goods.AnticheatStrategy;
import cn.com.duiba.anticheat.center.common.constants.CacheConstants;
import cn.com.duiba.wolf.redis.RedisAtomicClient;
import cn.com.duiba.wolf.utils.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author Yanf Guo 某个应用下，针对某个兑换项，没手势的用户第一笔兑换，发生4次后，第5次命中
 *
 *     // 手势不存在
 *     private static final int SWIPE_NOT_EXIST = 0;
 *     // 手势存在
 *     private static final int SWIPE_EXIST = 1;
 *     // 验证出错
 *     private static final int SWIPE_ERROR = 2;
 *     // 不需要验证手势
 *     private static final int SWIPE_SKIP = 3;
 */
@Component
public class AnticheatFirstNoSwipeStrategy implements AnticheatStrategy {

    // 手势不存在
    private static final int SWIPE_NOT_EXIST = 0;

    @Resource(name = "redisTemplate")
    private RedisAtomicClient redisAtomicClient;

    @Autowired
    private AnticheatStrategyConfigService anticheatStrategyConfigService;

    @Autowired
    private AnticheatDebugLogDao anticheatDebugLogDao;

    @Autowired
    private ConsumerFootprintDao consumerFootprintDao;

    @Override
    public int getEffectMode() {
        return anticheatStrategyConfigService.getCacheConfig(AnticheatStrategyConfigEntity.TYPE_FIRST_NO_SWIPE).getEffectMode();
    }

    @Override
    public int getCheckMode() {
        return anticheatStrategyConfigService.getCacheConfig(AnticheatStrategyConfigEntity.TYPE_FIRST_NO_SWIPE).getCheckMode();
    }

    @Override
    public boolean isEnable() {
        return anticheatStrategyConfigService.getCacheConfig(AnticheatStrategyConfigEntity.TYPE_FIRST_NO_SWIPE).getEnable();
    }

    @Override
    public boolean isBlackMode() {
        return true;
    }

    @Override
    public boolean isAppEnable(Long appId) {
        Map<Long, AnticheatStrategyAppConfigEntity> map = anticheatStrategyConfigService.getCacheAppConfig(AnticheatStrategyConfigEntity.TYPE_FIRST_NO_SWIPE);
        return map.containsKey(appId);
    }

    @Override
    public AnticheatStrategyResult checkCouponExchange(ConsumerParams consumer, GoodsParams goods, RequestParams request, BehaviorParams behavior) {
        Long consumerId = consumer.getConsumerId();
        Integer swipe = behavior.getSwipe();
        Long appId = consumer.getAppId();

        Integer gtype = goods.getGtype();
        Long gid = goods.getGid();

        // 判断没有手势，并且，是用户第一次兑换
        boolean ret = Objects.equals(SWIPE_NOT_EXIST, swipe) && isFirstExchange(consumerId);
        if (ret && check(appId, gtype, gid)) {
            Long debugId = doMatchProcess(consumer, goods, request);
            return new AnticheatStrategyResult(true, debugId);
        }
        return new AnticheatStrategyResult(false);
    }

    private boolean check(Long appId, Integer gtype, Long gid) {
        String key = getKey(appId, gtype, gid);
        long ecCount = redisAtomicClient.incrBy(key, 1,DateUtils.getToTomorrowSeconds(), TimeUnit.SECONDS);
        if (ecCount >= 5) {
            return true;
        }
        return false;
    }

    /**
     * 判断是否用户第一次兑换
     */
    private boolean isFirstExchange(Long consumerId) {
        ConsumerFootprintEntity footprint = consumerFootprintDao.findCouponByConsumerId(consumerId);
        if (footprint == null) {
            return true;
        }
        List<FootprintEntity> fpList = footprint.getColumnDataByType("coupon");
        return fpList.isEmpty();
    }

    private String getKey(Long appId, Integer gtype, Long gid) {
        return String.format("%s-first-out-mins-appId-%s-gtype-%s-gid-%s", CacheConstants.AC_ANTICHEAT_EXCHANGE, appId, gtype, gid);
    }

    private Long doMatchProcess(ConsumerParams consumer, GoodsParams goods, RequestParams request) {
        AnticheatDebugLogEntity debug = new AnticheatDebugLogEntity(true);
        debug.setAppId(consumer.getAppId());
        debug.setItemId(null);
        debug.setAppItemId(null);
        debug.setGtype(goods.getGtype());
        debug.setGid(goods.getGid());
        debug.setConsumerId(consumer.getConsumerId());
        debug.setPartnerUserId(consumer.getPartnerUserId());
        debug.setIp(request.getIp());
        debug.setStrategyType(AnticheatStrategyConfigEntity.TYPE_FIRST_NO_SWIPE);
        debug.setMessage(String.format("此用户 id(%s),没手势的用户第一笔兑换", consumer.getConsumerId()));

        anticheatDebugLogDao.insert(debug);

        return debug.getId();
    }
}
