package cn.com.duiba.apollo.center.api.service;

import cn.com.duiba.apollo.center.api.constants.ReservedKeys;
import cn.com.duiba.apollo.center.api.domain.dto.ResourceTemplateItemDto;
import cn.com.duiba.apollo.center.api.domain.params.KeyValue;
import cn.com.duiba.apollo.center.api.domain.params.ResourceRenderParams;
import cn.com.duiba.apollo.center.api.remoteservice.RemoteApolloTemplateService;
import cn.com.duiba.apollo.center.api.utils.RenderUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.regex.Matcher;

/**
 * @author liuyao
 */
@Slf4j
public class ResourceRenderService {

    private static final String NATIVE_RENDER_TEMPLATE = "apollo.%s.%s.%s";

    @Resource
    private RemoteApolloTemplateService remoteApolloTemplateService;

    public List<KeyValue> renderInstanceConfigByTemplate(ResourceRenderParams renderParams, List<KeyValue> instanceProperties){
        List<KeyValue> result = Lists.newArrayList();
        if(Objects.isNull(renderParams.getTemplateId())){
            return renderInstanceConfigByNativeTemplate(renderParams,instanceProperties);
        }
        List<ResourceTemplateItemDto> items = remoteApolloTemplateService.findTemplateItems(renderParams.getTemplateId());
        if(items.isEmpty()){
            return renderInstanceConfigByNativeTemplate(renderParams,instanceProperties);
        }
        Properties properties = new Properties();
        properties.put(ReservedKeys.TYPE_KEY,renderParams.getTypeKey());
        properties.put(ReservedKeys.INSTANCE_KEY,renderParams.getInstanceKey());
        //因为instanceKey可能被覆盖，所以原生配置中保留稳定的资源实例id
        properties.put(ReservedKeys.INSTANCE_ID,renderParams.getInstanceId().toString());
        Map<String,Boolean> passwordMap = Maps.newHashMap();
        for(KeyValue item:instanceProperties){
            properties.setProperty(item.getKey(), Optional.of(item.getValue()).orElse(""));
            passwordMap.put(item.getKey(),item.isPassword());
        }

        for(ResourceTemplateItemDto item:items){
            String key = replaceArgs(item.getKey(),properties);
            String value = replaceArgs(item.getValue(),properties);

            Set<String> dependenceKeys = RenderUtils.analysisTemplateDependenceKey(item.getValue());
            boolean password = false;
            for(String dependenceKey:dependenceKeys){
                if(passwordMap.getOrDefault(dependenceKey,false)){
                    password = true;
                }
            }
            KeyValue dto = new KeyValue();
            dto.setKey(key);
            dto.setValue(value);
            dto.setComment(item.getComment());
            dto.setPassword(password);
            result.add(dto);
        }
        return result;
    }

    private List<KeyValue> renderInstanceConfigByNativeTemplate(ResourceRenderParams renderParams, List<KeyValue> instanceProperties){
        for(KeyValue item:instanceProperties){
            String key = String.format(NATIVE_RENDER_TEMPLATE,renderParams.getTypeKey(),renderParams.getInstanceKey(),item.getKey());
            item.setKey(key);
        }
        return instanceProperties;
    }

    public static String replaceArgs(final String template, Properties properties){
        if(StringUtils.isBlank(template)){
            return template;
        }
        StringBuffer sb = new StringBuffer();
        try{
            Matcher matcher = RenderUtils.RENDER_PATTERN.matcher(template);
            while(matcher.find()){
                // 键名
                String name = matcher.group(1);
                // 键值
                String value = "";
                if(properties.containsKey(name)){
                    // 键值
                    value = properties.getProperty(name,"");
                    value = value.replaceAll("\\$", "\\\\\\$");
                }
                matcher.appendReplacement(sb, value);
            }
            matcher.appendTail(sb);
        }catch(Exception e){
            log.error("渲染失败",e);
        }
        return sb.toString();
    }

}
