package cn.com.duiba.apollo.client.rocketmq;

import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.*;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.common.message.MessageExt;
import org.apache.rocketmq.common.protocol.heartbeat.MessageModel;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;

@Slf4j
public class RocketMQListenerBindingContainer {

    @Resource
    private ConfigServerEventListener configServerEventListener;
    @Resource
    private ApolloClientRocketmqProperties apolloClientRocketmqProperties;

    private DefaultMQPushConsumer consumer;

    void destroy() {
        if (Objects.nonNull(consumer)) {
            consumer.shutdown();
        }
        log.info("container destroyed, {}", this.toString());
    }

    void init() throws Exception {


        ConsumeMode consumeMode = apolloClientRocketmqProperties.getOrderly()
                ? ConsumeMode.ORDERLY
                : ConsumeMode.CONCURRENTLY;

        MessageModel messageModel = apolloClientRocketmqProperties.getBroadcasting()
                ? MessageModel.BROADCASTING
                : MessageModel.CLUSTERING;

        consumer = new DefaultMQPushConsumer(apolloClientRocketmqProperties.getGroup());

        consumer.setNamesrvAddr(apolloClientRocketmqProperties.getNameServer());
        consumer.setConsumeThreadMax(apolloClientRocketmqProperties.getConcurrency());
        consumer.setConsumeThreadMin(apolloClientRocketmqProperties.getConcurrency());

        switch (messageModel) {
            case BROADCASTING:
                consumer.setMessageModel(
                        org.apache.rocketmq.common.protocol.heartbeat.MessageModel.BROADCASTING);
                break;
            case CLUSTERING:
                consumer.setMessageModel(
                        org.apache.rocketmq.common.protocol.heartbeat.MessageModel.CLUSTERING);
                break;
            default:
                throw new IllegalArgumentException("Property 'messageModel' was wrong.");
        }
        consumer.subscribe(apolloClientRocketmqProperties.getTopic(), null);

        switch (consumeMode) {
            case ORDERLY:
                consumer.setMessageListener(new DefaultMessageListenerOrderly());
                break;
            case CONCURRENTLY:
                consumer.setMessageListener(new DefaultMessageListenerConcurrently());
                break;
            default:
                throw new IllegalArgumentException("Property 'consumeMode' was wrong.");
        }
        try {
            consumer.start();
        }
        catch (MQClientException e) {
            throw new IllegalStateException("Failed to start RocketMQ push consumer", e);
        }
    }

    public class DefaultMessageListenerConcurrently
            implements MessageListenerConcurrently {

        @Override
        public ConsumeConcurrentlyStatus consumeMessage(List<MessageExt> msgs,ConsumeConcurrentlyContext context) {
            for (MessageExt messageExt : msgs) {
                try {
                    long now = System.currentTimeMillis();
                    configServerEventListener.onMessage(messageExt);
                    long costTime = System.currentTimeMillis() - now;
                    log.debug("topic:{} consume:{} cost:{} ms", messageExt.getTopic(),messageExt.getMsgId(), costTime);
                }
                catch (Exception e) {
                    log.error("consume message failed. messageExt:{}", messageExt, e);
                    return ConsumeConcurrentlyStatus.RECONSUME_LATER;
                }
            }

            return ConsumeConcurrentlyStatus.CONSUME_SUCCESS;
        }
    }

    public class DefaultMessageListenerOrderly implements MessageListenerOrderly {

        @Override
        public ConsumeOrderlyStatus consumeMessage(List<MessageExt> msgs,ConsumeOrderlyContext context) {
            for (MessageExt messageExt : msgs) {
                try {
                    long now = System.currentTimeMillis();
                    configServerEventListener.onMessage(messageExt);
                    long costTime = System.currentTimeMillis() - now;
                    log.debug("topic:{} consume:{} cost:{} ms", messageExt.getTopic(),messageExt.getMsgId(), costTime);
                }
                catch (Exception e) {
                    log.error("consume message failed. messageExt:{}", messageExt, e);
                    context.setSuspendCurrentQueueTimeMillis(1000);
                    return ConsumeOrderlyStatus.SUSPEND_CURRENT_QUEUE_A_MOMENT;
                }
            }

            return ConsumeOrderlyStatus.SUCCESS;
        }
    }


}
