package cn.com.duiba.application.boot.api.component.oauth2;

import cn.com.duiba.application.boot.api.domain.model.OauthInfo;
import cn.com.duiba.application.boot.api.domain.model.OauthToken;
import cn.com.duiba.wolf.utils.UUIDUtils;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Date;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

@Slf4j
public abstract class ApplicationBootOauth2Client {

    private String state = UUIDUtils.createSecureUUID();

    private volatile String accessToken;

    private LoadingCache<String, OauthInfo> authInfoCache = Caffeine.newBuilder().expireAfterWrite(30L,TimeUnit.MINUTES).build(new CacheLoader<String, OauthInfo>() {
        @Nullable
        public OauthInfo load(@Nonnull String accessToken) {
            return loadOauthInfo(accessToken);
        }
    });

    public String getAccessToken(){
        if(StringUtils.isBlank(accessToken)){
            refreshAccessToken();
        }
        OauthInfo oauthInfo = authInfoCache.get(accessToken);
        if(Objects.isNull(oauthInfo) || Objects.isNull(oauthInfo.getExpiration()) || new Date().after(oauthInfo.getExpiration())){
            refreshAccessToken();
            authInfoCache.refresh(accessToken);
        }
        return accessToken;
    }

    public OauthInfo getOauthInfo(String accessToken){
        OauthInfo oauthInfo = authInfoCache.get(accessToken);
        if(Objects.isNull(oauthInfo) || Objects.isNull(oauthInfo.getExpiration()) || new Date().after(oauthInfo.getExpiration())){
            authInfoCache.refresh(accessToken);
        }
        return authInfoCache.get(accessToken);
    }

    public void refreshOauthInfoCache(String accessToken){
        authInfoCache.refresh(accessToken);
    }

    private synchronized void refreshAccessToken(){
        try{
            if(StringUtils.isBlank(accessToken)){
                OauthToken oauthToken = loadOauthToken();
                accessToken = oauthToken.getAccessToken();
                return;
            }
            OauthInfo oauthInfo = loadOauthInfo(accessToken);
            if(Objects.isNull(oauthInfo) || Objects.isNull(oauthInfo.getExpiration()) || new Date().after(oauthInfo.getExpiration())){
                OauthToken oauthToken = loadOauthToken();
                accessToken = oauthToken.getAccessToken();
            }
        }catch (Exception e){
            log.error("获取accessToken失败",e);
            throw new RuntimeException(e);
        }
    }

    public abstract OauthToken loadOauthToken();

    public abstract OauthInfo loadOauthInfo(String accessToken);

    public String getState() {
        return state;
    }
}
