package cn.com.duiba.application.boot.api.component.stream.tee;

import cn.com.duiba.application.boot.api.ApplicationProperties;
import cn.com.duiba.application.boot.api.component.environment.ClusterKey;
import cn.com.duiba.application.boot.api.component.environment.Environment;
import cn.com.duiba.application.boot.api.component.environment.EnvironmentService;
import cn.com.duiba.application.boot.api.component.oauth2.ApplicationBootOauth2Client;
import cn.com.duiba.application.boot.stream.annotation.StreamListener;
import cn.com.duiba.application.boot.stream.support.StreamTemplate;
import cn.com.duiba.wolf.utils.UUIDUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.serializer.Hessian2Serializer;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.ApplicationContext;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.io.Serializable;
import java.util.Objects;
import java.util.Set;

@Slf4j
public class TotalEnvironmentalEventService {

    public static final String TEE_BINDING_NAME = "tee";

    @Resource
    private ApplicationContext applicationContext;
    @Resource
    private ApplicationBootOauth2Client applicationBootOauth2Client;
    @Resource
    private ApplicationProperties applicationProperties;
    @Resource
    private Environment currentEnvironment;
    @Resource
    private StreamTemplate streamTemplate;
    @Resource
    private EnvironmentService environmentService;

    public void publish(Object object){

        Objects.requireNonNull(object);

        if(!(object instanceof Serializable)){
            throw new RuntimeException("全环境消息["+object.getClass().getName()+"]未实现序列化接口");
        }

        TotalEnvironmentalEvent event = new TotalEnvironmentalEvent();

        charkSendLimit(object,currentEnvironment,applicationProperties.getApplicationName());

        event.setEventId(UUIDUtils.createUUID());
        event.setState(applicationBootOauth2Client.getState());
        event.setAccessToken(applicationBootOauth2Client.getAccessToken());
        event.setAppAlias(applicationProperties.getName());

        ClusterKey clusterKey = new ClusterKey();
        clusterKey.setCluster(environmentService.currentClusterKey());
        clusterKey.setEnvironment(currentEnvironment);
        event.setClusterKey(clusterKey);
        event.setClassName(object.getClass().getName());
        event.setPayloud(Hessian2Serializer.serialize(object));

        streamTemplate.send(TEE_BINDING_NAME,event);
    }

    //发布者来源于中继器
    @StreamListener(TEE_BINDING_NAME)
    void accept(TotalEnvironmentalEvent event){

        ClusterKey sourceClusterKey = event.getClusterKey();
        Assert.notNull(sourceClusterKey,"未能追寻事件源的环境");
        Assert.isTrue(StringUtils.isNotBlank(event.getAppAlias()),"未能追寻到源应用");

        if(Objects.isNull(event.getPayloud()) || StringUtils.isBlank(event.getClassName())){
            return;
        }
        try{
            Class.forName(event.getClassName());
        }catch (ClassNotFoundException e){
            log.debug("不支持类型["+event.getClassName()+"]消息的接收");
            return;
        }
        Object payloud = Hessian2Serializer.deserialize(event.getPayloud());
        try{
            charkSendLimit(payloud,sourceClusterKey.getEnvironment(),event.getAppAlias());
        }catch (Exception e){
            log.debug("无法接收",e);
            return;
        }
        applicationContext.publishEvent(payloud);
    }

    public void charkSendLimit(Object event,Environment environment,String appName){

        Assert.isTrue(environment!=null,"跨环境消息环境参数缺失");
        Assert.isTrue(StringUtils.isNotBlank(appName),"跨环境消息应用参数缺失");

        SendLimit sendLimit = AnnotationUtils.findAnnotation(event.getClass(),SendLimit.class);
        if(Objects.isNull(sendLimit)){
            return;
        }
        Set<String> appAlias = Sets.newHashSet(sendLimit.appAlias());
        if(!appAlias.isEmpty() && !appAlias.contains(appName)){
            throw new UnsupportedOperationException("当前系统不支持事件["+event.getClass().getName()+"]的发布");
        }
        Set<Environment> envs = Sets.newHashSet(sendLimit.environment());
        if(!envs.isEmpty() && !envs.contains(environment)){
            throw new UnsupportedOperationException("当前环境不支持事件["+event.getClass().getName()+"]的发布");
        }
    }

}
