/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cloud.stream.binder.rocketmq.integration;

import cn.com.duiba.application.boot.stream.binder.BinderMessageProducer;
import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.spring.core.RocketMQListener;
import org.springframework.cloud.stream.binder.rocketmq.consuming.RocketMQListenerBindingContainer;
import org.springframework.cloud.stream.binder.rocketmq.metrics.Instrumentation;
import org.springframework.cloud.stream.binder.rocketmq.metrics.InstrumentationManager;
import org.springframework.integration.endpoint.MessageProducerSupport;
import org.springframework.integration.support.MessageBuilder;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessagingException;
import org.springframework.retry.RetryCallback;
import org.springframework.retry.RetryContext;
import org.springframework.retry.RetryListener;

/**
 * @author <a href="mailto:fangjian0423@gmail.com">Jim</a>
 */
@Slf4j
public class RocketMQInboundChannelAdapter extends MessageProducerSupport implements BinderMessageProducer {

    private RocketMQListenerBindingContainer rocketMQListenerContainer;

    private final InstrumentationManager instrumentationManager;

    public RocketMQInboundChannelAdapter(
            RocketMQListenerBindingContainer rocketMQListenerContainer,
            InstrumentationManager instrumentationManager) {
        this.rocketMQListenerContainer = rocketMQListenerContainer;
        this.instrumentationManager = instrumentationManager;
    }

    @Override
    public void init() {
        super.onInit();

        BindingRocketMQListener listener = new BindingRocketMQListener();
        rocketMQListenerContainer.setRocketMQListener(listener);
        try {
            rocketMQListenerContainer.afterPropertiesSet();
        }
        catch (Exception e) {
            log.error("rocketMQListenerContainer init error: " + e.getMessage(), e);
            throw new IllegalArgumentException(
                    "rocketMQListenerContainer init error: " + e.getMessage(), e);
        }

        instrumentationManager.addHealthInstrumentation(
                new Instrumentation(rocketMQListenerContainer.getTopic()
                        + rocketMQListenerContainer.getConsumerGroup()));

        super.start();
    }

    @Override
    public void stopListener() {
        super.stop();
    }

    @Override
    protected void doStart() {
        try {
            rocketMQListenerContainer.start();
            instrumentationManager
                    .getHealthInstrumentation(rocketMQListenerContainer.getTopic()
                            + rocketMQListenerContainer.getConsumerGroup())
                    .markStartedSuccessfully();
        }
        catch (Exception e) {
            instrumentationManager
                    .getHealthInstrumentation(rocketMQListenerContainer.getTopic()
                            + rocketMQListenerContainer.getConsumerGroup())
                    .markStartFailed(e);
            log.error("RocketMQTemplate startup failed, Caused by " + e.getMessage());
            throw new MessagingException(MessageBuilder.withPayload(
                    "RocketMQTemplate startup failed, Caused by " + e.getMessage())
                    .build(), e);
        }
    }

    @Override
    protected void doStop() {
        rocketMQListenerContainer.stop();
    }


    protected class BindingRocketMQListener implements RocketMQListener<Message>, RetryListener {

        @Override
        public void onMessage(Message message) {
            RocketMQInboundChannelAdapter.this.sendMessage(message);
        }

        @Override
        public <T, E extends Throwable> boolean open(RetryContext context,
                                                     RetryCallback<T, E> callback) {
            return true;
        }

        @Override
        public <T, E extends Throwable> void close(RetryContext context,
                                                   RetryCallback<T, E> callback, Throwable throwable) {
        }

        @Override
        public <T, E extends Throwable> void onError(RetryContext context,
                                                     RetryCallback<T, E> callback, Throwable throwable) {

        }
    }

}
