/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cloud.stream.binder.rocketmq;

import cn.com.duiba.application.boot.stream.binder.Binder;
import cn.com.duiba.application.boot.stream.binder.BinderMessageHandler;
import cn.com.duiba.application.boot.stream.binder.BinderMessageProducer;
import cn.com.duiba.application.boot.stream.config.BindingProperties;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.apache.rocketmq.spring.core.RocketMQTemplate;
import org.springframework.cloud.stream.binder.rocketmq.consuming.RocketMQListenerBindingContainer;
import org.springframework.cloud.stream.binder.rocketmq.integration.RocketMQInboundChannelAdapter;
import org.springframework.cloud.stream.binder.rocketmq.integration.RocketMQMessageHandler;
import org.springframework.cloud.stream.binder.rocketmq.metrics.InstrumentationManager;
import org.springframework.cloud.stream.binder.rocketmq.properties.RocketMQConsumerProperties;
import org.springframework.cloud.stream.binder.rocketmq.properties.RocketMQProducerProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.util.StringUtils;

/**
 * @author <a href="mailto:fangjian0423@gmail.com">Jim</a>
 */
public class RocketMQMessageChannelBinder implements Binder {

    private InstrumentationManager instrumentationManager;

    private ApplicationContext applicationContext;

    private ObjectMapper objectMapper;

    @Override
    public BinderMessageHandler createProducerMessageHandler(BindingProperties bindingProperties) {

        String producerGroup = bindingProperties.getProducer().getGroup();
        if (StringUtils.isEmpty(producerGroup)) {
            throw new RuntimeException("'group must be configured for channel");
        }

        RocketMQProducerProperties producerProperties = bindingProperties.getProducer().bindProperties(RocketMQProducerProperties.class);

        RocketMQTemplate rocketMQTemplate = new RocketMQTemplate();

        rocketMQTemplate.setObjectMapper(objectMapper);

        DefaultMQProducer producer = new DefaultMQProducer(producerGroup);
        producer.setVipChannelEnabled(producerProperties.getVipChannelEnabled());
        producer.setNamesrvAddr(producerProperties.getNameServer());
        producer.setSendMsgTimeout(producerProperties.getSendMessageTimeout());
        producer.setRetryTimesWhenSendFailed(producerProperties.getRetryTimesWhenSendFailed());
        producer.setRetryTimesWhenSendAsyncFailed(producerProperties.getRetryTimesWhenSendAsyncFailed());
        producer.setCompressMsgBodyOverHowmuch(producerProperties.getCompressMessageBodyThreshold());
        producer.setRetryAnotherBrokerWhenNotStoreOK(producerProperties.isRetryNextServer());
        producer.setMaxMessageSize(producerProperties.getMaxMessageSize());
        rocketMQTemplate.setProducer(producer);

        RocketMQMessageHandler messageHandler = new RocketMQMessageHandler(
                rocketMQTemplate,producerGroup,
                instrumentationManager);
        messageHandler.setBeanFactory(applicationContext);
        messageHandler.setSync(producerProperties.getSync());
        return messageHandler;
    }

    @Override
    public BinderMessageProducer createConsumerEndpoint(BindingProperties bindingProperties) {

        String group = bindingProperties.getConsumer().getGroup();
        if (group == null || "".equals(group)) {
            throw new RuntimeException("'group must be configured for channel ");
        }
        String topic = bindingProperties.getConsumer().getTopic();

        RocketMQConsumerProperties consumerProperties = bindingProperties.getConsumer().bindProperties(RocketMQConsumerProperties.class);

        RocketMQListenerBindingContainer listenerContainer = new RocketMQListenerBindingContainer( consumerProperties);
        listenerContainer.setConsumerGroup(group);
        listenerContainer.setTopic(topic);
        listenerContainer.setConcurrency(bindingProperties.getConsumer().getConcurrency());
        listenerContainer.setSuspendCurrentQueueTimeMillis(consumerProperties.getSuspendCurrentQueueTimeMillis());
        listenerContainer.setDelayLevelWhenNextConsume(consumerProperties.getDelayLevelWhenNextConsume());
        listenerContainer.setNameServer(consumerProperties.getNameServer());

        return new RocketMQInboundChannelAdapter(listenerContainer, instrumentationManager);

    }


    public void setInstrumentationManager(InstrumentationManager instrumentationManager) {
        this.instrumentationManager = instrumentationManager;
    }

    public void setApplicationContext(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    public void setObjectMapper(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }
}