package cn.com.duiba.application.boot.stream.binding;

import cn.com.duiba.application.boot.stream.annotation.StreamListener;
import cn.com.duiba.application.boot.stream.channel.ChannelFactoryBean;
import cn.com.duiba.application.boot.stream.channel.ChannelKey;
import cn.com.duiba.application.boot.stream.channel.ChannelType;
import cn.com.duiba.application.boot.stream.config.SpringIntegrationProperties;
import org.apache.commons.lang3.StringUtils;
import org.springframework.aop.framework.Advised;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.SmartInitializingSingleton;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.expression.EvaluationContext;
import org.springframework.integration.context.IntegrationContextUtils;
import org.springframework.integration.handler.AbstractReplyProducingMessageHandler;
import org.springframework.messaging.MessageChannel;
import org.springframework.messaging.SubscribableChannel;
import org.springframework.messaging.core.DestinationResolver;
import org.springframework.messaging.handler.annotation.support.MessageHandlerMethodFactory;
import org.springframework.messaging.handler.invocation.InvocableHandlerMethod;
import org.springframework.util.Assert;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.util.ReflectionUtils;

import javax.annotation.Resource;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class StreamListenerAnnotationBeanPostProcessor implements BeanPostProcessor,SmartInitializingSingleton,ApplicationContextAware {

    private ConfigurableApplicationContext applicationContext;

    @Resource(name = IntegrationContextUtils.LIST_MESSAGE_HANDLER_FACTORY_BEAN_NAME)
    private MessageHandlerMethodFactory messageHandlerMethodFactory;
    @Resource
    private SpringIntegrationProperties springIntegrationProperties;

    // @checkstyle:off
    private final MultiValueMap<String, StreamListenerHandlerMethodMapping> mappedListenerMethods = new LinkedMultiValueMap<>();

    @Override
    public final Object postProcessAfterInitialization(Object bean, final String beanName) throws BeansException {
        Class<?> targetClass = AopUtils.isAopProxy(bean) ? AopUtils.getTargetClass(bean) : bean.getClass();
        if(!checkPackage(targetClass.getPackage())){
            return bean;
        }

        Method[] uniqueDeclaredMethods = ReflectionUtils.getUniqueDeclaredMethods(targetClass);
        for (Method method : uniqueDeclaredMethods) {
            StreamListener streamListener = AnnotatedElementUtils.findMergedAnnotation(method, StreamListener.class);
            if (streamListener != null && !method.isBridge()) {
                String bindingName = streamListener.value();
                if(StringUtils.isBlank(bindingName)){
                    continue;
                }
                mappedListenerMethods.add(bindingName,new StreamListenerHandlerMethodMapping(bean,method,bindingName));
            }
        }
        return bean;
    }


    @Override
    public void afterSingletonsInstantiated() {

        EvaluationContext evaluationContext = IntegrationContextUtils.getEvaluationContext(this.applicationContext.getBeanFactory());

        for (Map.Entry<String, List<StreamListenerHandlerMethodMapping>> mappedBindingEntry : this.mappedListenerMethods.entrySet()) {

            ArrayList<DispatchingStreamListenerMessageHandler.ConditionalStreamListenerMessageHandlerWrapper> handlers = new ArrayList<>();

            for (StreamListenerHandlerMethodMapping mapping : mappedBindingEntry.getValue()) {

                InvocableHandlerMethod invocableHandlerMethod = this.messageHandlerMethodFactory.createInvocableHandlerMethod(mapping.getTargetBean(),checkProxy(mapping.getMethod(), mapping.getTargetBean()));

                StreamListenerMessageHandler streamListenerMessageHandler = new StreamListenerMessageHandler(
                        invocableHandlerMethod,true,
                        this.springIntegrationProperties.getMessageHandlerNotPropagatedHeaders());

                streamListenerMessageHandler.setApplicationContext(this.applicationContext);
                streamListenerMessageHandler.setBeanFactory(this.applicationContext.getBeanFactory());
                if (StringUtils.isNotBlank(mapping.getOutputChannel())) {
                    streamListenerMessageHandler.setOutputChannelName(mapping.getOutputChannel());
                }
                streamListenerMessageHandler.afterPropertiesSet();
                handlers.add(new DispatchingStreamListenerMessageHandler.ConditionalStreamListenerMessageHandlerWrapper(
                                null, streamListenerMessageHandler));
            }
            if (handlers.size() > 1) {
                for (DispatchingStreamListenerMessageHandler.ConditionalStreamListenerMessageHandlerWrapper handler : handlers) {
                    Assert.isTrue(handler.isVoid(),
                            StreamListenerErrorMessages.MULTIPLE_VALUE_RETURNING_METHODS);
                }
            }
            AbstractReplyProducingMessageHandler handler;

            if (handlers.size() > 1 || handlers.get(0).getCondition() != null) {
                handler = new DispatchingStreamListenerMessageHandler(handlers,evaluationContext);
            }
            else {
                handler = handlers.get(0).getStreamListenerMessageHandler();
            }
            handler.setBeanFactory(this.applicationContext);
            handler.setApplicationContext(this.applicationContext);
            //handler.setChannelResolver(this.binderAwareChannelResolver);
            handler.afterPropertiesSet();
            this.applicationContext.getBeanFactory().registerSingleton(handler.getClass().getSimpleName() + handler.hashCode(), handler);

            ChannelKey channelKey = new ChannelKey();
            channelKey.setBindingName(mappedBindingEntry.getKey());
            channelKey.setChannelType(ChannelType.INPUT);

            String subscribableChannelBeanName = ChannelFactoryBean.getBeanNameByBindingName(channelKey);

            this.applicationContext.getBean(subscribableChannelBeanName, SubscribableChannel.class).subscribe(handler);//把方法回调注册到监听器
        }
        this.mappedListenerMethods.clear();

        //bindingService.createBinding();//注册完毕后，开启所有的监听
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = (ConfigurableApplicationContext) applicationContext;
    }

    private Method checkProxy(Method methodArg, Object bean) {
        Method method = methodArg;
        if (AopUtils.isJdkDynamicProxy(bean)) {
            try {
                // Found a @StreamListener method on the target class for this JDK proxy
                // ->
                // is it also present on the proxy itself?
                method = bean.getClass().getMethod(method.getName(),
                        method.getParameterTypes());
                Class<?>[] proxiedInterfaces = ((Advised) bean).getProxiedInterfaces();
                for (Class<?> iface : proxiedInterfaces) {
                    try {
                        method = iface.getMethod(method.getName(),
                                method.getParameterTypes());
                        break;
                    }
                    catch (NoSuchMethodException noMethod) {
                    }
                }
            }
            catch (SecurityException ex) {
                ReflectionUtils.handleReflectionException(ex);
            }
            catch (NoSuchMethodException ex) {
                throw new IllegalStateException(String.format(
                        "@StreamListener method '%s' found on bean target class '%s', "
                                + "but not found in any interface(s) for bean JDK proxy. Either "
                                + "pull the method up to an interface or switch to subclass (CGLIB) "
                                + "proxies by setting proxy-target-class/proxyTargetClass attribute to 'true'",
                        method.getName(), method.getDeclaringClass().getSimpleName()),
                        ex);
            }
        }
        return method;
    }

    private boolean checkPackage(Package pkg){
        if(pkg==null){
            return false;
        }
        String packageName = pkg.getName();
        for(String whitePackage: StreamListenerScanRegistrar.getScanPackages()){
            if(StringUtils.startsWith(packageName,whitePackage)){
                return true;
            }
        }
        return false;
    }


}

