package cn.com.duiba.application.boot.stream.configuration;

import cn.com.duiba.application.boot.stream.binder.BinderFactoryBean;
import cn.com.duiba.application.boot.stream.binder.BinderType;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.io.Resource;
import org.springframework.core.io.UrlResource;
import org.springframework.core.io.support.PropertiesLoaderUtils;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.net.URL;
import java.util.*;

@Slf4j
public class BinderTypeRegistry implements BeanDefinitionRegistryPostProcessor,ApplicationContextAware {

    private Map<String, BinderType> binderTypes = new HashMap<>();

    private Map<String,String> beanNameMap = Maps.newHashMap();

    private ConfigurableApplicationContext configurableApplicationContext;

    @Override
    public void postProcessBeanDefinitionRegistry(BeanDefinitionRegistry registry) throws BeansException {
        scanBinderTypes();//扫描所有的注册类
        reflectionBinderBuilderBeanName(registry);//注册FactoryBean
    }

    private void scanBinderTypes() {
        ClassLoader classLoader = configurableApplicationContext.getClassLoader();
        try {
            Enumeration<URL> resources = Objects.requireNonNull(classLoader).getResources("META-INF/stream.binders");
            if ((resources == null || !resources.hasMoreElements())) {
                log.debug("Failed to locate 'META-INF/stream.binders' resources on the classpath."
                        + " Assuming standard boot 'META-INF/spring.factories' configuration is used");
                return;
            }
            while (resources.hasMoreElements()) {
                URL url = resources.nextElement();
                UrlResource resource = new UrlResource(url);
                for (BinderType binderType : parseBinderConfigurations(classLoader,
                        resource)) {
                    binderTypes.put(binderType.getName(), binderType);
                }
            }
        } catch (IOException | ClassNotFoundException e) {
            throw new BeanCreationException("Cannot create binder factory:", e);

        }
    }

    private void reflectionBinderBuilderBeanName(BeanDefinitionRegistry registry) {

        for(Map.Entry<String, BinderType> entry:binderTypes.entrySet()){
            BinderType binderType = entry.getValue();
            String binderTypeName = entry.getKey();
            Class<?> binderFactoryBeanClass = binderType.getBinderFactoryBeanClass();
            if(Objects.isNull(binderFactoryBeanClass)){
                continue;
            }
            BeanDefinitionBuilder builder = BeanDefinitionBuilder.rootBeanDefinition(binderFactoryBeanClass);
            builder.addPropertyValue("binderName",binderTypeName);
            String beanName = binderTypeName+"Biner";
            registry.registerBeanDefinition(beanName,builder.getBeanDefinition());
            beanNameMap.put(binderTypeName,beanName);
        }
    }

    private static Collection<BinderType> parseBinderConfigurations(ClassLoader classLoader,
                                                                    Resource resource) throws IOException, ClassNotFoundException {
        Properties properties = PropertiesLoaderUtils.loadProperties(resource);
        Collection<BinderType> parsedBinderConfigurations = new ArrayList<>();
        for (Map.Entry<?, ?> entry : properties.entrySet()) {
            String binderType = (String) entry.getKey();
            String binderFactoryBeanClassName = (String)entry.getValue();
            if(StringUtils.isEmpty(binderFactoryBeanClassName)){
                continue;
            }
            Class binderFactoryClassName = ClassUtils.forName(binderFactoryBeanClassName, classLoader);
            if(!BinderFactoryBean.class.isAssignableFrom(binderFactoryClassName)){
                throw new RuntimeException("binder的提供者必须实现BinderFactoryBean接口");
            }
            parsedBinderConfigurations.add(new BinderType(binderType, binderFactoryClassName));
        }
        return parsedBinderConfigurations;
    }

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        //nothing
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        Assert.isInstanceOf(ConfigurableApplicationContext.class, applicationContext);
        this.configurableApplicationContext = (ConfigurableApplicationContext) applicationContext;
    }

    public String findBeanNameByBinderType(String binderType){
        if(!beanNameMap.containsKey(binderType)){
            throw new IllegalArgumentException("没有Binder["+binderType+"]的提供者");
        }
        return beanNameMap.get(binderType);
    }


}
