package cn.com.duibabiz.component.filters.bloom.daily;

import cn.com.duiba.wolf.entity.Pair;
import com.alibaba.fastjson.JSON;
import org.springframework.util.Assert;

import java.time.LocalDate;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Set;

/**
 * 按日分组
 * biz_tool_K164_p30b33226+20201219
 *
 * @author zengsp
 */
public class DailyKeyUtil {

    /**
     * 格式：20111203
     */
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.BASIC_ISO_DATE;

    /**
     * 获取今天的
     */
    public static String getTodayKey(String userKey) {
        return combineKey(userKey, LocalDate.now());
    }

    private static String combineKey(String userKey, LocalDate localDate) {
        return userKey + localDate.format(DATE_TIME_FORMATTER);
    }

    public static Pair<String, LocalDate> splitKey(String key) {
        int formatterLength = 8;
        return Pair.from(key.substring(0, key.length() - formatterLength),
                LocalDate.parse(key.substring(key.length() - formatterLength), DATE_TIME_FORMATTER));
    }

    /**
     * 为key加上日期后缀
     *
     * @param recentlyDays 1-今天，2-今天+昨天
     */
    public static Set<String> getRecentlyKeys(String userKey, int recentlyDays) {
        final LocalDate now = LocalDate.now();
        Set<String> set = new LinkedHashSet<>();
        for (int i = 0; i < recentlyDays; i++) {
            set.add(combineKey(userKey, now.minusDays(i)));
        }
        return set;
    }


    /**
     * 获取过期时间，包括今天
     * 1-今天过期
     * 2-明天过期
     */
    public static Date getExpireAt(Integer recordKeepDays) {
        recordKeepDays = Math.max(recordKeepDays, 1);
        return new Date(LocalDate.now()
                .plusDays(recordKeepDays - 1)
                .atTime(23, 59, 59, 59)
                .toInstant(ZoneOffset.of("+8")).toEpochMilli());
    }


    public static void main(String[] args) {
        String userKey = "userKey";
        String todayKey = getTodayKey(userKey);
        System.out.println(todayKey);
        Pair<String, LocalDate> pair = splitKey(todayKey);
        Assert.isTrue(Objects.equals(userKey, pair.getKey()), "key 拆分错误");
        Assert.isTrue(LocalDate.now().isEqual(pair.getValue()), "key 拆分错误");


        Set<String> recentlyKeys = getRecentlyKeys(userKey, 2);
        System.out.println(JSON.toJSONString(recentlyKeys));
        String[] strings = recentlyKeys.toArray(new String[0]);
        for (int i = 0; i < strings.length; i++) {
            String recentlyKey = strings[i];
            Pair<String, LocalDate> splitKey = splitKey(recentlyKey);
            Assert.isTrue(Objects.equals(userKey, splitKey.getKey()), "key 拆分错误");

            Assert.isTrue(LocalDate.now().minusDays(i).isEqual(splitKey.getValue()), "key 拆分错误");
        }

        Date expireAt = getExpireAt(1);
        System.out.println(expireAt.getTime());
        Assert.isTrue(expireAt.getTime() == LocalDate.now().atTime(23, 59, 59, 59)
                .toInstant(ZoneOffset.of("+8")).toEpochMilli(), "key 拆分错误");


        expireAt = getExpireAt(2);
        System.out.println(expireAt.getTime());
        Assert.isTrue(expireAt.getTime() == LocalDate.now().plusDays(1).atTime(23, 59, 59, 59)
                .toInstant(ZoneOffset.of("+8")).toEpochMilli(), "key 拆分错误");
    }
}
