package cn.com.duiba.biz.tool.duiba.client;

import cn.com.duiba.biz.tool.duiba.dto.ConsumerCookieDto;
import cn.com.duiba.consumer.center.api.dto.ConsumerDto;
import cn.com.duiba.consumer.center.api.remoteservice.RemoteConsumerService;
import cn.com.duiba.developer.center.api.domain.dto.AppSimpleDto;
import cn.com.duiba.developer.center.api.remoteservice.RemoteAppService;
import cn.com.duiba.idmaker.service.api.dto.kms.TimeBasedRollingKeyDto;
import cn.com.duiba.idmaker.service.api.remoteservice.kms.RemoteKmsService;
import cn.com.duiba.wolf.perf.timeprofile.RequestTool;
import cn.com.duiba.wolf.utils.BlowfishUtils;
import cn.com.duiba.wolf.utils.NumberUtils;
import cn.com.duiba.wolf.utils.UUIDUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Optional;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.io.BaseEncoding;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.aspectj.weaver.ast.Var;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.nio.charset.Charset;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

/**
 * 使用方法，在spring配置文件中声明即可:<bean id="requestLocal" class="cn.com.duiba.biz.tool.duiba.client.RequestLocal"/>
 * 实例化时会自动寻找RemoteAppService、RemoteConsumerService并注入，找不到则报错要求声明这两个bean
 *
 * Created by wenqi.huang on 2017/5/12.
 */
public class RequestLocal implements ApplicationContextAware, InitializingBean{

    private static final Logger logger = LoggerFactory.getLogger(RequestLocal.class);

    public static final String ADSLOTID = "adslotId";

    private ApplicationContext applicationContext;

    private static RemoteConsumerService remoteConsumerService;
    private static RemoteAppService remoteAppService;
    private static Object remoteAppDirectService;//实际上是RemoteAppDirectService的实例，由于麦拉项目没有引入activity-center的jar包会报错哦，所以用反射来调度
    private static DuibaConsumerCookieClient duibaConsumerCookieClient;

    //是否需要登录功能（如果禁用的话也会禁用商业流量的自动登录功能）
    private static boolean needLogin = true;

    //这个是用于兼容老的固定密钥加解密逻辑，以及商业流量
    private static String oldConsumerEncryptKey;
    //cookie的domain
    private static String cookieDomain;

    //appKey到appId的缓存
    private static final LoadingCache<String, Optional<Long>> appKey2IdCache;
    //appId到是否商业app的缓存
    private static final LoadingCache<Long, Boolean> appId2DirectCache;
    //appId到是否禁用的缓存
    private static final LoadingCache<Long, Boolean> appId2ForbidStatusCache;

    private static final ThreadLocal<HttpRequestDto> local = new ThreadLocal<>();

    //app被禁止访问的html代码
    private static final String forbiddenPageHtml;

    private static final String YJQ_PATTERN = "^26851.activity.*.m.duiba.com.cn";
    private static final String LZLJ_PATTERN = "index";

    static{
        appKey2IdCache = CacheBuilder.newBuilder().concurrencyLevel(32).softValues().maximumSize(100000)
                .refreshAfterWrite(8, TimeUnit.MINUTES)
                .expireAfterWrite(10, TimeUnit.MINUTES)
                .build(new CacheLoader<String, Optional<Long>>() {
                    @Override
                    public Optional<Long> load(String key) throws Exception {
                        AppSimpleDto app = remoteAppService.getAppByAppKey(key).getResult();
                        if(app != null){
                            return Optional.of(app.getId());
                        }
                        return Optional.absent();
                    }
                });
        appId2DirectCache = CacheBuilder.newBuilder().concurrencyLevel(32).softValues().maximumSize(100000)
                .refreshAfterWrite(25, TimeUnit.MINUTES)//25分钟后异步刷新
                .expireAfterWrite(30, TimeUnit.MINUTES)//30分钟后失效
                .build(new CacheLoader<Long, Boolean>() {
                    @Override
                    public Boolean load(Long key) throws Exception {
                        Method m = remoteAppDirectService.getClass().getDeclaredMethod("findByAppId", Long.class);
                        Long id = (Long)m.invoke(remoteAppDirectService, key);
                        if(id != null){
                            return true;
                        }
                        return false;
                    }
                });

        appId2ForbidStatusCache = CacheBuilder.newBuilder().concurrencyLevel(32).softValues().maximumSize(100000)
                .refreshAfterWrite(2, TimeUnit.MINUTES)//2分钟后异步刷新
                .expireAfterWrite(3, TimeUnit.MINUTES)//3分钟后失效
                .build(new CacheLoader<Long, Boolean>() {
                    @Override
                    public Boolean load(Long key) throws Exception {
                        AppSimpleDto app = remoteAppService.getSimpleApp(key).getResult();
                        if(app != null){
                            return app.isAppSwitch(AppSimpleDto.SwitchForbidApp);
                        }
                        return false;
                    }
                });

        try(InputStream in = RequestLocal.class.getResourceAsStream("/bt_common_html/no_permission.html")) {
            forbiddenPageHtml = IOUtils.toString(in);
        }catch(IOException e){
            throw new RuntimeException(e);
        }
    }

    public void setNeedLogin(boolean needLogin) {
        RequestLocal.needLogin = needLogin;
    }

    private static class HttpRequestDto {

        private ConsumerDto consumerDO;
        private AppSimpleDto consumerAppDO;
        private ConsumerCookieDto consumerCookieDto;//cookie中直接获取的信息
        private String tokenId;
        private String deap;//防作弊用
        private HttpServletRequest request;
        private HttpServletResponse response;

        /**
         * 尝试从request中恢复用户信息，并限流
         * @param request
         * @param response
         * @return true表示此次请求被限流，用户会看到繁忙页面；false表示此次请求被放行。
         * @throws IOException
         * @throws ServletException
         */
        public boolean initAndLimitQps(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
            this.request = request;
            this.response = response;

            ConsumerCookieDto consumerFromWdata4 = null;
            ConsumerCookieDto consumerFromWdata3 = null;
            //先尝试从wdata4中获取用户信息
            try {
                consumerFromWdata4 = duibaConsumerCookieClient.getConsumerCookieDto(request);
            }catch(Exception e){
                logger.info("get user from wdata4 error,will try to use wdata3.", e);
            }

            String appKey=request.getParameter("appKey");
            boolean openBs="openbs".equals(request.getParameter("openBs"));//商业活动入口需要永久cookie
            //如果取不到用户信息，则尝试从wdata3中获取用户信息
            if(consumerFromWdata4 == null){
                boolean preferCookieForEver = false;
                //签到日历且没有通过免登陆进入积分商城则按商业流量处理，设置cookie是永久的
                consumerFromWdata3 = parseOldWdata3(preferCookieForEver);
            }


            //TODO 对于之前的商业流量用户，由于wdata3是永久的，所以不会有tokenId，后续考虑强制设置tokenId,然后把wdata3改为httponly
            this.tokenId = RequestTool.getCookie(request, DuibaConsumerCookieClient.TOKEN_ID_COOKIE);
            //TODO to be deleted start
            if(this.tokenId == null || this.tokenId.isEmpty()){
                this.tokenId = RequestTool.getCookie(request, "wdata3");
            }
            //TODO to be deleted end

            //如果是商业活动，则可能需要重新植入用户cookie
            boolean preferUseWdata4 = false;
            if(null != appKey && needLogin && openBs){
                Optional<Long> appIdOptional = appKey2IdCache.getUnchecked(appKey);
                Long appId = appIdOptional.isPresent() ? appIdOptional.get() : null;
                //判断是否商业活动
                if(null!=appId && (openBs && isAppDirectByAppId(appId))){
                    this.consumerCookieDto = consumerFromWdata3;
                    removeWdata4Cookie();//如果是商业流量，需要确保移除wdata4 cookie，只保留wdata3，以防下次访问其他页面使用了wdata4 cookie
                    if((consumerCookieDto != null && !appId.equals(consumerCookieDto.getAppId()))
                            || consumerCookieDto == null){
                        makeCommercialUser(appId);//创建随机商业流量用户并登录
                    }
                }else{
                    preferUseWdata4 = true;
                }
            }else{
                preferUseWdata4 = true;
            }
            if(preferUseWdata4){
                this.consumerCookieDto = consumerFromWdata4;
            }

            // 获取请求的URI
            String requestURI = request.getRequestURI();
            // 判断URI中是否包含"index"
            boolean containsIndex = requestURI.contains(LZLJ_PATTERN);
            ConsumerCookieDto lzljInfoCookieDto = duibaConsumerCookieClient.getLzljConsumerCookieDto(request);
            if(lzljInfoCookieDto != null && containsIndex){
                this.consumerCookieDto = lzljInfoCookieDto;
                String lzljInfo = RequestTool.getCookie(request, "lzlj_info");
                Cookie[] cookies = request.getCookies();
                if(cookies != null){
                    for (Cookie c : cookies) {
                        if("wdata4".equals(c.getName())){
                            c.setMaxAge(0);
                            response.addCookie(c);
//                            String requestDomain = duibaConsumerCookieClient.getRequestDomain(request, cookieDomain);
                            Cookie cookie = new Cookie("wdata4", lzljInfo);
                            cookie.setPath("/");
                            cookie.setDomain(c.getDomain());
                            cookie.setMaxAge(24*60*60);
                            cookie.setHttpOnly(true);
                            response.addCookie(cookie);
                        }
                    }
                }
            }

            //判断如果app禁用了，则打印禁用页面
            Long appId = null;
            if(this.consumerCookieDto != null){
                appId = this.consumerCookieDto.getAppId();
            }else if(appKey != null){
                Optional<Long> appIdOptional = appKey2IdCache.getUnchecked(appKey);
                appId = appIdOptional.isPresent() ? appIdOptional.get() : null;
            }
            if(appId != null && isAppForbidden(appId)){
                showForbiddenPage(response);
                return true;
            }
            return false;
        }

        //移除wdata4 cookie
        private void removeWdata4Cookie(){
            Cookie[] cookies = this.request.getCookies();
            if(cookies == null || cookies.length == 0){
                return;
            }
            for(Cookie cookie : cookies){
                if(cookie.getName().equals(DuibaConsumerCookieClient.CONSUMER_WDATA4_COOKIE)
                        || cookie.getName().equals(DuibaConsumerCookieClient.LOGIN_TIME_COOKIE)){
                    cookie.setValue("");
                    cookie.setMaxAge(0);
                    cookie.setDomain(cookieDomain);
                    cookie.setPath("/");
                    cookie.setHttpOnly(true);
                    response.addCookie(cookie);
                }
            }
        }

        //创建商业流量用户并植入cookie
        private void makeCommercialUser(Long appId){
            String uid = "gen_" + UUIDUtils.createUUID();
            ConsumerDto c = new ConsumerDto(true);
            c.setAppId(appId);
            c.setPartnerUserId(uid);

            ConsumerDto temp = remoteConsumerService.insert(c);
            c.setId(temp.getId());

            RequestLocal.injectConsumerInfoIntoCookie(c, true);
        }

        //判断是否商业流量app
        private Boolean isAppDirectByAppId(Long appId) {
            if(appId == null){
                return false;
            }
            try {
                return appId2DirectCache.get(appId);
            } catch (ExecutionException e) {
                throw new RuntimeException(e);
            }
        }

        //判断app是否禁用了
        private Boolean isAppForbidden(Long appId) {
            try {
                return appId2ForbidStatusCache.get(appId);
            } catch (ExecutionException e) {
                throw new RuntimeException(e);
            }
        }

        //解析旧的用户cookie数据，或者商业流量数据
        private ConsumerCookieDto parseOldWdata3(boolean preferCookieForEver){
            String wdata3 = RequestTool.getCookie(request, "wdata3");
            if(wdata3 == null || wdata3.isEmpty()){
                return null;
            }
            String content = null;
            ConsumerCookieDto c;
            try {
                content = BlowfishUtils.decryptBlowfish(wdata3, oldConsumerEncryptKey);
                c = JSONObject.parseObject(content, ConsumerCookieDto.class);
            }catch(Exception e){
                logger.warn("wdata3:{}, content:{}", wdata3, content, e);
                return null;
            }
            //如果是商业流量，则cookie永久有效,仍然使用wdata3
            if(c.getAppId() == null){
                //容错代码，很久之前的商业流量用户的cookie中没有appId，查询出来并设置回cookie中
                logger.info("appId in wdata3 is null, wdata3:{},content:{}，set back", wdata3, content);
                if(c.getCid() != null) {
                    ConsumerDto consumer = remoteConsumerService.find(c.getCid());
                    if (consumer != null) {
                        c.setAppId(consumer.getAppId());
                        if(needLogin && isAppDirectByAppId(c.getAppId())){
                            c.setForEver(true);
                            injectConsumerInfoIntoCookie(consumer, true);
                        }
                    }
                }
            }
            if(c.isForEver()){
                return c;
            }else if(preferCookieForEver || isAppDirectByAppId(c.getAppId())){
                if(!c.isForEver()) {//重新设置cookie标记为永久
                    c.setForEver(true);
                    ConsumerDto consumer = new ConsumerDto();
                    consumer.setAppId(c.getAppId());
                    consumer.setPartnerUserId(c.getPartnerUserId());
                    consumer.setId(c.getCid());
                    if(needLogin) {
                        injectConsumerInfoIntoCookie(consumer, true);
                    }
                }
                return c;
            }else {//TODO 这个else分支等到下次发布需要删除，非商业流量只应该使用wdata4 cookie,目前存在只是为了兼容
                //非商业流量，cookie24小时内有效
                long now = System.currentTimeMillis();
                if (c.getTime() > now - 86400000) {
                    return c;
                }
            }
            return null;
        }

        //延迟初始化
        public ConsumerDto getConsumerDO(){
            if(consumerDO == null){
                if(consumerCookieDto == null){//如果cookie中没有有效的用户信息，则直接返回
                    return null;
                }else{
                    //返回经过代理的对象，获取appId/id/partnerUserId字段时不会读取数据库，获取其他字段才会从数据库加载
                    ConsumerDto temp = new ConsumerDto();
                    temp.setAppId(consumerCookieDto.getAppId());
                    temp.setId(consumerCookieDto.getCid());
                    temp.setPartnerUserId(consumerCookieDto.getPartnerUserId());

                    ProxyFactory proxyFactory = new ProxyFactory();
                    proxyFactory.setTarget(temp);
                    proxyFactory.addAdvice(new ConsumerMethodInterceptor(this));
                    consumerDO = (ConsumerDto)proxyFactory.getProxy();
                }
            }

            return consumerDO;
        }

        /**
         * 获取当前登录的用户信息,返回的ConsumerDto的地址是经过加密的
         * @return
         */
        public ConsumerDto getConsumerAddr3DESDto() {
            ConsumerDto src = getConsumerDO();
            src.getAddrProvince();//触发数据库获取
            ConsumerDto consumerDto = new ConsumerDto();
            org.springframework.beans.BeanUtils.copyProperties(src, consumerDto);
            if (consumerDto != null) {
                //用户所在省加密
                if (StringUtils.isNotBlank(consumerDto.getAddrProvince())) {
                    consumerDto.setAddrProvince(DESCrypto.encrypt3DE(consumerDto.getAddrProvince().trim()));
                }
                //用户所在市加密
                if (StringUtils.isNotBlank(consumerDto.getAddrCity())) {
                    consumerDto.setAddrCity(DESCrypto.encrypt3DE(consumerDto.getAddrCity().trim()));
                }
                //用户所在区加密
                if (StringUtils.isNotBlank(consumerDto.getAddrArea())) {
                    consumerDto.setAddrArea(DESCrypto.encrypt3DE(consumerDto.getAddrArea().trim()));
                }
                //用户所在详细地址加密
                if (StringUtils.isNotBlank(consumerDto.getAddrDetail())) {
                    consumerDto.setAddrDetail(DESCrypto.encrypt3DE(consumerDto.getAddrDetail().trim()));
                }
            }

            return consumerDto;
        }

        //延迟初始化
        public AppSimpleDto getConsumerAppDO() {
            if (consumerAppDO == null) {
                if(consumerCookieDto == null){//如果cookie中没有有效的用户信息，则直接返回
                    return null;
                }else{
                    //返回经过代理的对象，获取id字段时不会读取数据库，获取其他字段才会从数据库加载
                    AppSimpleDto temp = new AppSimpleDto();
                    temp.setId(consumerCookieDto.getAppId());

                    ProxyFactory proxyFactory = new ProxyFactory();
                    proxyFactory.setTarget(temp);
                    proxyFactory.addAdvice(new AppMethodInterceptor(this));
                    consumerAppDO = (AppSimpleDto)proxyFactory.getProxy();
                }
            }
            return consumerAppDO;
        }

        public HttpServletResponse getResponse() {
            if(response == null){
                throw new IllegalStateException("response must not be null, please invoke RequestLocal.setThreadLocallyAndLimitQps first");
            }
            return response;
        }
        public HttpServletRequest getRequest() {
            if(request == null){
                throw new IllegalStateException("request must not be null, please invoke RequestLocal.setThreadLocallyAndLimitQps first");
            }
            return request;
        }

        public String getTokenId() {
            return tokenId;
        }

        public String getDeap(){
            if (deap == null) {
                String mdeapVal = RequestTool.getCookie(request, "deap");
                String mdeap;
                try {
                    mdeap = new String(BaseEncoding.base64Url().withPadChar('.').decode(mdeapVal), Charset.forName("utf-8"));
                }catch(Exception e){
                    mdeap = mdeapVal;
                }
                if (StringUtils.isBlank(mdeap)) {
                    deap = "";//初始化，防止重复加载
                    return null;
                }

                String[] deaps = mdeap.split(",");
                for (String requestStr : deaps) {
                    if (requestStr.startsWith(getConsumerDO().getAppId() + "-")) {
                        deap = requestStr;
                        return deap;
                    }
                }
                if(deap == null){
                    deap = "";//初始化，防止重复加载
                }
            }

            return StringUtils.trimToNull(deap);
        }

        public boolean initAndLimitQpsForYjq(HttpServletRequest request, HttpServletResponse response) throws IOException {
            logger.info("yjq 解析cookie 开始。。。");
            this.request = request;
            this.response = response;

            ConsumerCookieDto consumerFromYjqInfo = null;
            //尝试从yjq_info中获取用户信息
            try {
                consumerFromYjqInfo = duibaConsumerCookieClient.getConsumerCookieDtoForYjq(request);
            } catch (Exception e) {
                logger.info("get user from yjq_info error.", e);
            }
            String appKey = request.getParameter("appKey");
            this.consumerCookieDto = consumerFromYjqInfo;
            logger.info("yjq 解析cookie consumerFromYjqInfo={}", JSON.toJSONString(consumerFromYjqInfo));
            //判断如果app禁用了，则打印禁用页面
            Long appId = null;
            if (this.consumerCookieDto != null) {
                appId = this.consumerCookieDto.getAppId();
            } else if (appKey != null) {
                Optional<Long> appIdOptional = appKey2IdCache.getUnchecked(appKey);
                appId = appIdOptional.isPresent() ? appIdOptional.get() : null;
            }
            if (appId != null && isAppForbidden(appId)) {
                showForbiddenPage(response);
                return true;
            }
            return false;
        }
    }

    private static class ConsumerMethodInterceptor implements MethodInterceptor{

        private boolean isLazyInitted = false;
        private HttpRequestDto httpRequestDto;

        public ConsumerMethodInterceptor(HttpRequestDto httpRequestDto){
            this.httpRequestDto = httpRequestDto;
        }

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            if(isLazyInitted){
                return invocation.proceed();
            }

            Method m = invocation.getMethod();
            if(m.getName().equals("getId") || m.getName().equals("getAppId") || m.getName().equals("getPartnerUserId") || m.getName().equals("isNotLoginUser")
            || m.getName().equals("toString") || m.getName().equals("equals") || m.getName().equals("hashCode")) {//m.getName().equals("isLazyInitted")
                //如果调用已经有了预置属性的方法，直接调用
                return invocation.proceed();
            }
            //调用其他get方法则从远程获取数据，并设置originalConsumer的属性。
            ConsumerDto originalConsumer = (ConsumerDto)invocation.getThis();
            ConsumerDto consumerInDb = remoteConsumerService.find(originalConsumer.getId());

            //可能调用方仍然持有proxy的实例，所以需要设置原始属性
            org.springframework.beans.BeanUtils.copyProperties(consumerInDb, originalConsumer);
            isLazyInitted = true;
            //同时改变原始引用为originalConsumer，加速下次的访问（下次访问不用再经过代理了）
            httpRequestDto.consumerDO = originalConsumer;

            return invocation.proceed();
        }
    }

    private static class AppMethodInterceptor implements MethodInterceptor{

        private boolean isLazyInitted = false;
        private HttpRequestDto httpRequestDto;

        public AppMethodInterceptor(HttpRequestDto httpRequestDto){
            this.httpRequestDto = httpRequestDto;
        }

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            if(isLazyInitted){
                return invocation.proceed();
            }

            Method m = invocation.getMethod();
            if(m.getName().equals("getId") || m.getName().equals("toString") || m.getName().equals("equals") || m.getName().equals("hashCode")) {//m.getName().equals("isLazyInitted")
                //如果调用已经有了预置属性的方法，直接调用
                return invocation.proceed();
            }
            //调用其他get方法则从远程获取数据，并设置originalConsumer的属性。
            AppSimpleDto originalApp = (AppSimpleDto)invocation.getThis();
            AppSimpleDto appInDb = remoteAppService.getSimpleApp(originalApp.getId()).getResult();

            //可能调用方仍然持有proxy的实例，所以需要设置原始属性
            org.springframework.beans.BeanUtils.copyProperties(appInDb, originalApp);
            isLazyInitted = true;
            //同时改变原始引用为originalConsumer，加速下次的访问（下次访问不用再经过代理了）
            httpRequestDto.consumerAppDO = originalApp;

            return invocation.proceed();
        }
    }

    private static HttpRequestDto get() {
        HttpRequestDto rl = local.get();
        if (rl == null) {
            rl = new HttpRequestDto();
            local.set(rl);
        }
        return rl;
    }

    /**
     * 获得当前登录的用户id
     * @return
     */
    public static Long getCid(){
        HttpRequestDto requestLocal = get();

        return requestLocal.consumerCookieDto == null ? null : requestLocal.consumerCookieDto.getCid();
    }

    /**
     * 获得当前登录的用户的appId
     * @return
     */
    public static Long getAppId(){
        HttpRequestDto requestLocal = get();

        return requestLocal.consumerCookieDto == null ? null : requestLocal.consumerCookieDto.getAppId();
    }

    /**
     * 获得当前登录的用户的partnerUserId
     * @return
     */
    public static String getPartnerUserId(){
        HttpRequestDto requestLocal = get();

        return requestLocal.consumerCookieDto == null ? null : requestLocal.consumerCookieDto.getPartnerUserId();
    }

    /**
     * 获取当前登录的用户信息,获取appId/id/partnerUserId字段时不会读取数据库，获取其他字段才会从数据库加载
     * @return
     */
    public static ConsumerDto getConsumerDO() {
        HttpRequestDto requestLocal = get();

        return requestLocal.getConsumerDO();
    }

    /**
     * 获取当前登录的用户信息,返回的ConsumerDto的地址是经过加密的
     * @return
     */
    public static ConsumerDto getConsumerAddr3DESDto() {
        HttpRequestDto requestLocal = get();

        return requestLocal.getConsumerAddr3DESDto();
    }

    /**
     * 获取当前登录的用户所在的app信息,获取id字段时不会读取数据库，获取其他字段才会从数据库加载
     * @return
     */
    public static AppSimpleDto getConsumerAppDO() {
        HttpRequestDto requestLocal = get();

        return requestLocal.getConsumerAppDO();
    }

    /**
     * 获取tokenId， tokenId的生命周期和wdata4相同，这个tokenId是唯一可以在js里直接获得的用户相关的cookie，（其他用户cookie全都设为了httpOnly，防止被XSS攻击获得）；作用主要有：用于传递给同盾做会话id。
     * @return
     */
    public static String getTokenId(){
        HttpRequestDto requestLocal = get();

        return requestLocal.getTokenId();
    }

    /**
     * 获取deap,防作弊用
     * @return
     */
    public static String getDeap(){
        HttpRequestDto requestLocal = get();

        return requestLocal.getDeap();
    }

    /**
     * 获取ip
     * @return
     */
    public static String getIp(){
        return RequestTool.getIpAddr(getRequest());
    }
    /**
     * 获取UserAgent
     * @return
     */
    public static String getUserAgent(){
        return RequestTool.getUserAgent(getRequest());
    }

    /**
     * 获取广告slotId
     * getSlotId:(这里用一句话描述这个方法的作用). <br/>
     *
     * @return
     * @author zp
     * @since JDK 1.6
     */
    public static String getSlotId() {
        HttpServletRequest request = getRequest();
        String slotId = request.getParameter(ADSLOTID);
        if (StringUtils.isBlank(slotId)) {
            slotId = RequestTool.getCookie(request, "_coll_slot");
        }
        return slotId;
    }

    /**
     * 获取ip
     * @return
     */
    public static HttpServletRequest getRequest(){
        HttpRequestDto requestLocal = get();

        return requestLocal.getRequest();
    }


    /**
     * 判断是否是ios
     *
     * @return
     */
    public static boolean isIos() {
        return RequestTool.isIos(getRequest());
    }

    /**
     * getResponse
     *
     * @return
     */
    public static HttpServletResponse getResponse() {
        HttpRequestDto requestLocal = get();

        return requestLocal.getResponse();
    }

    /**
     * 当前用户是否为登录状态(如果用户的partnerUserId为not_login也视为未登录状态)
     * @return
     */
    public static boolean isLoginConsumer(){
        String partnerUserId = getPartnerUserId();
        return partnerUserId != null && !ConsumerDto.NOTLOGINUSERID.equals(partnerUserId);
    }

    /**
     * 当前用户是否为分享用户
     * @return
     */
    public static boolean isShareConsumer(){
        return ConsumerDto.SHAREUSERID.equals(getPartnerUserId());
    }

    /**
     * 登录请求，用于把用户信息放入cookie(包括wdata4、w_ts、_ac、tokenId四个cookie)，调用此方法会顺便把用户信息放到ThreadLocal中，以便后续请求能获取到正确的用户信息
     * @param consumer
     * @return 设置到cookie中的用户信息
     */
    public static void injectConsumerInfoIntoCookie(ConsumerDto consumer){
        injectConsumerInfoIntoCookie(consumer, false);
    }

    /**
     * 多应用商品登录请求，用于把用户信息放入cookie(包括wdata4、w_ts、_ac、tokenId四个cookie)，调用此方法会顺便把用户信息放到ThreadLocal中，以便后续请求能获取到正确的用户信息
     * @param consumer
     * @return 设置到cookie中的用户信息
     */
    public static void injectConsumerInfoIntoCookie(ConsumerDto consumer, Integer isMultiApp){
        injectConsumerInfoIntoCookie(consumer, false, isMultiApp);
    }

    /**
     * 登录请求，用于把用户信息放入cookie(包括wdata4、w_ts、_ac、tokenId四个cookie)，调用此方法会顺便把用户信息放到ThreadLocal中，以便后续请求能获取到正确的用户信息
     * @param consumer
     * @param isFromCommercial 是否来自商业流量, 如果是商业流量，则用户信息会放入wdata3中，永久有效；否则用户信息放入wdata4中，24小时有效
     * @return 设置到cookie中的用户信息
     */
    public static void injectConsumerInfoIntoCookie(ConsumerDto consumer, boolean isFromCommercial){
        if(!needLogin){
            throw new IllegalStateException("not allowed to login");
        }
        HttpRequestDto requestLocal = get();

        ConsumerCookieDto consumerCookieDto = duibaConsumerCookieClient.injectConsumerInfoIntoCookie(consumer, requestLocal.getRequest(), requestLocal.getResponse(), cookieDomain, oldConsumerEncryptKey, isFromCommercial);

        //重置当前登录用户的属性
        requestLocal.consumerCookieDto = consumerCookieDto;
        requestLocal.consumerDO = null;
        requestLocal.consumerAppDO = null;
        requestLocal.tokenId = (String)requestLocal.getRequest().getAttribute("c_tokenId");
    }

    /**
     * 多应用商品登录请求，用于把用户信息放入cookie(包括wdata4、w_ts、_ac、tokenId四个cookie)，调用此方法会顺便把用户信息放到ThreadLocal中，以便后续请求能获取到正确的用户信息
     * @param consumer
     * @param isFromCommercial 是否来自商业流量, 如果是商业流量，则用户信息会放入wdata3中，永久有效；否则用户信息放入wdata4中，24小时有效
     * @param isMultiApp 是否多应用商品
     * @return 设置到cookie中的用户信息
     */
    public static void injectConsumerInfoIntoCookie(ConsumerDto consumer, boolean isFromCommercial, Integer isMultiApp){
        if(!needLogin){
            throw new IllegalStateException("not allowed to login");
        }
        HttpRequestDto requestLocal = get();

        ConsumerCookieDto consumerCookieDto = duibaConsumerCookieClient.injectConsumerInfoIntoCookie(consumer, requestLocal.getRequest(), requestLocal.getResponse(), cookieDomain, oldConsumerEncryptKey, isFromCommercial, isMultiApp);

        //重置当前登录用户的属性
        requestLocal.consumerCookieDto = consumerCookieDto;
        requestLocal.consumerDO = null;
        requestLocal.consumerAppDO = null;
        requestLocal.tokenId = (String)requestLocal.getRequest().getAttribute("c_tokenId");
    }

    /**
     * 有价券定制登录请求，把用户信息放入cookie（yjq_info），调用此方法会顺便把用户信息放到ThreadLocal中，以便后续请求能获取到正确的用户信息
     *
     * @param consumer
     */
    public static void injectConsumerInfoIntoCookieForYjq(ConsumerDto consumer) {
        logger.info("yjq 注入cookie consumer={}", JSON.toJSONString(consumer));
        HttpRequestDto requestLocal = get();

        ConsumerCookieDto consumerCookieDto = duibaConsumerCookieClient.injectConsumerInfoIntoCookieForYjq(consumer, requestLocal.getRequest(), requestLocal.getResponse(), cookieDomain, oldConsumerEncryptKey);

        //重置当前登录用户的属性
        requestLocal.consumerCookieDto = consumerCookieDto;
        requestLocal.consumerDO = null;
        requestLocal.consumerAppDO = null;
        requestLocal.tokenId = (String) requestLocal.getRequest().getAttribute("c_tokenId");

    }


    /**
     * 设置request和response ，必须在获取用户信息之前调用（一般是在LoginFilter开始时）,内部会自动解析出用户信息，如果是商业流量，还会自动设置永久的用户cookie
     * @param request
     * @param response
     * @return true表示此app被禁止访问，用户会看到禁止访问页面, 调用方应当判断返回值，为true时应该中断处理流程；false表示此次请求被放行。
     */
    public static boolean setThreadLocallyAndLimitQps(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
        clearThreadLocally();//先清理

        if(request == null || response == null){
            throw new IllegalArgumentException("request or response must not be null");
        }
        //静态资源不处理
        if (org.apache.commons.lang3.StringUtils.endsWithAny(request.getRequestURI(), ".js", ".css", ".png", ".jpg", ".gif")) {
            return false;
        }

        HttpRequestDto dto = get();
        String serverName = request.getServerName();
        boolean matches = Pattern.matches(YJQ_PATTERN, serverName);
        if (matches) {
            return dto.initAndLimitQpsForYjq(request, response);
        }
        return dto.initAndLimitQps(request, response);
    }

    /**
     * 获取当前开发者根域名，设置cookie的domain使用
     * duiba.com.cn则返回duiba.com.cn
     * 若x-host设置的则返回x-host
     */
    public static String getCookieDomain(HttpServletRequest request) {
        String domain = duibaConsumerCookieClient.getRequestDomain(request, cookieDomain);
        if (domain == null) {
            return cookieDomain;
        }
        return domain;
    }
    /**
     * 清空线程本地变量,必须在处理完用户请求之后调用（一般是在LoginFilter调用链结束时）
     */
    public static void clearThreadLocally() {
        local.remove();
    }


    private static void showForbiddenPage(HttpServletResponse response) throws IOException{
        response.setContentType("text/html;charset=UTF-8");
        response.setCharacterEncoding("utf-8");
        try(PrintWriter out=response.getWriter()){
            out.write(forbiddenPageHtml);
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        RemoteAppService remoteAppService = applicationContext.getBean(RemoteAppService.class);
        if(remoteAppService == null){
            throw new IllegalStateException("there must exists a bean of class RemoteAppService(in developer-center)");
        }
        RemoteConsumerService remoteConsumerService = applicationContext.getBean(RemoteConsumerService.class);
        if(remoteConsumerService == null){
            throw new IllegalStateException("there must exists a bean of class RemoteConsumerService(in consumer-center)");
        }
        RemoteKmsService remoteKmsService = applicationContext.getBean(RemoteKmsService.class);
        if(remoteKmsService == null){
            throw new IllegalStateException("there must exists a bean of class RemoteKmsService(in idmaker-service)");
        }
        Class clazz = Class.forName("cn.com.duiba.developer.center.api.remoteservice.devapp.RemoteAppDirectService");
        Object remoteAppDirectService = applicationContext.getBean(clazz);
        if (remoteAppDirectService == null) {
            throw new IllegalStateException("there must exists a bean of class RemoteAppDirectService(in activity-center)");
        }

        RequestLocal.remoteAppDirectService = remoteAppDirectService;
        if(needLogin) {
            cookieDomain = applicationContext.getEnvironment().getProperty("app.cross.domain");
            if(cookieDomain == null || cookieDomain.isEmpty()){
                throw new IllegalStateException("property:[app.cross.domain] does not exist");
            }
            if(cookieDomain.startsWith(".") || cookieDomain.startsWith("-")){//不能以.或者-开头
                cookieDomain = cookieDomain.substring(1);
            }
        }
        oldConsumerEncryptKey = applicationContext.getEnvironment().getProperty("app.consumer.encrypt.key");
        if(oldConsumerEncryptKey == null || oldConsumerEncryptKey.isEmpty()){
            throw new IllegalStateException("property:[app.consumer.encrypt.key] does not exist");
        }
        RequestLocal.remoteAppService = remoteAppService;
        RequestLocal.remoteConsumerService = remoteConsumerService;
        RequestLocal.duibaConsumerCookieClient = new DuibaConsumerCookieClient(remoteKmsService);
    }
}
