package cn.com.duibabiz.component.filters.bloom.basic;

import com.google.common.base.Charsets;
import com.google.common.hash.Funnel;
import com.google.common.hash.Funnels;
import com.google.common.hash.Hashing;


/**
 * 布隆过滤器配置类
 * 注意,改了之后，会影响hash判断不一致
 *
 * @author zengsp
 */
public class BloomFilterConfig {
    /**
     * hash函数的个数
     */
    private final int numHashFunctions;

    /**
     * bit数组长度
     */
    private final int bitSize;

    private static final Funnel<CharSequence> FUNNEL = Funnels.stringFunnel(Charsets.UTF_8);

    /**
     * 支持扩容临界时的并发数量
     */
    private final int growBuffer;
    /**
     * 数据量
     */
    private final int expectedInsertions;
    /**
     * 假阳性概率
     */
    private final double fpp;

    public BloomFilterConfig(int growBuffer, int expectedInsertions, double fpp) {
        this.growBuffer = growBuffer;
        this.expectedInsertions = expectedInsertions;
        this.fpp = fpp;

        this.bitSize = optimalNumOfBits(expectedInsertions, fpp);
        this.numHashFunctions = optimalNumOfHashFunctions(expectedInsertions, bitSize);
    }

    public int getNumHashFunctions() {
        return numHashFunctions;
    }

    public int getBitSize() {
        return bitSize;
    }

    int[] murmurHashOffset(String value) {
        int[] offset = new int[numHashFunctions];

        long hash64 = Hashing.murmur3_128().hashObject(value, FUNNEL).asLong();
        int hash1 = (int) hash64;
        int hash2 = (int) (hash64 >>> 32);
        for (int i = 1; i <= numHashFunctions; i++) {
            int nextHash = hash1 + i * hash2;
            if (nextHash < 0) {
                nextHash = ~nextHash;
            }
            offset[i - 1] = nextHash % bitSize;
        }

        return offset;
    }

    /**
     * 计算bit数组长度
     */
    private static int optimalNumOfBits(long n, double p) {
        if (p == 0) {
            p = Double.MIN_VALUE;
        }
        return (int) (-n * Math.log(p) / (Math.log(2) * Math.log(2)));
    }

    /**
     * 计算hash方法执行次数
     */
    private static int optimalNumOfHashFunctions(long n, long m) {
        return Math.max(1, (int) Math.round((double) m / n * Math.log(2)));
    }

    public int getGrowBuffer() {
        return growBuffer;
    }

    public int getExpectedInsertions() {
        return expectedInsertions;
    }

    public double getFpp() {
        return fpp;
    }
}
