package cn.com.duiba.boot.utils;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import org.slf4j.Logger;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import static org.slf4j.LoggerFactory.getLogger;

/**
 * 这个工具类的作用是检测如果某个markName在10秒内错误次数超过warnThreshold对应的值，则自动打印一条error日志<br/>
 * （日志格式：#onealert# 检测到${markName}在5秒内错误次数超过warnThreshold次，请确认），<br/>
 * 【统一日志平台已经配好规则，遇到任意一条包含#onealert#的日志即会触发告警】；<br/>
 * 业务方需要配合在触发降级逻辑的地方自行写入WarningUtils.markThresholdWarning代码；
 * 后续每个mark对应的warnThreshold会支持在统一平台动态修改生效（动态修改目前暂时不支持）。
 */
public class WarningUtils {

    private static final Logger logger = getLogger(WarningUtils.class);

    private WarningUtils(){}

    private static final LoadingCache<String, AtomicInteger> markName2CountCache = Caffeine.newBuilder()
            .expireAfterWrite(10, TimeUnit.SECONDS)
            .maximumSize(10000)
            .build(key -> new AtomicInteger());

    /**
     * 这个方法的作用是检测如果某个markName在10秒内错误次数超过warnThreshold对应的值，则自动打印一条error日志<br/>
     * （日志格式：#onealert# 检测到${markName}在5秒内错误次数超过warnThreshold次，请确认），<br/>
     * 【统一日志平台已经配好规则，遇到任意一条包含#onealert#的日志即会触发告警】；<br/>
     *
     * @param markName 标签名，会打印在日志中。便于快速判断问题，后期可以根据此markName在统一平台动态修改warnThreshold
     * @param warnThreshold 告警阈值，当10秒内该接口被调用了warnThreshold次，会触发日志告警（该日志10秒最多打印一条）
     */
    public static void markThresholdWarning(String markName, int warnThreshold){
        AtomicInteger errorCount = markName2CountCache.get(markName);
        int count = errorCount.incrementAndGet();
        if(count == warnThreshold) {
            logger.error("#onealert# 检测到 markName:{} 在10秒内错误次数超过{}次，请确认", markName, warnThreshold, new IllegalStateException("错误次数过多"));
        }
    }

}
