package cn.com.duiba.boot.utils;

import org.springframework.core.env.*;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

public class SpringEnvironmentUtils {

    public static final String DEV = "dev";
    public static final String TEST = "test";
    public static final String PRE = "pre";
    public static final String PROD = "prod";

    private SpringEnvironmentUtils(){}

    /**
     * 从spring的Environment获得扁平化的所有key value对
     * @param environment
     * @return
     */
    public static LinkedHashMap<String, Object> getFlatEnvironments(Environment environment) {
        final LinkedHashMap<String, Object> properties = new LinkedHashMap<>();
        for (Iterator it = ((AbstractEnvironment) environment).getPropertySources().iterator(); it.hasNext(); ) {
            PropertySource propertySource = (PropertySource) it.next();
            handlePropertySource(propertySource, properties);
        }
        return properties;
    }

    private static void handlePropertySource(PropertySource propertySource, LinkedHashMap<String, Object> properties){
        if (propertySource instanceof EnumerablePropertySource) {
            for (String key : ((EnumerablePropertySource) propertySource).getPropertyNames()) {
                properties.putIfAbsent(key, propertySource.getProperty(key));
            }
        }
        if (propertySource instanceof PropertiesPropertySource) {
            for (Map.Entry<String, Object> e : ((MapPropertySource) propertySource).getSource().entrySet()) {
                properties.putIfAbsent(e.getKey(), e.getValue());
            }
        }
        if (propertySource instanceof CompositePropertySource) {
            for (Map.Entry<String, Object> e : getPropertiesInCompositePropertySource((CompositePropertySource) propertySource).entrySet()) {
                properties.putIfAbsent(e.getKey(), e.getValue());
            }
        }
    }

    private static LinkedHashMap<String, Object> getPropertiesInCompositePropertySource(CompositePropertySource compositePropertySource) {
        final LinkedHashMap<String, Object> properties = new LinkedHashMap<>();
        compositePropertySource.getPropertySources().forEach(propertySource -> {
            if (propertySource instanceof EnumerablePropertySource) {
                for (String key : ((EnumerablePropertySource) propertySource).getPropertyNames()) {
                    properties.putIfAbsent(key, propertySource.getProperty(key));
                }
            }
            if (propertySource instanceof MapPropertySource) {
                for (Map.Entry<String, Object> e : ((MapPropertySource) propertySource).getSource().entrySet()) {
                    properties.putIfAbsent(e.getKey(), e.getValue());
                }
            }
            if (propertySource instanceof CompositePropertySource) {
                for (Map.Entry<String, Object> e : getPropertiesInCompositePropertySource((CompositePropertySource) propertySource).entrySet()) {
                    properties.putIfAbsent(e.getKey(), e.getValue());
                }
            }
        });
        return properties;
    }

    private static String[] getActiveProfiles(){
        if(MainApplicationContextHolder.getApplicationContext() == null){
            throw new IllegalStateException("应用还没有启动，你过早地调用了判断环境的方法");
        }
        return MainApplicationContextHolder.getApplicationContext().getEnvironment().getActiveProfiles();
    }

    /**
     * 是否在开发环境
     * @return
     */
    public static boolean isDevEnv(){
        for(String p : getActiveProfiles()){
            if(DEV.equals(p)){
                return true;
            }
        }
        return false;
    }

    /**
     * 是否在测试环境
     * @return
     */
    public static boolean isTestEnv(){
        for(String p : getActiveProfiles()){
            if(TEST.equals(p)){
                return true;
            }
        }
        return false;
    }

    /**
     * 是否在预发环境
     * @return
     */
    public static boolean isPreEnv(){
        for(String p : getActiveProfiles()){
            if(p != null && p.startsWith(PRE)){
                return true;
            }
        }
        return false;
    }

    /**
     * 是否在线上环境
     * @return
     */
    public static boolean isProdEnv(){
        String[] activeProfiles = MainApplicationContextHolder.getApplicationContext().getEnvironment().getActiveProfiles();
        for(String p : activeProfiles){
            if(p != null && p.startsWith(PROD)){
                return true;
            }
        }
        return false;
    }

    /**
     * 获取当前所在的环境
     * @return
     */
    public static String getCurrentEnv() {

        if(SpringEnvironmentUtils.isDevEnv()) {
            return SpringEnvironmentUtils.DEV;
        }

        if(SpringEnvironmentUtils.isTestEnv()) {
            return SpringEnvironmentUtils.TEST;
        }

        if(SpringEnvironmentUtils.isPreEnv()) {
            return SpringEnvironmentUtils.PRE;
        }

        if(SpringEnvironmentUtils.isProdEnv()) {
            return SpringEnvironmentUtils.PROD;
        }

        throw new IllegalStateException("没有获取系统环境");
    }

}
