package cn.com.duiba.boot.perftest;

import com.alibaba.ttl.TransmittableThreadLocal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Arrays;

public final class PerfTestContext {
    private static final Logger logger = LoggerFactory.getLogger(PerfTestContext.class);
    private PerfTestContext(){}

    private static volatile boolean _perfDebug = false;

    //当前线程是否在压测模式
    private Boolean isPerfTestMode;
    // 压测场景id
    private String sceneId;
    // 是否独立容器集群压测
    private boolean isTestCluster;
    //debug模式使用，记录设置压测标记时的调用堆栈，方便排查错误
    private StackTraceElement[] stackTraceElements;
    //debug模式使用，记录设置压测标记时的时间，方便排查错误
    private long timestamp;

    /**
     * 获得最近设置压测标记为true时的时间戳，需要在启动脚本中增加-DperfDebug=true
     * @return
     */
    public long getTimestamp() {
        return timestamp;
    }

    /**
     * 获得最近设置压测标记为true时的方法栈，需要在启动脚本中增加-DperfDebug=true
     * @return
     */
    public StackTraceElement[] getStackTraceElements() {
        return stackTraceElements;
    }

    private PerfTestContext copy() {
        PerfTestContext copyValue = new PerfTestContext();
        copyValue.isPerfTestMode = this.isPerfTestMode;
        copyValue.sceneId = this.sceneId;
        copyValue.isTestCluster = this.isTestCluster;
        copyValue.stackTraceElements = this.stackTraceElements;
        copyValue.timestamp = this.timestamp;
        return copyValue;
    }

    private static class TransmittableThreadLocalHolder {

        private TransmittableThreadLocalHolder() {
        }

        protected static final TransmittableThreadLocal<PerfTestContext> threadLocal2PressureTest = new TransmittableThreadLocal<PerfTestContext>() {

            @Override
            public PerfTestContext copy(PerfTestContext parentValue) {
                return parentValue != null ? parentValue.copy() : null;
            }

            @Override
            protected PerfTestContext childValue(PerfTestContext parentValue) {
                return parentValue != null ? parentValue.copy() : null;
            }
        };
    }

    /**
     * 判断当前是否是压测请求
     * @return
     */
    public static boolean isCurrentInPerfTestMode(){
        PerfTestContext c = TransmittableThreadLocalHolder.threadLocal2PressureTest.get();
        return c != null && c.isPerfTestMode != null && c.isPerfTestMode;
    }

    /**
     * 获取当前压测的场景id
     * @return
     */
    static String getCurrentSceneId() {
        PerfTestContext c = TransmittableThreadLocalHolder.threadLocal2PressureTest.get();
        if (c == null) {
            return null;
        }
        return c.sceneId;
    }

    /**
     * 是否独立容器集群
     * @return
     */
    static boolean isTestCluster() {
        PerfTestContext c = TransmittableThreadLocalHolder.threadLocal2PressureTest.get();
        return c != null && c.isTestCluster;
    }

    /**
     * 标记为压测请求，需要设置场景id
     * 注意：除非特殊情况，否则框架使用者不应该调用此接口
     * @param sceneId
     * @param isTestCluster
     */
    static void markAsPerfTest(String sceneId, boolean isTestCluster) {
        set(true, sceneId, isTestCluster);
    }

    /**
     * 标记为生产请求
     * 注意：除非特殊情况，否则框架使用者不应该调用此接口
     */
    static void markAsNormal() {
        set(false, null, false);
    }

    /**
     * 设置是否压测请求，以及压测场景id，框架使用者不应该调用此接口
     * @param isPerfTestMode
     * @param sceneId
     */
    private static void set(boolean isPerfTestMode, String sceneId, boolean isTestCluster) {
        if (isPerfTestMode) {
            PerfTestContext c = new PerfTestContext();
            c.isPerfTestMode = true;
            c.sceneId = sceneId;
            c.isTestCluster = isTestCluster;
            if (_perfDebug) {
                c.stackTraceElements = Thread.currentThread().getStackTrace();
                c.timestamp = System.currentTimeMillis();
            }
            TransmittableThreadLocalHolder.threadLocal2PressureTest.set(c);
        } else {
            TransmittableThreadLocalHolder.threadLocal2PressureTest.remove();
        }
    }

    /**
     * 设置是否压测请求，框架使用者不应该调用此接口
     * @param val
     */
    public static void _setPerfTestMode(Boolean val) {
        if (val == null) {
            TransmittableThreadLocalHolder.threadLocal2PressureTest.remove();
            return;
        }
        PerfTestContext c = TransmittableThreadLocalHolder.threadLocal2PressureTest.get();
        if (c == null) {
            c = new PerfTestContext();
        }
        c.isPerfTestMode = val;
        if (_perfDebug && val) {
            c.stackTraceElements = Thread.currentThread().getStackTrace();
            c.timestamp = System.currentTimeMillis();
        }
        TransmittableThreadLocalHolder.threadLocal2PressureTest.set(c);
    }

    public static PerfTestContext _currentContext(){
        return TransmittableThreadLocalHolder.threadLocal2PressureTest.get();
    }

    public static void debugInfo(String typeName){
        if (logger.isDebugEnabled()) {
            StringBuilder sb = new StringBuilder();
            Arrays.stream(Thread.currentThread().getStackTrace()).filter(x->x.getClassName().startsWith("cn.com.duiba.")).forEach(x->
                sb.append("\n\t").append(x.getClassName()).append(".").append(x.getMethodName()).append(":").append(x.getLineNumber())
            );
            logger.debug("Perf [{}] From {}",typeName, sb);
        }
    }
}
