/**
 * Copyright 2012 Netflix, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.com.duiba.boot.netflix.feign.hystrix;

import java.lang.annotation.*;


/**
 * This annotation used to specify some methods which should be processes as hystrix commands.
 */
@Target({ElementType.METHOD,ElementType.TYPE})
@Retention(RetentionPolicy.RUNTIME)
@Inherited
@Documented
public @interface FeignHystrixCommand {

    /**
     * The command group key is used for grouping together commands such as for reporting,
     * alerting, dashboards or team/library ownership.（当FeignHystrixCommand被同时注解到类和相同类的方法上时，优先使用方法上的非空值）
     * <p/>
     * default => the runtime class name of annotated method
     *
     * @return group key
     */
    String groupKey() default "";

    /**
     * Hystrix command key.(当FeignHystrixCommand被注解在类上时，此属性忽略)
     * <p/>
     * default => the name of annotated method. for example:
     * <code>
     *     ...
     *     @HystrixCommand
     *     public User getUserById(...)
     *     ...
     *     the command name will be: 'getUserById'
     * </code>
     *
     * @return command key
     */
    String commandKey() default "";

    /**
     * The thread-pool key is used to represent a
     * HystrixThreadPool for monitoring, metrics publishing, caching and other such uses.（当FeignHystrixCommand被同时注解到类和相同类的方法上时，优先使用方法上的非空值）
     *
     * @return thread pool key
     */
    String threadPoolKey() default "";

    /**
     * Specifies command properties or thread pool properties.（当FeignHystrixCommand被同时注解到类和相同类的方法上时，类上的属性会对所有方法生效，方法上的属性如果同名，会覆盖类上的同名属性）
     *
     * @return command properties
     */
    FeignHystrixProperty[] properties() default {};

    /**
     * Defines exceptions which should be ignored and wrapped to throw in HystrixBadRequestException.
     * 这里指定的异常不会触发熔断或降级
     *
     * @return exceptions to ignore
     */
    Class<? extends Throwable>[] ignoreExceptions() default {};

}

