package cn.com.duiba.boot.serializer;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.Registration;
import com.esotericsoftware.kryo.factories.ReflectionSerializerFactory;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.kryo.serializers.CompatibleFieldSerializer;
import lombok.extern.slf4j.Slf4j;
import org.objenesis.strategy.StdInstantiatorStrategy;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ConcurrentSkipListSet;

/**
 * kryo序列化工具，仅供录制回归使用
 * Created by guoyanfei .
 * 2019-04-18 .
 */
@Slf4j
public class KryoSerializer {

    private static final int BUFFER_SIZE = 4096 * 4;

    private static final Class<?> ARRAY_AS_LIST;
    private static final Class<?> UNMODIFIABLE_COLLECTION;
    private static final Class<?> UNMODIFIABLE_LIST;
    private static final Class<?> UNMODIFIABLE_MAP;
    private static final Class<?> UNMODIFIABLE_NAVIGABLEMAP;
    private static final Class<?> UNMODIFIABLE_NAVIGABLESET;
    private static final Class<?> UNMODIFIABLE_RANDOMACCESSLIST;
    private static final Class<?> UNMODIFIABLE_SET;
    private static final Class<?> UNMODIFIABLE_SORTEDMAP;
    private static final Class<?> UNMODIFIABLE_SORTEDSET;

    private static final Class<?> GOOGLE_LISTS_ABSTRACTLISTWRAPPER;
    private static final Class<?> GOOGLE_LISTS_CHARSEQUENCEASLIST;
    private static final Class<?> GOOGLE_LISTS_ONEPLUSARRAYLIST;
    private static final Class<?> GOOGLE_LISTS_PARTITION;
    private static final Class<?> GOOGLE_LISTS_RANDOMACCESSLISTWRAPPER;
    private static final Class<?> GOOGLE_LISTS_RANDOMACCESSPARTITION;
    private static final Class<?> GOOGLE_LISTS_RANDOMACCESSREVERSELIST;
    private static final Class<?> GOOGLE_LISTS_REVERSELIST;
    private static final Class<?> GOOGLE_LISTS_STRINGASIMMUTABLELIST;
    private static final Class<?> GOOGLE_LISTS_TRANSFORMINGRANDOMACCESSLIST;
    private static final Class<?> GOOGLE_LISTS_TRANSFORMINGSEQUENTIALLIST;
    private static final Class<?> GOOGLE_LISTS_TWOPLUSARRAYLIST;

    static {
        ARRAY_AS_LIST = classForName("java.util.Arrays$ArrayList");
        UNMODIFIABLE_COLLECTION = classForName("java.util.Collections$UnmodifiableCollection");
        UNMODIFIABLE_LIST = classForName("java.util.Collections$UnmodifiableList");
        UNMODIFIABLE_MAP = classForName("java.util.Collections$UnmodifiableMap");
        UNMODIFIABLE_NAVIGABLEMAP = classForName("java.util.Collections$UnmodifiableNavigableMap");
        UNMODIFIABLE_NAVIGABLESET = classForName("java.util.Collections$UnmodifiableNavigableSet");
        UNMODIFIABLE_RANDOMACCESSLIST = classForName("java.util.Collections$UnmodifiableRandomAccessList");
        UNMODIFIABLE_SET = classForName("java.util.Collections$UnmodifiableSet");
        UNMODIFIABLE_SORTEDMAP = classForName("java.util.Collections$UnmodifiableSortedMap");
        UNMODIFIABLE_SORTEDSET = classForName("java.util.Collections$UnmodifiableSortedSet");

        GOOGLE_LISTS_ABSTRACTLISTWRAPPER = classForName("com.google.common.collect.Lists$AbstractListWrapper");
        GOOGLE_LISTS_CHARSEQUENCEASLIST = classForName("com.google.common.collect.Lists$CharSequenceAsList");
        GOOGLE_LISTS_ONEPLUSARRAYLIST = classForName("com.google.common.collect.Lists$OnePlusArrayList");
        GOOGLE_LISTS_PARTITION = classForName("com.google.common.collect.Lists$Partition");
        GOOGLE_LISTS_RANDOMACCESSLISTWRAPPER = classForName("com.google.common.collect.Lists$RandomAccessListWrapper");
        GOOGLE_LISTS_RANDOMACCESSPARTITION = classForName("com.google.common.collect.Lists$RandomAccessPartition");
        GOOGLE_LISTS_RANDOMACCESSREVERSELIST = classForName("com.google.common.collect.Lists$RandomAccessReverseList");
        GOOGLE_LISTS_REVERSELIST = classForName("com.google.common.collect.Lists$ReverseList");
        GOOGLE_LISTS_STRINGASIMMUTABLELIST = classForName("com.google.common.collect.Lists$StringAsImmutableList");
        GOOGLE_LISTS_TRANSFORMINGRANDOMACCESSLIST = classForName("com.google.common.collect.Lists$TransformingRandomAccessList");
        GOOGLE_LISTS_TRANSFORMINGSEQUENTIALLIST = classForName("com.google.common.collect.Lists$TransformingSequentialList");
        GOOGLE_LISTS_TWOPLUSARRAYLIST = classForName("com.google.common.collect.Lists$TwoPlusArrayList");
    }

    private static Class<?> classForName(String className) {
        try {
            return Class.forName(className);
        } catch (ClassNotFoundException e) {
            log.error("{} class not found", className, e);
            return null;
        }
    }

    private static ThreadLocal<Kryo> kryoThreadLocal = ThreadLocal.withInitial(() -> {
        Kryo kryo = new Kryo(){
            public Registration writeClass (Output output, Class type) {
                if(type == ARRAY_AS_LIST){//java.util.Arrays$ArrayList用KRYO的默认实现在反序列化时会报错，这里用ArrayList代替
                    type = ArrayList.class;
                } else if (type == UNMODIFIABLE_COLLECTION) {
                    type = ArrayList.class;
                } else if (type == UNMODIFIABLE_LIST) {
                    type = ArrayList.class;
                } else if (type == UNMODIFIABLE_MAP) {
                    type = HashMap.class;
                } else if (type == UNMODIFIABLE_NAVIGABLEMAP) {
                    type = ConcurrentSkipListMap.class;
                } else if (type == UNMODIFIABLE_NAVIGABLESET) {
                    type = ConcurrentSkipListSet.class;
                } else if (type == UNMODIFIABLE_RANDOMACCESSLIST) {
                    type = ArrayList.class;
                } else if (type == UNMODIFIABLE_SET) {
                    type = HashSet.class;
                } else if (type == UNMODIFIABLE_SORTEDMAP) {
                    type = ConcurrentSkipListMap.class;
                } else if (type == UNMODIFIABLE_SORTEDSET) {
                    type = ConcurrentSkipListSet.class;
                } else if (type == GOOGLE_LISTS_ABSTRACTLISTWRAPPER) {
                    type = ArrayList.class;
                } else if (type == GOOGLE_LISTS_CHARSEQUENCEASLIST) {
                    type = ArrayList.class;
                } else if (type == GOOGLE_LISTS_ONEPLUSARRAYLIST) {
                    type = ArrayList.class;
                } else if (type == GOOGLE_LISTS_PARTITION) {
                    type = ArrayList.class;
                } else if (type == GOOGLE_LISTS_RANDOMACCESSLISTWRAPPER) {
                    type = ArrayList.class;
                } else if (type == GOOGLE_LISTS_RANDOMACCESSPARTITION) {
                    type = ArrayList.class;
                } else if (type == GOOGLE_LISTS_RANDOMACCESSREVERSELIST) {
                    type = ArrayList.class;
                } else if (type == GOOGLE_LISTS_REVERSELIST) {
                    type = ArrayList.class;
                } else if (type == GOOGLE_LISTS_STRINGASIMMUTABLELIST) {
                    type = ArrayList.class;
                } else if (type == GOOGLE_LISTS_TRANSFORMINGRANDOMACCESSLIST) {
                    type = ArrayList.class;
                } else if (type == GOOGLE_LISTS_TRANSFORMINGSEQUENTIALLIST) {
                    type = ArrayList.class;
                } else if (type == GOOGLE_LISTS_TWOPLUSARRAYLIST) {
                    type = ArrayList.class;
                }
                return super.writeClass(output, type);
            }
        };

        //支持增删字段
        kryo.setDefaultSerializer(new ReflectionSerializerFactory(CompatibleFieldSerializer.class));
//		kryo.getFieldSerializerConfig().setFixedFieldTypes(true);
//		kryo.getFieldSerializerConfig().setOptimizedGenerics(true);
        kryo.setInstantiatorStrategy(new Kryo.DefaultInstantiatorStrategy(new StdInstantiatorStrategy()));
        return kryo;
    });

    private KryoSerializer() {
    }

    /**
     * 反序列化
     * @param bytes
     * @return
     */
    public static Object deserialize(byte[] bytes) {
        if (bytes == null || bytes.length == 0) {
            return null;
        }
        Kryo kryo = kryoThreadLocal.get();
        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        Input input = new Input(in, BUFFER_SIZE);
        Object obj = kryo.readClassAndObject(input);
        input.close();
        return obj;
    }

    /**
     * 序列化
     * @param object
     * @return
     */
    public static byte[] serialize(Object object) {
        if (object == null) {
            return null;
        }
        Kryo kryo = kryoThreadLocal.get();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        Output output = new Output(out, BUFFER_SIZE);
        kryo.writeClassAndObject(output, object);
        output.close();
        return out.toByteArray();
    }

}
