package cn.com.duiba.boot.utils;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;

import java.io.ByteArrayOutputStream;
import java.nio.charset.Charset;
import java.util.zip.DataFormatException;
import java.util.zip.Deflater;
import java.util.zip.Inflater;

/**
 * zlib(即DEFLATE)压缩解压工具类
 */
public class DeflateUtils {

    /**
     * 压缩level=1的默认压缩
     * @param message 字符串
     * @return 字节数组
     */
    public static byte[] compress(String message) {
        return compress(message, 1);
    }

    /**
     * 字符串压缩
     * @param message 字符串
     * @param level 压缩级别 1-9
     * @return 字节数组
     */
    public static byte[] compress(String message, int level) {
        return compress(stringToByte(message), level);
    }

    /**
     * 字节数组压缩
     * @param input 字节数组
     * @param level 压缩级别
     * @return 压缩后的字节数组
     */
    private static byte[] compress(byte[] input, int level) {
        if(ArrayUtils.isEmpty(input)){
            return input;
        }
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        Deflater deflater = new Deflater(level);
        try {
            deflater.setInput(input);
            deflater.finish();
            byte[] buf = new byte[2048];
            while (!deflater.finished()) {
                int count = deflater.deflate(buf);
                bos.write(buf, 0, count);
            }
            deflater.end();
        } finally {
            IOUtils.closeQuietly(bos);
        }
        return bos.toByteArray();
    }

    /**
     * 解压缩
     * @param input 压缩字节数组
     * @return 解压缩后的字节数组
     */
    public static byte[] uncompress(byte[] input) {
        if(ArrayUtils.isEmpty(input)){
            return null;
        }
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        Inflater inflater = new Inflater();
        try {
            inflater.setInput(input);
            byte[] buf = new byte[2048];
            while (!inflater.finished()) {
                int count = inflater.inflate(buf);
                bos.write(buf, 0, count);
            }
        } catch (DataFormatException e) {
            throw new RuntimeException(e);
        } finally {
            inflater.end();
            IOUtils.closeQuietly(bos);
        }
        return bos.toByteArray();
    }

    /**
     * 解压缩后转成字符串
     * @param input 压缩字节数组
     * @return 字符串
     */
    public static String uncompressToString(byte[] input) {
        return byteToString(uncompress(input));
    }

    /**
     * 将字符串转化为字节数组
     *
     * @param str 字符串
     * @return 字节数组
     */
    private static byte[] stringToByte(String str) {
        if (StringUtils.isNotBlank(str)) {
            return str.getBytes(Charset.forName("UTF-8"));
        }
        return null;
    }

    /**
     * 将byte数组转化为字符串
     *
     * @param bytes byte数组
     * @return 字符串
     */
    private static String byteToString(byte[] bytes) {
        if (ArrayUtils.isNotEmpty(bytes)) {
            return new String(bytes, Charset.forName("UTF-8"));
        }
        return null;
    }

}
