package cn.com.duiba.boot.utils;

import com.google.common.primitives.UnsignedBytes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.net.util.IPAddressUtil;

import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;
import java.util.regex.Pattern;

/**
 * Created by wenqi.huang on 2017/6/19.
 */
public class NetUtils {

    private NetUtils(){}

    private static final Logger log = LoggerFactory.getLogger(NetUtils.class);

    public static final String ANYHOST = "0.0.0.0";//NOSONAR
    private static final Pattern IP_PATTERN = Pattern.compile("\\d{1,3}(\\.\\d{1,3}){3,5}$");

    private static volatile String localIP = null;

    /**
     * 获取本机内网IP(结果会缓存)
     *
     * @return
     */
    public static String getLocalIp() {
        if (localIP != null) {
            return localIP;
        }
        try {
            InetAddress localAddress = InetAddress.getLocalHost();
            if (isValidLocalAddress(localAddress)) {
                localIP = localAddress.getHostAddress();
                return localIP;
            }
        } catch (Throwable e) {
            log.warn("Failed to retriving ip address, " + e.getMessage(), e);
        }
        Enumeration netInterfaces;
        try {
            netInterfaces = NetworkInterface.getNetworkInterfaces();
        } catch (SocketException e) {
            log.error("getLocalIp error", e);
            return null;
        }
        while (netInterfaces.hasMoreElements()) {
            NetworkInterface ni = (NetworkInterface) netInterfaces.nextElement();
            Enumeration address = ni.getInetAddresses();
            while (address.hasMoreElements()) {
                InetAddress ip = (InetAddress) address.nextElement();
                if (ip.isSiteLocalAddress() && !ip.isLoopbackAddress() && ip.getHostAddress().indexOf(':') == -1) {//内网IP
                    localIP = ip.getHostAddress();
                    return localIP;
                }
            }
        }
        return localIP;
    }

    /**
     * 判断是否合法的本机内网ip
     * @return
     */
    private static boolean isValidLocalAddress(InetAddress address){
        if (address == null || address.isLoopbackAddress() || !address.isSiteLocalAddress()) {
            return false;
        }
        if(address instanceof Inet6Address){
            return false;
        }
        String name = address.getHostAddress();
        return (name != null
                && ! ANYHOST.equals(name)
                && IP_PATTERN.matcher(name).matches());
    }

    /**
     * 是否是内网ip
     * 以下范围的属于内网IP：<br/>
     *       10.0.0.0~10.255.255.255（A类）<br/>
     *       172.16.0.0~172.31.255.255（B类）<br/>
     *       192.168.0.0~192.168.255.255（C类）<br/>
     *
     * @param ip
     * @return
     */
    public static boolean isLanIp(String ip) {
        byte[] addr = IPAddressUtil.textToNumericFormatV4(ip);
        return isLanIp(addr);
    }

    /**
     * 判断是不是内网IP等保留地址
     *
     * https://en.wikipedia.org/wiki/Reserved_IP_addresses
     *
     * @param ip
     * @return
     */
    public static boolean isReservedIp(String ip) {
        byte[] addr = IPAddressUtil.textToNumericFormatV4(ip);
        return isReservedIp(addr);
    }

    //10.x.x.x/8 10.0.0.0–10.255.255.255
    private static final byte SECTION_1 = 10;
    //172.16.x.x/12 172.16.0.0–172.31.255.255
    private static final short SECTION_2 = 172;
    private static final byte SECTION_3 = 16;
    private static final byte SECTION_4 = 31;
    //192.168.x.x/16 192.168.0.0–192.168.255.255
    private static final short SECTION_5 = 192;
    private static final short SECTION_6 = 168;
    //100.64.0.0/10  （100.64.0.0–100.127.255.255）
    private static final byte SECTION_7 = 100;
    private static final byte SECTION_8 = 64;
    private static final byte SECTION_9 = 127;
    //169.254.0.0/16	169.254.0.0–169.254.255.255
    private static final short SECTION_10 = 169;
    private static final short SECTION_11 = 254;
    //192.0.0.0/24	192.0.0.0–192.0.0.255
    private static final short SECTION_12 = 192;
    private static final byte SECTION_13 = 0;
    //192.0.2.0/24	192.0.2.0–192.0.2.255
    private static final byte SECTION_14 = 2;
    //192.88.99.0/24	192.88.99.0–192.88.99.255
    private static final byte SECTION_15 = 88;
    private static final byte SECTION_16 = 99;
    //198.18.0.0/15	198.18.0.0–198.19.255.255
    private static final short SECTION_17 = 198;
    private static final byte SECTION_18 = 18;
    private static final byte SECTION_19 = 19;
    //198.51.100.0/24	198.51.100.0–198.51.100.255
    private static final byte SECTION_20 = 51;
    private static final byte SECTION_21 = 100;
    //203.0.113.0/24	203.0.113.0–203.0.113.255
    private static final short SECTION_22 = 203;
    private static final byte SECTION_23 = 0;
    private static final byte SECTION_24 = 113;
    //224.0.0.0/4	224.0.0.0–239.255.255.255
    private static final short SECTION_25 = 224;
    private static final short SECTION_26 = 239;
    //240.0.0.0/4	240.0.0.0–255.255.255.255
    private static final short SECTION_27 = 240;
    private static final short SECTION_28 = 255;
    //255.255.255.255/32	255.255.255.255
    private static final short SECTION_29 = 255;
    //0.0.0.0/8	0.0.0.0–0.255.255.255
    private static final byte SECTION_30 = 0;
    //127.0.0.0/8	127.0.0.0–127.255.255.255
    private static final byte SECTION_31 = 127;

    private static boolean isLanIp(byte[] addr) {
        if(addr == null || addr.length < 2){
            return false;
        }
        final int b0 = UnsignedBytes.toInt(addr[0]);
        final int b1 = UnsignedBytes.toInt(addr[1]);
        if(b0 == SECTION_1){
            return true;
        }else if(b0 == SECTION_2 && b1 >= SECTION_3 && b1 <= SECTION_4){
            return true;
        }else if(b0 == SECTION_5 && b1 == SECTION_6){
            return true;
        }

        return false;
    }

    private static boolean isReservedIp(byte[] addr) { //NOSONAR
        if(addr == null || addr.length < 2){
            return false;
        }

        final int b0 = UnsignedBytes.toInt(addr[0]);
        final int b1 = UnsignedBytes.toInt(addr[1]);
        final int b2 = UnsignedBytes.toInt(addr[2]);
        final int b3 = UnsignedBytes.toInt(addr[3]);

        if(b0 == SECTION_1){
            return true;
        }else if(b0 == SECTION_2 && b1 >= SECTION_3 && b1 <= SECTION_4){
            return true;
        }else if(b0 == SECTION_5 && b1 == SECTION_6){
            return true;
        }else if(b0 == SECTION_7 && b1 >= SECTION_8 && b1 <= SECTION_9){
            return true;
        }else if(b0 == SECTION_10 && b1 == SECTION_11){
            return true;
        }else if(b0 == SECTION_12 && b1 == SECTION_13
                && (b2 == SECTION_13 || b2 == SECTION_14)){
            return true;
        }else if(b0 == SECTION_12 && b1 == SECTION_15
                && b2 == SECTION_16){
            return true;
        }else if(b0 == SECTION_17 && b1 >= SECTION_18 && b1 <= SECTION_19){
            return true;
        }else if(b0 == SECTION_17 && b1 == SECTION_20 && b2 == SECTION_21){
            return true;
        }else if(b0 == SECTION_22 && b1 == SECTION_23 && b2 == SECTION_24){
            return true;
        }else if(b0 >= SECTION_25 && b0 <= SECTION_26){
            return true;
        }else if(b0 >= SECTION_27 && b1 <= SECTION_28){
            return true;
        }else if(b0 == SECTION_29 && b1 == SECTION_29 && b2 == SECTION_29 && b3 == SECTION_29){
            return true;
        }else if(b0 == SECTION_30){
            return true;
        }else if(b0 == SECTION_31){
            return true;
        }

        return false;
    }
}
