package cn.com.duiba.boot.ext.autoconfigure.logger;

import ch.qos.logback.core.Appender;
import ch.qos.logback.core.rolling.RollingFileAppender;
import cn.com.duiba.boot.ext.autoconfigure.perftest.PerfTestRoutingDataSource;
import org.apache.commons.dbcp2.BasicDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnResource;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.io.ClassPathResource;

import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.joran.JoranConfigurator;

import java.util.Map;

/**
 * Created by xugf on 2016/11/25
 */
@Configuration
@ConditionalOnClass({LoggerContext.class})
@ConditionalOnResource(resources="/logback/loginsight-logback.xml")
public class LoggerAutoConfiguration {
	private static final Logger logger = LoggerFactory.getLogger(LoggerAutoConfiguration.class);

	/**
	 * 附加logback配置文件
	 * @return
	 */
	@Bean
	public ApplicationListener loggerConfiguar() {
		return new ApplicationListener<ContextRefreshedEvent>(){

			private boolean flag = true;

			@Override
			public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {
				if(flag){
					try {
						LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
						JoranConfigurator configurator = new JoranConfigurator();
						configurator.setContext(loggerContext);
						//logback 添加新的配置文件
						configurator.doConfigure(new ClassPathResource("/logback/loginsight-logback.xml").getInputStream());

						Appender loginsightAppender = loggerContext.getLogger("loginsight_log").getAppender("LOGINSIGHT_APPENDER");
						loggerContext.getLogger(ch.qos.logback.classic.Logger.ROOT_LOGGER_NAME).addAppender(loginsightAppender);
					} catch (Exception e) {
						logger.error(e.getMessage(), e);
					}
					flag = false;
				}
			}
		};
	}

	/**
	 * 数据库连接池检查报警,当连接池数目过满时打印报警日志
	 */
	@Configuration
	@ConditionalOnResource(resources="/logback/loginsight-logback.xml")
	@ConditionalOnClass({BasicDataSource.class, LoggerContext.class})
	public static class dataSourceMonitorConfigurer{

		@Bean(name = "_dataSourceMonitor")
		public DataSourceMonitor _dataSourceMonitor(){
			return new DataSourceMonitor();
		}

		@Bean
		public ApplicationListener dataSourceMonitor() {
			return new ApplicationListener<ContextRefreshedEvent>() {

				@Override
				public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {
					ApplicationContext context = applicationStartedEvent.getApplicationContext();
					DataSourceMonitor dataSourceMonitor = context.getBean("_dataSourceMonitor", DataSourceMonitor.class);
					dataSourceMonitor.startMonitorThread(context);
				}
			};
		}

		private static class DataSourceMonitor implements DisposableBean{
			private boolean initted = false;
			private Map<String,BasicDataSource> dataSourceMap;
			private Map<String,PerfTestRoutingDataSource> routingdataSourceMap;

			private Thread dataSourceMonitorThread;

			public synchronized void startMonitorThread(ApplicationContext context){
				if (!initted) {
					initted = true;
					dataSourceMap = context.getBeansOfType(BasicDataSource.class);
					routingdataSourceMap = context.getBeansOfType(PerfTestRoutingDataSource.class);

					if(dataSourceMap.isEmpty() && routingdataSourceMap.isEmpty()){
						return;
					}

					dataSourceMonitorThread = new Thread(){

						@Override
						public void run(){
							while(true){
								for(Map.Entry<String, BasicDataSource> entry : dataSourceMap.entrySet()){
									scanDataSource(entry.getKey(), entry.getValue());
								}
								for(Map.Entry<String, PerfTestRoutingDataSource> entry : routingdataSourceMap.entrySet()){
									scanDataSource(entry.getKey()+"_original", entry.getValue().getOriginalDataSource());
									scanDataSource(entry.getKey()+"_shade", entry.getValue().getShadeDataSource());
								}

								try {
									Thread.sleep(10000);//每隔10S检查一次,看连接数是否正常
								} catch (InterruptedException e) {
									break;
								}
								if(Thread.currentThread().isInterrupted()){
									break;
								}
							}
						}

					};

					dataSourceMonitorThread.start();
				}
			}

			private void scanDataSource(String key, BasicDataSource ds){
				if(ds != null && !ds.isClosed()) {
					int numIdleConnections = ds.getNumIdle();//池中有多少个空闲连接，它们可以被checkout
					int numBusyConnections = ds.getNumActive();//池中被checkout的连接有多少个
					int numConnections = numBusyConnections + numIdleConnections;////当前共几个连接
					int max = ds.getMaxTotal();
					if (numBusyConnections >= 3 && numConnections >= max - 3 && numBusyConnections >= numConnections - 3) {
						logger.error("datasource [" + key + "]'s connectionPool is too full,max:" + max + ",busy:" + numBusyConnections + ",idle:" + numIdleConnections);
					}
				}
			}

			@Override
			public synchronized void destroy() throws Exception {
				if(dataSourceMonitorThread != null) {
					dataSourceMonitorThread.interrupt();
				}
			}
		}

	}

}
