package cn.com.duiba.boot.ext.autoconfigure.web;

import org.springframework.beans.factory.BeanFactory;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.web.WebMvcAutoConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.web.accept.ContentNegotiationManager;
import org.springframework.web.servlet.DispatcherServlet;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;
import org.springframework.web.servlet.view.ContentNegotiatingViewResolver;

import javax.servlet.Servlet;

/**
 * 把ContentNegotiatingViewResolver配置为最低顺序，以防mvcAutoConfiguration自动注入最高优先级的ContentNegotiatingViewResolver
 * <br/>
 * 这样做是为了防止线上出现大量html.vm/jpg.vm/xml.vm找不到的问题。
 * 此问题重现的一种方法是在访问时先加入.html后缀，然后用正常链接访问。
 * <br/>
 * ContentNegotiatingViewResolver的大概目的是自动判断当前访问的页面，然后自动附加html等后缀寻找对应文件，我们不需要该特性
 * Created by wenqi.huang on 2017/1/5.
 */
@Configuration
@AutoConfigureBefore({WebMvcAutoConfiguration.class})
@ConditionalOnClass({ Servlet.class, DispatcherServlet.class,
        WebMvcConfigurerAdapter.class })
@ConditionalOnWebApplication
public class WebMvcFixAutoConfiguration {

    @Bean(name = "viewResolver")
    @ConditionalOnMissingBean(name = "viewResolver", value = ContentNegotiatingViewResolver.class)
    public ContentNegotiatingViewResolver viewResolver(BeanFactory beanFactory) {
        ContentNegotiatingViewResolver resolver = new ContentNegotiatingViewResolver();
        resolver.setContentNegotiationManager(
                beanFactory.getBean(ContentNegotiationManager.class));
        // ContentNegotiatingViewResolver uses all the other view resolvers to locate
        // a view so it should have a high precedence
        resolver.setOrder(Ordered.LOWEST_PRECEDENCE);
        return resolver;
    }

}
