package cn.com.duibaboot.ext.autoconfigure.threadpool;

import cn.com.duibaboot.ext.autoconfigure.threadpool.wrapper.ScheduledThreadPoolExecutorWrapper;
import cn.com.duibaboot.ext.autoconfigure.threadpool.wrapper.ThreadPoolExecutorWrapper;
import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.Ordered;
import org.springframework.scheduling.concurrent.ThreadPoolExecutorFactoryBean;

import javax.annotation.Resource;
import java.util.concurrent.*;

/**
 * 自动构建线程池，使用文档：http://cf.dui88.com/pages/viewpage.action?pageId=4505751
 * Created by wenqi.huang on 2017/2/7.
 */
@Configuration
@ConditionalOnClass(ThreadPoolExecutorFactoryBean.class)
@EnableConfigurationProperties(ThreadPoolProperties.class)
public class ThreadPoolAutoConfiguration {

    private static Logger logger = LoggerFactory.getLogger(ThreadPoolAutoConfiguration.class);

    @Resource
    private ThreadPoolProperties threadPoolProperties;

    /**
     * 构建普通线程池,返回的线程池实现了SmartLifecycle，会在spring关闭的时候首先被关闭
     * @return
     */
    @Bean(name = "executorService", destroyMethod = "shutdown")
    @ConditionalOnProperty(name="duiba.threadpool.enabled", havingValue="true", matchIfMissing = false)
    public ExecutorService executorService(){
        BlockingQueue<Runnable> queue;
        int queueSize = threadPoolProperties.getQueueSize();
        if(queueSize == 0){
            queue = new SynchronousQueue<>();
        }else{
            queue = new ArrayBlockingQueue<>(queueSize);
        }

        int maxSize = Math.max(threadPoolProperties.getMaxSize(), threadPoolProperties.getCoreSize());
        ThreadPoolExecutor threadPool = new ThreadPoolExecutor(threadPoolProperties.getCoreSize(),
                maxSize,
                60L, TimeUnit.SECONDS,
                queue,
                new NamedThreadFactory("DuibaBiz"),
                new AbortPolicyWithReport());

        return new ThreadPoolExecutorWrapper(threadPool, threadPoolProperties.getShutdownTimeout());
    }

    /**
     * 配置定时调度线程池,返回的线程池实现了SmartLifecycle，会在spring关闭的时候首先被关闭
     * @return
     */
    @Bean(name = "scheduledExecutorService", destroyMethod = "shutdown")
    @ConditionalOnProperty(name="duiba.threadpool.scheduled.enabled", havingValue="true", matchIfMissing = false)
    public ScheduledExecutorService scheduledExecutorService(){
        ScheduledThreadPoolExecutor scheduledThreadPool = new ScheduledThreadPoolExecutor(threadPoolProperties.getScheduled().getCoreSize(),
                new NamedThreadFactory("DuibaBizScheduled"),
                new AbortPolicyWithReport());

        return new ScheduledThreadPoolExecutorWrapper(scheduledThreadPool, threadPoolProperties.getScheduled().getShutdownTimeout());
    }

    /**
     * 设置默认的UncaughtExceptionHandler，当线程运行报错时打印错误到slf4j中。
     * @return
     */
    @Configuration
    public static class DuibaUncaughtExceptionHandlerConfiguarApplicationListener implements ApplicationListener<ContextRefreshedEvent>, Ordered {

        private boolean flag = true;

        @Override
        public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {
            if(flag){
                if(Thread.getDefaultUncaughtExceptionHandler() == null){
                    Thread.setDefaultUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
                        @Override
                        public void uncaughtException(Thread t, Throwable e) {
                            if(e instanceof ThreadDeath){
                                //捕获到这个异常表示当前线程希望安静地退出，满足它吧，这里就不打印任何日志了。
                            }
                            logger.error("Thread["+t.getId()+","+t.getName()+"] caught error:", e);
                        }
                    });
                }
                flag = false;
            }
        }

        @Override
        public int getOrder() {
            return -15;
        }
    }

}
